#define _GNU_SOURCE
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fnmatch.h>
#include <stdarg.h>
#include <errno.h>
#include "stdrusty.h"
#include "ccontrol.h"

static void insert_arg(char *argv[], unsigned argc, unsigned pos, char *arg)
{
	memmove(argv+pos+1, argv+pos, (argc+1) * sizeof(char *));
	argv[pos] = arg;
}

void __verbose(const char *msg, ...)
{
	va_list arglist;
	char *str, pidstr[CHAR_SIZE(int)];

	int_to_string(pidstr, getpid());
	write(STDERR_FILENO, "ccontrol: ", strlen("ccontrol: "));
	write(STDERR_FILENO, pidstr, strlen(pidstr));
	write(STDERR_FILENO, ": ", strlen(": "));
	write(STDERR_FILENO, msg, strlen(msg));

	va_start(arglist, msg);
	while ((str = va_arg(arglist, char *)) != NULL)
		write(STDERR_FILENO, str, strlen(str));
	va_end(arglist);

	write(STDERR_FILENO, "\n", 1);
}

static bool make_target_match(char *argv[], char *targets)
{
	char *p, *target = "";
	unsigned int i;

	if (!targets)
		return false;

	/* Heuristic: choose last arg which is neither option nor variable */
	for (i = 0; argv[i]; i++) {
		if (argv[i][0] == '-')
			continue;
		if (strchr(argv[i], '='))
			continue;
		target = argv[i];
	}

	for (p = strtok(targets, " \t"); p; p = strtok(NULL, " \t"))
		if (fnmatch(p, target, 0) == 0)
			return true;
	return false;
}

/* Earlier arg parsing might have revealed distcc is useless. */
static void do_section(struct section sec,
		       enum type type,
		       bool no_distcc,
		       char *argv[], unsigned argc)
{
	argv[0] = sec.names[type];

	switch (type) {
	case TYPE_MAKE:
		/* Ensure make calls us for children, not /usr/bin/make. */
		setenv("MAKE", "make", 1);

		/* Children on non-parallel'ed makes are not parallel. */
		if (getenv("CCONTROL_NO_PARALLEL"))
			break;

		if (make_target_match(argv, sec.no_parallel))
			setenv("CCONTROL_NO_PARALLEL", "1", 1);
		else {
			verbose(sec, "Inserting arg -j in make!", NULL);
			insert_arg(argv, argc++, 1, "-j");
		}
		break;
	case TYPE_LD:
		break;
	case TYPE_CC:
	case TYPE_CPLUSPLUS:
		/* ccache + distcc needs special care. */
		if (sec.distcc && !no_distcc) {
			verbose(sec, "Setting DISTCC_HOSTS=", 
				sec.distcc_hosts, NULL);
			setenv("DISTCC_HOSTS", sec.distcc_hosts, 1);
			if (sec.ccache) {
				verbose(sec, "Setting CCACHE_PREFIX=",
					sec.distcc, NULL);
				setenv("CCACHE_PREFIX", sec.distcc, 1);
			} else {
				verbose(sec, "Prefixing arg ",
					sec.distcc, NULL);
				insert_arg(argv, argc++, 0, sec.distcc);
			}
		}

		/* Don't set CCACHE_UNIFY: misses not so bad with distcc. */
		if (sec.ccache) {
			verbose(sec, "Prefixing arg ", sec.ccache, NULL);
			insert_arg(argv, argc++, 0, sec.ccache);
		}
	}
}

#ifndef TESTING
static void __attribute__((noreturn)) run_command(struct section sec,
						  char *argv[])
{
	if (sec.verbose) {
		unsigned int i;
		for (i = 0; argv[i]; i++)
			verbose(sec, "ARG ", argv[i], NULL);
	}
	execv(argv[0], argv);
	fatal("failed to exec ", errno, argv[0], NULL);
}
#else
static void __attribute__((noreturn)) run_command(struct section sec,
						  char *argv[])
{
	if (getenv("DISTCC_HOSTS"))
		printf("DISTCC_HOSTS='%s' ", getenv("DISTCC_HOSTS"));

	while (argv[0]) {
		printf("%s ", argv[0]);
		argv++;
	}
	printf("\n");

	exit(0);
}
#endif

int main(int argc, char *orig_argv[])
{
	enum type type;
	bool slow;
	struct section sec;
	char configname[PATH_MAX];
	char *argv[argc + 3];

	/* Run low priority; people like to use apps while compiling. */
	nice(10);

	strcpy(configname, getenv("HOME"));
	strcat(configname, "/.ccontrol/default");

	/* Make room to add args. */
	memcpy(argv, orig_argv, (argc + 1)*sizeof(argv[0]));

#ifdef TESTING
	if (getenv("ARGV0"))
		argv[0] = getenv("ARGV0");
#endif
	sec = read_config(configname);
	type = what_am_i(argv, sec.distcc != NULL, &slow);

	do_section(sec, type, slow, argv, argc);

	grab_lock(configname, sec, slow, type);

	run_command(sec, argv);
}
