/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <algorithm>
#include <unistd.h>
#include <fcntl.h>
#include <sys/select.h>

#include "Game.H"

int
Game::turnTimeout = 120;

Game::Game(int ServerFD)
    : serverFD(ServerFD)
{
}

Game::GameState
Game::getGameState() const 
{
    return gameState;
}

void
Game::addPlayer(int FD, Player *NewPlayer)
{
    assert(players.find(FD)==players.end());
    players[FD] = NewPlayer;
    creatures.push_back(NewPlayer);
}

int
Game::numPlayers() const
{
    return players.size();
}

void
Game::ApplySpells()
{
    map<int, Player *>::iterator player;

    // List of all spells cast
    map<Spell, bool> allSpellsCast;

    // Need to accumlate list of spells on each creature
    // and where they came from
    map<BaseCreature *, vector<Player::SpellCast> > spellsOnTarget;
    
    Spell leftHandSpell, rightHandSpell;
    int leftTargetID, rightTargetID;

    // Keep track of all spells which are cast and create
    // a list of spells which may be applied for each creature
    // in the game
    for (player = players.begin(); player != players.end(); player++) {
	player->second->getSpellsCast(&leftHandSpell, &leftTargetID,
				 &rightHandSpell, &rightTargetID);
	Player::SpellCast spell;
	spell.spellWorked  = false;

	// Accumulate left hand spells
	if (leftHandSpell!=SPL_NONE) {
	    spell.source = player->second;
	    spell.target = BaseCreature::mapIDToPointer(leftTargetID);
	    spell.spellCast = leftHandSpell;
	    
	    allSpellsCast[leftHandSpell] = true;
	    spellsOnTarget[spell.target].push_back(spell); 
	}

	// Accumulate right hand spells 
	if (rightHandSpell!=SPL_NONE) {
	    spell.source = player->second;
	    spell.target = BaseCreature::mapIDToPointer(rightTargetID);
	    spell.spellCast = rightHandSpell;
	    
	    allSpellsCast[rightHandSpell] = true;
	    spellsOnTarget[spell.target].push_back(spell); 
	}
    }

    // Resolve spell conflicts
    

    // Sort spells for each target in order that they should be applied
    

    // Apply spells on each creature
    map<BaseCreature *, vector<Player::SpellCast> >::iterator target;
    vector<Player::SpellCast>::iterator spellToApply;
    vector<Player::SpellCast> spellsCast;
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end();
	 target++) {

	// List of spells on single target
	spellsCast = target->second;
	for (spellToApply = spellsCast.begin();
	     spellToApply != spellsCast.end();
	     spellToApply++) {
	    switch (spellToApply->spellCast) {
	    case SPL_SUMMON_GOBLIN:
	    case SPL_SUMMON_OGRE:
	    case SPL_SUMMON_TROLL:
	    case SPL_SUMMON_GIANT:
	    case SPL_SUMMON_ELEMENTAL:
		// Need to handling summoning here and assign
		// creature to player
		break;

	    case SPL_CHARM_MONSTER:
		// Will need to call charm monster
		// and reparent to new owner.
		
	    default:
		spellToApply->spellWorked = 
		    target->first->applySpell(spellToApply->spellCast,
					      &spellToApply->resultMessage);
	    }

	    // Send update to all players saying what happened
	    for (player = players.begin(); player != players.end(); player++) {
		player->second->sendSpellCast(*spellToApply);
	    }
	}
    }
    
    // Send notification to all player saying spells observed effect
    // has ended
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendSpellCastEnd();
    }
}


int
Game::numPlayersAlive()
{
    map<int, Player *>::iterator player;
    int live_count = 0;

    for (player = players.begin(); player != players.end(); player++) {
	if (!player->second->isStateSet(CS_DEAD)) {
	    live_count++;
	}
    }
    return live_count;
}

int
Game::PlayersJoining() 
{
    fd_set readFs, writeFs;
    struct sockaddr_in sa;
    socklen_t salen;
    int playerFd;
    map<int, Player *>::iterator player;
    int maxFD;

    while (numPlayers()<2) {
	FD_ZERO(&readFs);
	FD_ZERO(&writeFs);

	// Accepting new connections
	FD_SET(serverFD, &readFs);
	maxFD = serverFD;

	for (player=players.begin(); player!=players.end(); player++) {
	    FD_SET(player->first, &readFs);
	    maxFD = max(maxFD, player->first);

	    if (player->second->connection().pendingDataToWrite()) {
		// Need to send data to player
		FD_SET(player->first, &writeFs);
	    }
	}

	printf("Waiting for new clients\n");

	if (select(maxFD+1, &readFs, &writeFs, NULL, NULL)<0) {
	    perror("Select on accepting new clients failed");
	    exit(1);
	}


	if (FD_ISSET(serverFD, &readFs)) {
	    // New player joining 
	    salen = sizeof(sa);
	    playerFd = accept(serverFD, (struct sockaddr *)&sa, &salen);

	    if (playerFd<0) {
		perror("Accept failed for new client");
		exit(1);
	    }

	    // Set connection non blocking
	    int flags;
	    flags = fcntl(playerFd, F_GETFL);
	    if (flags<0) {
		perror("fcntl F_GETFL failed");
		exit(1);
	    }
	    flags = flags | O_NONBLOCK;
	    if (fcntl(playerFd, F_SETFL, flags)<0) {
		perror("failed to make connection non blocking");
		exit(1);
	    }

	    addPlayer(playerFd, new Player(playerFd));
	}

	// Check for player player communication
	for (player=players.begin(); player!=players.end(); player++) {
	    if (FD_ISSET(player->first, &readFs)) {
		// Data to read
		player->second->connection().readData();

		// Process messages if any received
		while (player->second->connection().haveMessageToRead()) {
		    player->second->ProcessMessage();
		}
	    }

	    if (FD_ISSET(player->first, &writeFs)) {
		// Can write more data to player
		player->second->connection().writeData();
	    }
	}
    }
    
    return numPlayers();
}

void
Game::StartGame()
{
    printf("Starting Game\n");

    map<int, Player *>::iterator player;

    assert(numPlayers()>1);

    // synchronise before starting
    WaitForResponses(turnTimeout);

    // Send new player info
    printf("Sending new player info\n");
    for (player = players.begin(); player != players.end(); player++) {
	map<int, Player *>::iterator newPlayer;
	for (newPlayer = players.begin(); 
	     newPlayer != players.end(); 
	     newPlayer++) {
	    player->second->sendNewPlayerInfo(newPlayer->second);
	}
    }
    
    // Send creature states
    // Send state of all creatures to all players
    vector<BaseCreature *>::iterator creature;
    for (creature = creatures.begin();
	 creature != creatures.end();
	 creature++) {
	for (player = players.begin(); player != players.end(); player++) {
	    player->second->sendCreatureState(*creature);
	}
    }
    // Send end marker
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendCreatureStateEnd();
    }

    // Keep playing while at least one player is alive
    while (numPlayersAlive()>1) {
	printf("\n\n--------------------------------------------------\n");
	printf("New round\n");

	// Get gestures from live players
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		player->second->askForGestures();
	    }
	}
	printf("Get gestures\n");
	WaitForResponses(turnTimeout);

	// Apply state which may affect gestures
	// eg paralysis, confusion etc. May need to ask more questions here...

	// Work out what spells have been cast
	vector<Spell> leftHandSpells, rightHandSpells;
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		// Get spell lists
		player->second->getGestureHistory().getSpells(&leftHandSpells, 
							      &rightHandSpells);
		// Ask players what spells they want to cast
		// (where a hand may cast multiple spells can only cast 1)
		printf("Player %i %i spells\n",
		       leftHandSpells.size(), rightHandSpells.size());
		player->second->askForSpellSelection(leftHandSpells, 
						     rightHandSpells);
	    }
	}

	printf("Get spell lists\n");
	WaitForResponses(turnTimeout);

	// Send players gestures they have seen
	// modulo players that can't see (looked after by player class)

	// Work out which creatures are valid targets (visible+alive)
	vector<int> visibleCreatures;

	for (creature = creatures.begin(); 
	     creature != creatures.end(); 
	     creature++) {
	    if ( !((*creature)->getState() & (CS_DEAD|CS_INVISIBILITY)) ) {
		visibleCreatures.push_back((*creature)->getID());
	    }
	}

	map<int, Player *>::iterator sourcePlayer;
	for (player = players.begin(); player != players.end(); player++) {
	    // Send gesture updates to dead players
	    for (sourcePlayer = players.begin(); 
		 sourcePlayer != players.end(); 
		 sourcePlayer++) {
		// Only send gestures for live players 
		if (!sourcePlayer->second->isStateSet(CS_DEAD)) {
		    player->second->sendGesturesSeen(sourcePlayer->second);
		}
	    }

	    // Send end marker for gestures seen
	    player->second->sendGesturesSeenEnd();

	    // Ask players where to direct spells
	    if (!player->second->isStateSet(CS_DEAD)) {
		printf("Asking player for spell directions\n");
		player->second->askForSpellDirections(visibleCreatures);
	    }
	}

	printf("Get spell directions\n");
	WaitForResponses(turnTimeout);
	
	// Apply spell effects
	ApplySpells();
	
	// Ask players for who monsters should attack


// 	WaitForResponses(turnTimeout);

	// "Remove" dead monsters
	// TODO: Need to properly remove elementals.
	// "Remove" dead players??
	// Check if creatures died
	for (creature = creatures.begin();
	     creature != creatures.end();
	     creature++) {
	    if ( (*creature)->getNumHitPoints()<=0) {
		(*creature)->setState(CS_DEAD);
	    }
	}

	// update (eg update counters)
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		player->second->decrementCounters();
	    }
	}

	// Send state of all creatures to all players
	for (creature = creatures.begin();
	     creature != creatures.end();
	     creature++) {
	    for (player = players.begin(); player != players.end(); player++) {
		player->second->sendCreatureState(*creature);
	    }
	}
	// Send end marker
	for (player = players.begin(); player != players.end(); player++) {
	    player->second->sendCreatureStateEnd();
	}
	printf("End of round\n");

    }

    if (numPlayersAlive()==0) {
	printf("oops don't handle draws yet!\n");
	exit(1);
    } else {
	// Look for winner
	Player *winner=NULL;
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		winner = player->second;
		break;
	    }
	}
	assert(winner!=NULL);

	// Send notification to all players
	for (player = players.begin(); player != players.end(); player++) {
	    player->second->sendEndGame(winner);
	}
	
    }

}

void
Game::WaitForResponses(int Timeout)
{
    struct timeval timeout;
    timeout.tv_sec = Timeout;
    timeout.tv_usec = 0;

    fd_set readFs, writeFs;
    int maxFD = 0;
    bool waitingForPlayer;
    int numFDReady;
    map<int, Player *>::iterator player;
    while (1) {
	
	waitingForPlayer = false;

	FD_ZERO(&readFs);
	FD_ZERO(&writeFs);

	// Should eventually allow observers, but ignore for now
	// so don't accept new connections

	for (player=players.begin(); player!=players.end(); player++) {
	    FD_SET(player->first, &readFs);
	    maxFD = max(maxFD, player->first);

	    if (player->second->connection().pendingDataToWrite()) {
		// Need to send data to player
		FD_SET(player->first, &writeFs);
		waitingForPlayer = true;
	    }

	    if (player->second->amWaitingForResponse()) {
		waitingForPlayer = true;
	    } 
	}

	if (!waitingForPlayer) {
	    // All players have responded 
	    break;
	}

	numFDReady = select(maxFD+1, &readFs, &writeFs, NULL, &timeout);

	if (numFDReady==0) {
	    printf("Timeout for responses\n");
	    break;
	} else if (numFDReady<0) {
	    perror("Select on accepting new clients failed");
	    exit(1);
	}

	// Check for player communication
	for (player=players.begin(); player!=players.end(); player++) {
	    if (FD_ISSET(player->first, &readFs)) {
		// Data to read
		player->second->connection().readData();

		// Process messages if any received
		while (player->second->connection().haveMessageToRead()) {
		    player->second->ProcessMessage();
		}
	    }

	    if (FD_ISSET(player->first, &writeFs)) {
		// Can write more data to player
		player->second->connection().writeData();
	    }
	}
    }
}
