/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include "BaseCreature.H"
#include <assert.h>

int BaseCreature::IDCount = 1;
map<int, BaseCreature *> BaseCreature::IDMap;

BaseCreature::BaseCreature(int DamageInflicted, int NumHitPoints)
    : damageInflicted(DamageInflicted), hitPoints(NumHitPoints), 
      maxNumHitPoints(NumHitPoints), state(0), counterProtFromEvil(0),
      counterDisease(0), counterPoison(0), counterHaste(0)
{
    ID = IDCount++;
    IDMap[ID] = this;
}

BaseCreature::~BaseCreature()
{
}

int
BaseCreature::getID() const
{
    return ID;
}

BaseCreature *
BaseCreature::mapIDToPointer(int ID)
{
    assert(ID<IDCount);
    assert(ID!=0);
    return IDMap[ID];
}

int
BaseCreature::getNumHitPoints() const 
{
    return hitPoints;
}

int 
BaseCreature::getMaxNumHitPoints() const
{
    return maxNumHitPoints;
}

void
BaseCreature::setNumHitPoints(int NumHitPoints)
{
    hitPoints = NumHitPoints;
}

void
BaseCreature::changeNumHitPoints(int Delta)
{
    hitPoints += Delta;
    if (hitPoints > maxNumHitPoints) hitPoints = maxNumHitPoints;
}

void
BaseCreature::setName(const char *Name)
{
    name = Name;
}

const char *
BaseCreature::getName() const
{
    return name.c_str();
}

unsigned int
BaseCreature::getState() const
{
    return state;
}

bool
BaseCreature::isStateSet(int StateBits) const
{
    return state & StateBits;
}

void
BaseCreature::setState(unsigned int State)
{
    state = State;
}

void
BaseCreature::setStateBits(unsigned int StateBits)
{
    state |= StateBits;
}

void
BaseCreature::clearStateBits(unsigned int StateBits)
{
    state &= (StateBits ^ BASE_CREATEURE_STATE_BITS_MASK);
}

int
BaseCreature::getDamageInflicted() const
{
    return damageInflicted;
}

bool
BaseCreature::applySpell(Spell SpellToApply, string *Message)
{
    switch (SpellToApply) {

    case SPL_SHIELD:
	return applyShieldSpell(Message);
	break;

    case SPL_REMOVE_ENCHANTMENT:
	return applyRemoveEnchantmentSpell(Message);
	break;

    case SPL_MAGIC_MIRROR:
	return applyMagicMirrorSpell(Message);
	break;

    case SPL_COUNTER_SPELL:
    case SPL_COUNTER_SPELL1:
	return applyCounterSpellSpell(Message);
	break;

    case SPL_DISPEL_MAGIC:
	return applyDispelMagicSpell(Message);
	break;
	
    case SPL_RAISE_DEAD:
	return applyRaiseDeadSpell(Message);
	break;

    case SPL_CURE_LIGHT_WOUNDS:
	return applyCureLightWoundsSpell(Message);
	break;

    case SPL_CURE_HEAVY_WOUNDS:
	return applyCureHeavyWoundsSpell(Message);
	break;

    case SPL_SUMMON_GOBLIN:
    case SPL_SUMMON_OGRE:
    case SPL_SUMMON_TROLL:
    case SPL_SUMMON_GIANT:
    case SPL_SUMMON_ELEMENTAL:
	assert(0);
	break;

    case SPL_MISSILE:
	return applyMissileSpell(Message);
	break;

    case SPL_FINGER_OF_DEATH:
	return applyFingerOfDeathSpell(Message);
	break;

    case SPL_LIGHTNING_BOLT:
    case SPL_LIGHTNING_BOLT1:
	return applyLightningBoltSpell(Message);
	break;

    case SPL_CAUSE_LIGHT_WOUNDS:
	return applyCauseLightWoundsSpell(Message);
	break;

    case SPL_CAUSE_HEAVY_WOUNDS:
	return applyCauseHeavyWoundsSpell(Message);
	break;

    case SPL_FIREBALL:
	return applyFireballSpell(Message);
	break;

    case SPL_FIRESTORM:
	return applyFireStormSpell(Message);
	break;

    case SPL_ICESTORM:
	return applyIceStormSpell(Message);
	break;

    case SPL_AMNESIA:
	return applyAmnesiaSpell(Message);
	break;

    case SPL_CONFUSION:
	return applyConfusionSpell(Message);
	break;

    case SPL_CHARM_PERSON:
	return applyCharmPersonSpell(Message);
	break;

    case SPL_CHARM_MONSTER:
	return applyCharmMonsterSpell(Message);
	break;

    case SPL_PARALYSIS:
	return applyParalysisSpell(Message);
	break;

    case SPL_FEAR:
	return applyFearSpell(Message);
	break;

    case SPL_ANTI_SPELL:
	return applyAntiSpell(Message);
	break;

    case SPL_PROTECTION_FROM_EVIL:
	return applyProtectionFromEvilSpell(Message);
	break;
	
    case SPL_RESIST_HEAT:
	return applyResistHeatSpell(Message);
	break;

    case SPL_RESIST_COLD:
	return applyResistColdSpell(Message);
	break;

    case SPL_DISEASE:
	return applyDiseaseSpell(Message);
	break;
	
    case SPL_POISON:
	return applyPoisonSpell(Message);
	break;

    case SPL_BLINDNESS:
	return applyBlindnessSpell(Message);
	break;

    case SPL_INVISIBILITY:
	return applyInvisibilitySpell(Message);
	break;

    case SPL_HASTE:
	return applyHasteSpell(Message);
	break;

    case SPL_TIME_STOP:
	return applyTimeStopSpell(Message);
	break;

    case SPL_DELAYED_EFFECT:
	return applyDelayedEffectSpell(Message);
	break;

    case SPL_PERMANENCY:
	return applyPermanencySpell(Message);
	break;

    case SPL_SURRENDER:
	applySurrender(Message);
	return true;
	break;

    case SPL_STAB:
    case SPL_STAB1:
	return applyStab(Message);
	break;

    case SPL_FINAL_MARKER:
    case SPL_NONE:
	assert(0);

    }
    return false;
}

bool
BaseCreature::applyShieldSpell(string *Message)
{
    setStateBits(CS_SHIELD);
    return true;
}

bool
BaseCreature::applyMagicMirrorSpell(string *Message)
{
    /* Not sure what to do here yet */
    return true;
}

bool
BaseCreature::applyCounterSpellSpell(string *Message)
{
    /* This spell has other effects which are handled elsewhere,
       but it also doubles as a shield spell, the state of which
       we set here */
    return applyShieldSpell(Message);
}

bool
BaseCreature::applyRaiseDeadSpell(string *Message)
{
    /* Should raise dead be applied last, or first - impacts
       on what happens here */
    if (getNumHitPoints() <= 0) {
	setNumHitPoints(maxNumHitPoints);
	clearStateBits(ENCHANTMENT_SPELLS);
	clearStateBits(CS_DEAD);
    } else {
	changeNumHitPoints(5);
    };
    return true;
}

bool
BaseCreature::applyCureLightWoundsSpell(string *Message)
{
    changeNumHitPoints(1);
    return true;
}

bool
BaseCreature::applyCureHeavyWoundsSpell(string *Message)
{
    changeNumHitPoints(2);
    return true;
}

bool
BaseCreature::applyMissileSpell(string *Message)
{
    if (!isStateSet(CS_SHIELD)) {
	changeNumHitPoints(-1);
	return true;
    } else {
	return false;
    }
}

bool
BaseCreature::applyFingerOfDeathSpell(string *Message)
{
    setNumHitPoints(0);
    setStateBits(CS_DEAD);
    return true;
}

bool
BaseCreature::applyLightningBoltSpell(string *Message)
{
    changeNumHitPoints(-5);
    return true;
}

bool
BaseCreature::applyCauseLightWoundsSpell(string *Message)
{
    changeNumHitPoints(-2);
    return true;
}

bool
BaseCreature::applyCauseHeavyWoundsSpell(string *Message)
{
    changeNumHitPoints(-3);
    return true;
}

bool
BaseCreature::applyFireballSpell(string *Message)
{
    if (!isStateSet(CS_RESIST_HEAT)) {
	changeNumHitPoints(-5);
	return true;
    } else {
	return false;
    }
}

bool
BaseCreature::applyFireStormSpell(string *Message)
{
    changeNumHitPoints(-5);
    return true;
}

bool
BaseCreature::applyIceStormSpell(string *Message)
{
    changeNumHitPoints(-5);
    return true;
}


bool
BaseCreature::applyProtectionFromEvilSpell(string *Message)
{
    setStateBits(CS_PROT_FROM_EVIL);
    counterProtFromEvil = 4;
    return true;
}

bool
BaseCreature::applyResistHeatSpell(string *Message)
{
    setStateBits(CS_RESIST_HEAT);
    return true;
}

bool
BaseCreature::applyResistColdSpell(string *Message)
{
    setStateBits(CS_RESIST_COLD);
    return true;
}

bool
BaseCreature::applyDiseaseSpell(string *Message)
{
    if (!isStateSet(CS_DISEASED)) {
	setStateBits(CS_DISEASED);
	counterDisease = 7;
	return true;
    } else {
	return false;
    }
}

bool 
BaseCreature::applyPoisonSpell(string *Message)
{
    if (!isStateSet(CS_POISONED)) {
	setStateBits(CS_POISONED);
	counterPoison = 7;
	return true;
    } else {
	return false;
    }
}

bool
BaseCreature::applyHasteSpell(string *Message)
{
    setStateBits(CS_HASTE);
    counterHaste = 4;
    return true;
}

bool
BaseCreature::applyTimeStopSpell(string *Message)
{
    setStateBits(CS_TIMESTOP);
    return true;
}

void
BaseCreature::applySurrender(string *Message)
{
    setStateBits(CS_SURRENDER);
}

bool 
BaseCreature::applyStab(string *Message)
{
    if (!isStateSet(CS_SHIELD|CS_PROT_FROM_EVIL)) {
	changeNumHitPoints(-1);
	return true;
    } else {
	return false;
    }
}

void
BaseCreature::decrementCounters() 
{
    if (counterProtFromEvil) 
	counterProtFromEvil--;

    if (counterDisease) {
	counterDisease--;
	if (counterDisease==0) {
	    setStateBits(CS_DEAD);
	}
    }

    if (counterPoison) {
	counterPoison--;
	if (counterPoison==0) {
	    setStateBits(CS_DEAD);
	}
    }

    if (counterHaste)
	counterHaste--;

    if (!counterProtFromEvil)
	clearStateBits(CS_SHIELD);

}
