/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>
#include <netinet/in.h>

#include "SCClient.H"
#include "protocol.h"


SCClient::SCClient(int Connection, string Name)
    : remoteConnection(Connection), rcvdAllGestures(false),
      rcvdAllSpells(false), rcvdAllCreatureInfo(false)
{
    // Initialise connection to server
    while (!remoteConnection.haveMessageToRead()) {
	printf("Waiting for initialisation data\n");
	readData();
    }

    // Get welcome message
    Message *msg = remoteConnection.getNextMessage();
    assert(msg->command==MSG_ASK_WELCOME);
    serverWelcomeMessage = msg->data;
    free(msg);

    // Send client details
    msg = Message::createMessage(MSG_RCV_CLIENT_DETAILS, 
				 Name.length()+1);
    strcpy(msg->data, Name.c_str());
    assert(msg->Ok());
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
SCClient::readData()
{
    remoteConnection.readData();
}

bool
SCClient::messageAvailable() const
{
    return remoteConnection.haveMessageToRead();
}

void
SCClient::writeData()
{
    remoteConnection.writeData();
}

bool
SCClient::haveDataToWrite() const
{
    return remoteConnection.pendingDataToWrite();
}

int
SCClient::getConnectionFD() const
{
    return remoteConnection.getFD();
}

const string &
SCClient::getServerWelcomeMessage() const
{
    return serverWelcomeMessage;
}

int
SCClient::processMessage()
{
    assert(remoteConnection.haveMessageToRead());

    Message *msg = remoteConnection.getNextMessage();
    assert(msg->Ok());

    switch (msg->command) {
    case MSG_SEND_NEWPLAYER_INFO:
	rcvNewPlayerInfo(msg);
	break;

    case MSG_ASK_FOR_GESTURES:
	// De-facto start of round
	rcvdAllGestures = false;
	rcvdAllSpells = false;
	rcvdAllCreatureInfo = false;
	spellResults.clear();
	break;

    case MSG_SEND_GESTURES_SEEN:
	rcvGesturesSeen(msg);
	break;
	
    case MSG_ASK_FOR_SPELLS_CAST:
	rcvAskForSpellsCast(msg);
	break;

    case MSG_ASK_SPELL_DIRECTIONS:
	rcvAskSpellDirections(msg);
	break;

    case MSG_SEND_SPELL_CAST:
	rcvSpellCast(msg);
	break;

    case MSG_SEND_CREATURE_STATE:
	rcvCreatureState(msg);
	break;

    default:
	printf("Unknown message %i received\n", msg->command);


    }
    return msg->command;
}

void
SCClient::sendGestures(Gesture LeftHand, Gesture RightHand)
{
    int left, right;
    Message *msg = Message::createMessage(MSG_RCV_GESTURES_USED,
					  sizeof(int)*2);
    left = htonl(LeftHand);
    right = htonl(RightHand);

    memcpy(msg->data, &left, sizeof(int));
    memcpy(msg->data+sizeof(int), &right, sizeof(int));

    assert(msg->Ok());
    
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
SCClient::getCreatureInfo(int ID, CreatureState *Creature) const
{
    assert(creatureData.find(ID)!=creatureData.end());
    *Creature =  creatureData.find(ID)->second;
}

bool
SCClient::haveAllCreatureInfo() const
{
    return rcvdAllCreatureInfo;
}

void
SCClient::getAllCreatureIDs(vector<int> *CreatureIDs) const
{
    CreatureIDs->clear();
    map<int, CreatureState>::const_iterator creature;

    for (creature=creatureData.begin(); creature!=creatureData.end();
	 creature++) {
	CreatureIDs->push_back(creature->first);
    }
}

bool
SCClient::haveAllGestures() const
{
    return rcvdAllGestures;
}

void
SCClient::getGesturesForPlayer(int ID, Gesture *Left, Gesture *Right)
{
    assert(creatureData.find(ID)!=creatureData.end());
    assert(creatureData.find(ID)->second.owner==0);
    assert(currentGestureMapping.find(ID)!=currentGestureMapping.end());

    *Left = currentGestureMapping.find(ID)->second.primaryGesture;
    *Right = currentGestureMapping.find(ID)->second.secondaryGesture;
}

void
SCClient::getPossibleSpells(vector<Spell> *LeftHandSpells,
			    vector<Spell> *RightHandSpell) 
{
    *LeftHandSpells = possibleLeftHandSpells;
    *RightHandSpell = possibleRightHandSpells;
}

void
SCClient::sendSpellSelection(Spell LeftHandSpell, Spell RightHandSpell)
{
    int leftSpell, rightSpell;
    Message *msg = Message::createMessage(MSG_RCV_SPELLS_CAST,
					  sizeof(int)*2);
    leftSpell = htonl((int)LeftHandSpell);
    rightSpell = htonl((int)RightHandSpell);
    memcpy(msg->data, &leftSpell, sizeof(leftSpell));
    memcpy(msg->data+sizeof(leftSpell), &rightSpell,
	   sizeof(rightSpell));

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
SCClient::getSpellsCast(Spell *LeftHandSpell, Spell *RightHandSpell,
			vector<int> *ValidTargetIDs)
{
    *LeftHandSpell = leftSpellCast;
    *RightHandSpell = rightSpellCast;
    *ValidTargetIDs = validTargets;
}

const vector<SCClient::SpellResult> &
SCClient::getSpellResults() const
{
    return spellResults;
}

bool
SCClient::haveAllSpellsCast() const
{
    return rcvdAllSpells;
}

void
SCClient::sendSpellTargets(int LeftHandTarget, int RightHandTarget)
{
    Message *msg = Message::createMessage(MSG_RCV_SPELL_DIRECTIONS,
					  sizeof(int)*2);
    LeftHandTarget = htonl(LeftHandTarget);
    RightHandTarget = htonl(RightHandTarget);
    memcpy(msg->data, &LeftHandTarget, sizeof(LeftHandTarget));
    memcpy(msg->data+sizeof(LeftHandTarget), &RightHandTarget,
	   sizeof(RightHandTarget));

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}



void
SCClient::rcvNewPlayerInfo(Message *Msg)
{
    int ID;
    
    memcpy(&ID, Msg->data, sizeof(int));
    ID = ntohl(ID);

    string name = Msg->data+2*sizeof(int);

    printf("Received new player %s with ID %i\n", name.c_str(), ID);

    CreatureState cs;
    cs.name = name;
    cs.hitPoints = 0;
    cs.state = 0;
    cs.owner = 0;
    assert(creatureData.find(ID)==creatureData.end());
    creatureData[ID] = cs;
}

void
SCClient::rcvGesturesSeen(Message *Msg)
{
    int ID, leftGesture, rightGesture;

    memcpy(&ID, Msg->data, sizeof(int));
    ID = ntohl(ID);

    if (ID==0) {
	// End marker 
	rcvdAllGestures = true;
    } else {
	memcpy(&leftGesture, Msg->data+sizeof(int), sizeof(int));
	memcpy(&rightGesture, Msg->data+sizeof(int)*2, sizeof(int));
	
	leftGesture = ntohl(leftGesture);
	rightGesture = ntohl(rightGesture);
	currentGestureMapping[ID]
	    = Spells::GesturePair(Gesture(leftGesture), Gesture(rightGesture));
    }
}

void
SCClient::rcvAskSpellDirections(Message *Msg)
{
    int leftSpell, rightSpell, numTargets;
    int index = 0;

    memcpy(&leftSpell, Msg->data + index, sizeof(leftSpell));
    index += sizeof(leftSpell);
    memcpy(&rightSpell, Msg->data + index, sizeof(rightSpell));
    index += sizeof(rightSpell);
    memcpy(&numTargets, Msg->data + index, sizeof(numTargets));
    index += sizeof(numTargets);

    numTargets = ntohl(numTargets);

    int targetID;
    validTargets.clear();
    for (int i=0; i<numTargets; i++) {
	memcpy(&targetID, Msg->data+index, sizeof(targetID));
	index += sizeof(targetID);
	targetID = ntohl(targetID);
	validTargets.push_back(targetID);
    }

    leftSpellCast = Spell(ntohl(leftSpell));
    rightSpellCast = Spell(ntohl(rightSpell));
}

void
SCClient::rcvSpellCast(Message *Msg)
{
    SpellResult sr;
    int index = 0;

    memcpy(&sr.sourcePlayer, Msg->data+index, sizeof(int));
    index += sizeof(int);
    sr.sourcePlayer = ntohl(sr.sourcePlayer);

    if (sr.sourcePlayer==0) {
	// End marker
	rcvdAllSpells = true;
    } else {

	memcpy(&sr.targetCreature, Msg->data+index, sizeof(int));
	index += sizeof(int);
	sr.targetCreature = ntohl(sr.targetCreature);
	
	memcpy(&sr.spellCast, Msg->data+index, sizeof(int));
	index += sizeof(int);
	sr.spellCast = (Spell)ntohl(sr.spellCast);
	
	int worked;
	memcpy(&worked, Msg->data+index, sizeof(worked));
	index += sizeof(int);
	sr.spellWorked = (bool)worked;
	
	// skip length (trust server)
	index += sizeof(int);
	sr.resultMessage = Msg->data+index;

	spellResults.push_back(sr);
    }
}

void 
SCClient::rcvCreatureState(Message *Msg)
{
    int creatureID, hp, state;

    memcpy(&creatureID, Msg->data, sizeof(int));
    creatureID = ntohl(creatureID);

    if (creatureID==0) {
	// End marker
	rcvdAllCreatureInfo = true;
    } else {

	memcpy(&hp, Msg->data+sizeof(int), sizeof(int));
	memcpy(&state, Msg->data+sizeof(int)*2, sizeof(int));
	
	assert(creatureData.find(creatureID)!=creatureData.end());
	creatureData[creatureID].hitPoints = ntohl(hp);
	creatureData[creatureID].state = ntohl(state);
    }
}

void
SCClient::rcvAskForSpellsCast(Message *Msg)
{
    possibleLeftHandSpells.clear();
    possibleRightHandSpells.clear();

    int numSpells, index = 0;
    memcpy(&numSpells, Msg->data+index, sizeof(numSpells));
    index += sizeof(numSpells);
    numSpells = ntohl(numSpells);

    int spell;
    for (int i=0; i<numSpells; i++) {
	memcpy(&spell, Msg->data+index, sizeof(int));
	index += sizeof(int);
	possibleLeftHandSpells.push_back(Spell(ntohl(spell)));
    }

    // right hand spells
    memcpy(&numSpells, Msg->data+index, sizeof(numSpells));
    index += sizeof(numSpells);
    numSpells = ntohl(numSpells);
    
    for (int i=0; i<numSpells; i++) {
	memcpy(&spell, Msg->data+index, sizeof(int));
	index += sizeof(int);
	possibleRightHandSpells.push_back(Spell(ntohl(spell)));
    }
}
