#!/usr/bin/env python2.3

### protocol.py - part of ecar, an entry for the LCA 2005 hackfest.
### http://lca2005.linux.org.au/hackfest/
### (C) Russell Steicke, 2005.
### Licensed under GPL, see the file COPYING.


# Protocol for LCA 2005 hackfest.


__all__ = [
    'DEFAULT_SPELLCAST_SERVER_PORT',
    'msgTypeToName',
    'gestureTypeToName',
    'monsterTypeToName',
    'creatureStateTypeToName',
    'eventTypeToName',
    'SpellcastProtocolException',
    'SpellcastMessage',
    'SpellcastConnection',
    ]


from __future__ import division

import sys
from socket import socket, error as socket_error, AF_INET, SOCK_STREAM
from errno import EAGAIN
from os import read, write
from StringIO import StringIO

df = False
def d(s):
    if df:
        print s


DEFAULT_SPELLCAST_SERVER_PORT = 12701

MSG_ASK_WELCOME                    = 1
MSG_RCV_CLIENT_DETAILS             = 2
MSG_ASK_FOR_GESTURES               = 3
MSG_RCV_GESTURES_USED              = 4
MSG_ASK_FOR_SPELLS_CAST            = 5
MSG_RCV_SPELLS_CAST                = 6
MSG_SEND_GESTURES_SEEN             = 7
MSG_ASK_SPELL_DIRECTIONS           = 8
MSG_RCV_SPELL_DIRECTIONS           = 9
MSG_SEND_SPELL_CAST                = 10
MSG_SEND_CREATURE_STATE            = 11
MSG_SEND_END_GAME                  = 12
MSG_SEND_NEWPLAYER_INFO            = 13
MSG_ASK_MONSTER_DIRECTIONS         = 15
MSG_RCV_MONSTER_DIRECTIONS         = 16
MSG_SEND_NEW_MONSTER_INFO          = 17
MSG_SEND_MONSTER_ATTACK_INFO       = 18
MSG_SEND_EVENT_INFO                = 19
MSG_SEND_START_GAME                = 20
MSG_ASK_CHARM_PERSON_CTRL_HAND     = 21
MSG_ASK_PARALYSIS_CTRL_HAND        = 22
MSG_ASK_CHARM_PERSON_CTRL_GESTURE  = 23
MSG_RCV_CHARM_PERSON_CTRL_HAND     = 24
MSG_RCV_PARALYSIS_CTRL_HAND        = 25
MSG_RCV_CHARM_PERSON_CTRL_GESTURE  = 26
MSG_USERNAME_IN_USE_ALREADY        = 27
MSG_SEND_ROUND_BEGIN               = 28

_msgTypeToNameDictionary = {
    MSG_ASK_WELCOME                    : "MSG_ASK_WELCOME",
    MSG_RCV_CLIENT_DETAILS             : "MSG_RCV_CLIENT_DETAILS",
    MSG_ASK_FOR_GESTURES               : "MSG_ASK_FOR_GESTURES",
    MSG_RCV_GESTURES_USED              : "MSG_RCV_GESTURES_USED",
    MSG_ASK_FOR_SPELLS_CAST            : "MSG_ASK_FOR_SPELLS_CAST",
    MSG_RCV_SPELLS_CAST                : "MSG_RCV_SPELLS_CAST",
    MSG_SEND_GESTURES_SEEN             : "MSG_SEND_GESTURES_SEEN",
    MSG_ASK_SPELL_DIRECTIONS           : "MSG_ASK_SPELL_DIRECTIONS",
    MSG_RCV_SPELL_DIRECTIONS           : "MSG_RCV_SPELL_DIRECTIONS",
    MSG_SEND_SPELL_CAST                : "MSG_SEND_SPELL_CAST",
    MSG_SEND_CREATURE_STATE            : "MSG_SEND_CREATURE_STATE",
    MSG_SEND_END_GAME                  : "MSG_SEND_END_GAME",
    MSG_SEND_NEWPLAYER_INFO            : "MSG_SEND_NEWPLAYER_INFO",
    MSG_ASK_MONSTER_DIRECTIONS         : "MSG_ASK_MONSTER_DIRECTIONS",
    MSG_RCV_MONSTER_DIRECTIONS         : "MSG_RCV_MONSTER_DIRECTIONS",
    MSG_SEND_NEW_MONSTER_INFO          : "MSG_SEND_NEW_MONSTER_INFO",
    MSG_SEND_MONSTER_ATTACK_INFO       : "MSG_SEND_MONSTER_ATTACK_INFO",
    MSG_SEND_EVENT_INFO                : "MSG_SEND_EVENT_INFO",
    MSG_SEND_START_GAME                : "MSG_SEND_START_GAME",
    MSG_ASK_CHARM_PERSON_CTRL_HAND     : "MSG_ASK_CHARM_PERSON_CTRL_HAND",
    MSG_ASK_PARALYSIS_CTRL_HAND        : "MSG_ASK_PARALYSIS_CTRL_HAND",
    MSG_ASK_CHARM_PERSON_CTRL_GESTURE  : "MSG_ASK_CHARM_PERSON_CTRL_GESTURE",
    MSG_RCV_CHARM_PERSON_CTRL_HAND     : "MSG_RCV_CHARM_PERSON_CTRL_HAND",
    MSG_RCV_PARALYSIS_CTRL_HAND        : "MSG_RCV_PARALYSIS_CTRL_HAND",
    MSG_RCV_CHARM_PERSON_CTRL_GESTURE  : "MSG_RCV_CHARM_PERSON_CTRL_GESTURE",
    MSG_USERNAME_IN_USE_ALREADY        : "MSG_USERNAME_IN_USE_ALREADY",
    MSG_SEND_ROUND_BEGIN               : "MSG_SEND_ROUND_BEGIN"
    }


def msgTypeToName(msgtype):
    try:
        return _msgTypeToNameDictionary[msgtype]
    except KeyError:
        return "UnknownMsgType%d" % msgtype 


GST_NOTHING   = 0
GST_KNIFE     = 1
GST_FINGER    = 2
GST_PALM      = 3
GST_SNAP      = 4
GST_WAVE      = 5
GST_POINT     = 6
GST_CLAP      = 7
GST_ANTISPELL = 8
GST_FOG       = 9

_gestureTypeToNameDictionary = {
    GST_NOTHING     : "GST_NOTHING",
    GST_KNIFE       : "GST_KNIFE",
    GST_FINGER      : "GST_FINGER",
    GST_PALM        : "GST_PALM",
    GST_SNAP        : "GST_SNAP",
    GST_WAVE        : "GST_WAVE",
    GST_POINT       : "GST_POINT",
    GST_CLAP        : "GST_CLAP",
    GST_ANTISPELL   : "GST_ANTISPELL",
    GST_FOG         : "GST_FOG",
    }

def gestureTypeToName(gtype):
    try:
        return _gestureTypeToNameDictionary[gtype]
    except KeyError:
        return "UnknownGestureType%d" % gtype


SC_MONSTER_GOBLIN         = 1
SC_MONSTER_OGRE           = 2
SC_MONSTER_TROLL          = 3
SC_MONSTER_GIANT          = 4
SC_MONSTER_FIRE_ELEMENTAL = 5
SC_MONSTER_ICE_ELEMENTAL  = 6

_monsterTypeToNameDictionary = {
    SC_MONSTER_GOBLIN           : "Goblin",
    SC_MONSTER_OGRE             : "Ogre",
    SC_MONSTER_TROLL            : "Troll",
    SC_MONSTER_GIANT            : "Giant",
    SC_MONSTER_FIRE_ELEMENTAL   : "Fire elemental",
    SC_MONSTER_ICE_ELEMENTAL    : "Ice elemental"
    }

def monsterTypeToName(mtype):
    try:
        return _monsterTypeToNameDictionary[mtype]
    except KeyError:
        return "UnknownMonsterType%d" % mtype


CS_NORMAL              = 0
CS_DEAD                = 1
CS_SHIELD              = 1<<1
CS_AMNESIA             = 1<<2
CS_PERM_AMNESIA        = 1<<3
CS_CONFUSION           = 1<<4
CS_PERM_CONFUSION      = 1<<5
CS_CHARM_PERSON        = 1<<6
CS_PERM_CHARM_PERSON   = 1<<7
CS_PARALYSIS           = 1<<8
CS_PERM_PARALYSIS      = 1<<9
CS_FEAR                = 1<<10
CS_PERM_FEAR           = 1<<11
CS_ANTI_SPELL          = 1<<12
CS_PROT_FROM_EVIL      = 1<<13
CS_PERM_PROT_FROM_EVIL = 1<<14
CS_RESIST_HEAT         = 1<<15
CS_RESIST_COLD         = 1<<16
CS_DISEASED            = 1<<17
CS_POISONED            = 1<<18
CS_BLINDED             = 1<<19
CS_PERM_BLINDED        = 1<<20
CS_INVISIBILITY        = 1<<21
CS_PERM_INVISIBILITY   = 1<<22
CS_HASTE               = 1<<23
CS_PERM_HASTE          = 1<<24
CS_TIMESTOP            = 1<<25
CS_DELAYED_EFFECT      = 1<<26
CS_PERMANENCY          = 1 << 27
CS_SURRENDER           = 1 << 28

_creatureStateTypeToNameDictionary = {
    CS_NORMAL                : "CS_NORMAL",
    CS_DEAD                  : "Dead",
    CS_SHIELD                : "Shield",
    CS_AMNESIA               : "Amnesia",
    CS_PERM_AMNESIA          : "Permanent amnesia",
    CS_CONFUSION             : "Confusion",
    CS_PERM_CONFUSION        : "Permanent confusion",
    CS_CHARM_PERSON          : "Charm person",
    CS_PERM_CHARM_PERSON     : "Permanent charm person",
    CS_PARALYSIS             : "Paralysis",
    CS_PERM_PARALYSIS        : "Permanent Paralysis",
    CS_FEAR                  : "Fear",
    CS_PERM_FEAR             : "Permanent fear",
    CS_ANTI_SPELL            : "Anti spell",
    CS_PROT_FROM_EVIL        : "Protection from evil",
    CS_PERM_PROT_FROM_EVIL   : "Permanent protection from evil",
    CS_RESIST_HEAT           : "Resist heat",
    CS_RESIST_COLD           : "Resist cold",
    CS_DISEASED              : "Diseased",
    CS_POISONED              : "Poisoned",
    CS_BLINDED               : "Blinded",
    CS_PERM_BLINDED          : "Permanently blinded",
    CS_INVISIBILITY          : "Invisibility",
    CS_PERM_INVISIBILITY     : "Permanent invisibility",
    CS_HASTE                 : "Haste",
    CS_PERM_HASTE            : "Permanent haste",
    CS_TIMESTOP              : "Timestop",
    CS_DELAYED_EFFECT        : "Delayed effect",
    CS_PERMANENCY            : "Permanency",
    CS_SURRENDER             : "Surrender"
    }

def creatureStateTypeToName(cstype):
    d("Looking for state %x %d" % (cstype,cstype))
    try:
        s = _creatureStateTypeToNameDictionary[cstype]
        d("State %s" % s)
        return s
    except KeyError:
        return None


SC_EVT_GENERIC_MESSAGE  = 1
SC_EVT_ELEMENTAL_MERGE  = 2
SC_EVT_ELEMENTAL_CANCEL = 3
SC_EVT_PLAYER_AMNESIA   = 4
SC_EVT_CONFUSION        = 5
SC_EVT_PLAYER_FEAR      = 6

_eventTypeToNameDictionary = {
    SC_EVT_GENERIC_MESSAGE  : "SC_EVT_GENERIC_MESSAGE",
    SC_EVT_ELEMENTAL_MERGE  : "SC_EVT_ELEMENTAL_MERGE",
    SC_EVT_ELEMENTAL_CANCEL : "SC_EVT_ELEMENTAL_CANCEL",
    SC_EVT_PLAYER_AMNESIA   : "SC_EVT_PLAYER_AMNESIA",
    SC_EVT_CONFUSION        : "SC_EVT_CONFUSION",
    SC_EVT_PLAYER_FEAR      : "SC_EVT_PLAYER_FEAR",
    }

_eventNameToTypeDictionary = {
    "SC_EVT_GENERIC_MESSAGE"  : SC_EVT_GENERIC_MESSAGE,
    "SC_EVT_ELEMENTAL_MERGE"  : SC_EVT_ELEMENTAL_MERGE,
    "SC_EVT_ELEMENTAL_CANCEL" : SC_EVT_ELEMENTAL_CANCEL,
    "SC_EVT_PLAYER_AMNESIA"   : SC_EVT_PLAYER_AMNESIA,
    "SC_EVT_CONFUSION"        : SC_EVT_CONFUSION,
    "SC_EVT_PLAYER_FEAR"      : SC_EVT_PLAYER_FEAR,
    }

def eventTypeToName(eventtype):
    if _eventTypeToNameDictionary.has_key(eventtype):
        return _eventTypeToNameDictionary[eventtype]
    else:
        return None


SC_MSG_START        = 0xDeadBeefL
SC_MSG_START_string = '\xde\xad\xbe\xef'
SC_MSG_END          = 0xAbcdFedaL
SC_MSG_END_string   = '\xab\xcd\xfe\xda'


def stringToInt(s, offset=0):
    return ord(s[offset])   * 16777216L + \
           ord(s[offset+1]) * 65536 + \
           ord(s[offset+2]) * 256 + \
           ord(s[offset+3])


def intToString(anint):
    return chr((anint & 0xff000000L) >>24) + \
           chr((anint & 0x00ff0000L) >>16) + \
           chr((anint & 0x0000ff00L) >>8)  + \
           chr((anint & 0x000000ffL))


class SpellcastProtocolException( Exception ):

    def __init__(self, msg):
        Exception.__init__(self, "Spellcast protocol error: "+msg)


class SpellcastMessage(dict):

    '''A message from or to the spellcast server.

    This has three parts:

    - The message type (an integer).

    - The data of the message (for received messages this is the data
      on the wire, including the start marker, message length, message
      type and end marker, ; for sent messages this will be
      constructed on the fly when the message is sent).

    - A dictionary of field names mapped to field values.  The
      dictionary contents varies between messages.  The message type
      and message length (from the on-the-wire protocol) are not
      included in these values.

    The field names are in camel case, eg "leftGesture".  Field values
    are int or string as appropriate.  Conversion to and from network
    order and to and from protocol representation of strings is done
    internally.

    This class inherits from builtin dict, so we can use indexing
    directly on the instance, eg myMessage["leftGesture"].'''

    def __init__(self, msgtype=-1, data=None):
        self.msgtype = msgtype
        # We can make a blank message, and fill in fields later.
        if data is None:
            self.data = ''
            #d("New empty SpellcastMessage: type is %s" % msgTypeToName(self.msgtype))
            return
        # Data is a string.  It contains the start marker, message
        # length, message type or end marker.
        self.data = data
        d("New message, length is %d" % len(self.data))
        self.msgtype = stringToInt(self.data, 8)
        #d("New SpellcastMessage: type is %s, len is %d" % (msgTypeToName(self.msgtype),len(self.data)))

        if self.msgtype == MSG_ASK_WELCOME:
            # Construct the welcome message text
            self['protocolVersion'] = stringToInt(self.data, 12)
            self['welcomeMessage'] = self.getStringFromMessage(16)

        elif self.msgtype == MSG_RCV_CLIENT_DETAILS:
            pass
        elif self.msgtype == MSG_ASK_FOR_GESTURES:
            # This message has no data, so only the message type is important.
            pass
        elif self.msgtype == MSG_RCV_GESTURES_USED:
            pass

        elif self.msgtype == MSG_ASK_FOR_SPELLS_CAST:
            # From server, it asks us what we've cast.  Minimum length of this
            # message is 28 bytes.
            if len(self.data) < 28:
                raise SpellcastProtocolException("Short MSG_ASK_FOR_SPELLS_CAST message: len=%d" % len(self.data))
            nLeftSpells = stringToInt(self.data, 12)
            if nLeftSpells > 45:
                raise SpellcastProtocolException("MSG_ASK_FOR_SPELLS_CAST asks for too many spells: %d" % nLeftSpells)
            if len(self.data) < (24 + 4*nLeftSpells):
                raise SpellcastProtocolException("Short MSG_ASK_FOR_SPELLS_CAST message: len=%d, nLeftSpells=%d" \
                                                 % (len(self.data), nLeftSpells))
            self['nLeftSpells'] = nLeftSpells
            leftSpells = []
            for i in range(nLeftSpells):
                leftSpells.append(stringToInt(self.data, 16+4*i))
            self['leftSpells'] = leftSpells
            nRightSpells = stringToInt(self.data, 16+4*nLeftSpells)
            if nRightSpells > 45:
                raise SpellcastProtocolException("MSG_ASK_FOR_SPELLS_CAST asks for too many spells: %d" % nRightSpells)
            if len(self.data) < (24 + 4*nLeftSpells + 4*nRightSpells):
                raise SpellcastProtocolException("Short MSG_ASK_FOR_SPELLS_CAST message: len=%d, nLeftSpells=%d, nRightSpells=%d" \
                                                 % (len(self.data), nLeftSpells, nRightSpells))
            self['nRightSpells'] = nRightSpells
            rightSpells = []
            for i in range(nRightSpells):
                rightSpells.append(stringToInt(self.data, 20+4*nLeftSpells+4*i))
            self['rightSpells'] = rightSpells
            pass

        elif self.msgtype == MSG_RCV_SPELLS_CAST:
            pass

        elif self.msgtype == MSG_SEND_GESTURES_SEEN:
            # Tells us what gestures were used by a player.  This message should
            # be 32 bytes long, but can be 28 bytes in SC 1.1 if
            # playerNumber==0.
            if len(self.data) != 32 and len(self.data) != 28:
                raise SpellcastProtocolException(
                    "Incorrect MSG_SEND_GESTURES_SEEN message length (not 28 or 32): %d" % len(self.data))
            self['playerNumber'] = stringToInt(self.data, 12)
            if self['playerNumber'] != 0:
                if len(self.data) != 32:
                    raise SpellcastProtocolException(
                        "Incorrect MSG_SEND_GESTURES_SEEN message length (expected 32): %d" % len(self.data))
                self['leftGesture'] = stringToInt(self.data, 16)
                self['rightGesture'] = stringToInt(self.data, 20)
                self['antiSpelled'] = stringToInt(self.data, 24)
            else:
                self['leftGesture'] = 0
                self['rightGesture'] = 0
                self['antiSpelled'] = 0

        elif self.msgtype == MSG_ASK_SPELL_DIRECTIONS:
            # Asks where to cast the spells.  Minimum length of this message is
            # 32 bytes.
            if len(self.data) < 32:
                raise SpellcastProtocolException("Short MSG_ASK_SPELL_DIRECTIONS message: len=%d" % len(self.data))
            if len(self.data) % 4:
                raise SpellcastProtocolException("Strange MSG_ASK_SPELL_DIRECTIONS length: len=%d" % len(self.data))
            self['leftSpell'] = stringToInt(self.data, 12)
            self['rightSpell'] = stringToInt(self.data, 16)
            nTargets = stringToInt(self.data, 20)
            self['nTargets'] = nTargets
            targets = []
            for i in range(nTargets):
                targets.append(stringToInt(self.data, 24+4*i))
            self['targets'] = targets

        elif self.msgtype == MSG_RCV_SPELL_DIRECTIONS:
            pass
        elif self.msgtype == MSG_SEND_SPELL_CAST:
            # These messages are either length=20 (for end of spells) or minimum
            # length 37.
            self['source'] = stringToInt(self.data, 12)
            if self['source'] == 0 and len(self.data) == 20:
                self['target'] = 0
                self['spell'] = 0
                self['spellWorked'] = 0
                self['message'] = ''
            else:
                if len(self.data) < 37:
                    raise SpellcastProtocolException("Short MSG_SEND_SPELL_CAST message: len=%d" % len(self.data))
                self['target'] = stringToInt(self.data, 16)
                self['spell'] = stringToInt(self.data, 20)
                self['spellWorked'] = stringToInt(self.data, 24)
                messageLength = stringToInt(self.data, 28) # FIXME, use this length
                self['message'] = self.getStringFromMessage(32, messageLength)
 
        elif self.msgtype == MSG_SEND_CREATURE_STATE:
            self['creatureNumber'] = stringToInt(self.data, 12)
            if self['creatureNumber'] != 0:
                # Make sure health is converted to signed, because stringToInt()
                # returns an unsigned long.  This is harder to do in python than
                # it should be.
                health = stringToInt(self.data, 16)
                if health > 0x7fffffffL:
                    health = health - 0x100000000L
                self['health'] = health
                self['state'] = stringToInt(self.data, 20)
                self['stateDescription'] = creatureStateTypeToName(self['state'])
                d("Creature %d: health=%d: state is %d (%s)" % \
                  (self['creatureNumber'], self['health'], self['state'], self['stateDescription']))

        elif self.msgtype == MSG_SEND_END_GAME:
            nWinners = stringToInt(self.data, 12)
            self['nWinners'] = nWinners
            winners = []
            for w in range(nWinners):
                winners.append(stringToInt(self.data, 16+4*w))
            self['winners'] = winners

        elif self.msgtype == MSG_SEND_NEWPLAYER_INFO:
            self['playerNumber'] = stringToInt(self.data, 12)
            self['playerName'] = self.getStringFromMessage(20)

        elif self.msgtype == MSG_ASK_MONSTER_DIRECTIONS:
            nMonsters = stringToInt(self.data, 12)
            monsters = []
            for i in range(nMonsters):
                monsters.append(stringToInt(self.data, 16+4*i))
            nTargets = stringToInt(self.data, 16+nMonsters*4)
            targets = []
            for i in range(nTargets):
                targets.append(stringToInt(self.data, 20+nMonsters*4+i*4))
            self['nMonsters'] = nMonsters
            self['monsters'] = monsters
            self['nTargets'] = nTargets
            self['targets'] = targets

        elif self.msgtype == MSG_RCV_MONSTER_DIRECTIONS:
            pass

        elif self.msgtype == MSG_SEND_NEW_MONSTER_INFO:
            self['monsterID'] = stringToInt(self.data, 12)
            self['monsterType'] = stringToInt(self.data, 16)
            self['ownerID'] = stringToInt(self.data, 20)
            nameLength = stringToInt(self.data, 24)
            self['nameLength'] = nameLength
            self['monsterName'] = self.getStringFromMessage(28, nameLength)

        elif self.msgtype == MSG_SEND_MONSTER_ATTACK_INFO:
            source = stringToInt(self.data, 12)
            self['source'] = source
            if source != 0:
                self['target'] = stringToInt(self.data, 16)
                self['damage'] = stringToInt(self.data, 20)
                messageLength = stringToInt(self.data, 24)
                self['messageLength'] = messageLength
                self['message'] = self.getStringFromMessage(28, messageLength)

        elif self.msgtype == MSG_SEND_EVENT_INFO:
            self['eventType'] = stringToInt(self.data, 12)
            self['messageSource'] = stringToInt(self.data, 16)
            self['source'] = stringToInt(self.data, 20)
            self['target'] = stringToInt(self.data, 24)
            self['messageLength'] = stringToInt(self.data, 28)
            self['message'] = self.getStringFromMessage(32, self['messageLength'])

        elif self.msgtype == MSG_SEND_START_GAME:
            self['myPlayerID'] = stringToInt(self.data, 12)
            self['turnTimeout'] = stringToInt(self.data, 16)

        elif self.msgtype == MSG_ASK_CHARM_PERSON_CTRL_HAND:
            self['target'] = stringToInt(self.data, 12)

        elif self.msgtype == MSG_ASK_PARALYSIS_CTRL_HAND:
            self['target'] = stringToInt(self.data, 12)

        elif self.msgtype == MSG_ASK_CHARM_PERSON_CTRL_GESTURE:
            self['target'] = stringToInt(self.data, 12)
            self['hand'] = stringToInt(self.data, 16)

        elif self.msgtype == MSG_RCV_CHARM_PERSON_CTRL_HAND:
            pass
        elif self.msgtype == MSG_RCV_PARALYSIS_CTRL_HAND:
            pass
        elif self.msgtype == MSG_RCV_CHARM_PERSON_CTRL_GESTURE:
            pass

        elif self.msgtype == MSG_USERNAME_IN_USE_ALREADY:
            pass
        elif self.msgtype == MSG_SEND_ROUND_BEGIN:
            self['roundNumber'] = stringToInt(self.data, 12)

        else:
            raise SpellcastProtocolException("Unknown message type: %d" % self.msgtype)


    def getStringFromMessage(self, startIndex, maxLength=999):
        '''Get a string from somewhere out of a message.'''
        endIndex = startIndex
        length = 0
        while (endIndex < (len(self.data)-4)) \
                  and (self.data[endIndex] != chr(0)) \
                  and ((endIndex-startIndex) < maxLength):
            endIndex += 1
        return self.data[startIndex:endIndex]


    def __str__(self):
        '''String representation for a message.'''
        return "<SpellcastMessage:%s:%d>" % (msgTypeToName(self.msgtype), self.msgtype)


    def getMessageName(self):
        return msgTypeToName(self.msgtype)[4:] # Remove the MSG_ prefix


    def makeProtocolString(self):
        buf = StringIO()
        if self.msgtype == MSG_ASK_WELCOME:
            pass

        elif self.msgtype == MSG_RCV_CLIENT_DETAILS:
            clientDetails = str(self['clientDetails'])
            buf.write(clientDetails)
            buf.write(chr(0))

        elif self.msgtype == MSG_ASK_FOR_GESTURES:
            pass

        elif self.msgtype == MSG_RCV_GESTURES_USED:
            d("Making MSG_RCV_GESTURES_USED: gestures are %s %s" % \
              (gestureTypeToName(self['leftGesture']), gestureTypeToName(self['rightGesture'])))
            buf.write(intToString(self['leftGesture']))
            buf.write(intToString(self['rightGesture']))
        elif self.msgtype == MSG_ASK_FOR_SPELLS_CAST:
            pass

        elif self.msgtype == MSG_RCV_SPELLS_CAST:
            buf.write(intToString(self['leftSpell']))
            buf.write(intToString(self['rightSpell']))

        elif self.msgtype == MSG_SEND_GESTURES_SEEN:
            pass
        elif self.msgtype == MSG_ASK_SPELL_DIRECTIONS:
            pass

        elif self.msgtype == MSG_RCV_SPELL_DIRECTIONS:
            buf.write(intToString(self['leftTarget']))
            buf.write(intToString(self['rightTarget']))

        elif self.msgtype == MSG_SEND_SPELL_CAST:
            pass
        elif self.msgtype == MSG_SEND_CREATURE_STATE:
            pass
        elif self.msgtype == MSG_SEND_END_GAME:
            pass
        elif self.msgtype == MSG_SEND_NEWPLAYER_INFO:
            pass
        elif self.msgtype == MSG_ASK_MONSTER_DIRECTIONS:
            pass
        elif self.msgtype == MSG_RCV_MONSTER_DIRECTIONS:
            monsterDirections = self['monsterDirections']
            buf.write(intToString(len(monsterDirections)))
            for i in range(len(monsterDirections)):
                md = monsterDirections[i]
                buf.write(intToString(md[0]))
                buf.write(intToString(md[1]))

        elif self.msgtype == MSG_SEND_NEW_MONSTER_INFO:
            pass
        elif self.msgtype == MSG_SEND_MONSTER_ATTACK_INFO:
            pass
        elif self.msgtype == MSG_SEND_EVENT_INFO:
            pass
        elif self.msgtype == MSG_SEND_START_GAME:
            pass

        elif self.msgtype == MSG_ASK_CHARM_PERSON_CTRL_HAND:
            pass
        elif self.msgtype == MSG_ASK_PARALYSIS_CTRL_HAND:
            pass
        elif self.msgtype == MSG_ASK_CHARM_PERSON_CTRL_GESTURE:
            pass
        elif self.msgtype == MSG_RCV_CHARM_PERSON_CTRL_HAND:
            buf.write(intToString(self['target']))
            buf.write(intToString(self['hand']))

        elif self.msgtype == MSG_RCV_PARALYSIS_CTRL_HAND:
            buf.write(intToString(self['target']))
            buf.write(intToString(self['hand']))

        elif self.msgtype == MSG_RCV_CHARM_PERSON_CTRL_GESTURE:
            buf.write(intToString(self['target']))
            buf.write(intToString(self['gesture']))

        s = buf.getvalue()
        buf.close()
        return SC_MSG_START_string + \
               intToString(len(s)+16) + \
               intToString(self.msgtype) + \
               s + \
               SC_MSG_END_string


class SpellcastConnection:

    def __init__(self, host="localhost", port=DEFAULT_SPELLCAST_SERVER_PORT):
        self.host = host
        self.port = port
        self.gamesocket = None
        self.savedData = ''
        self.lastSocketError = 0


    def connect(self):
        self.gamesocket = socket(AF_INET, SOCK_STREAM)
        self.gamesocket.connect( (self.host, self.port) )
        self.gamesocket.setblocking(0)


    def getConnectedPortString(self):
        if self.gamesocket is not None:
            sockname = self.gamesocket.getsockname()
            return "(%s:%d)" % sockname
        return "(unknown)"


    def disconnect(self):
        self.gamesocket.close()


    def readInt(self):
        bytes = read(self.gamesocket.fileno(), 4)
        iint = stringToInt(bytes)
        d("readInt() returns %x" % iint)
        return iint


    def readStartMarker(self):
        startmarker = self.readInt()
        d("startmarker is %x" % startmarker)
        if startmarker != SC_MSG_START:
            raise SpellcastProtocolException("bad start marker: 0x%08x" % startmarker)


    def readMessage(self):
        '''Read a message, if a full message is available.'''

        if self.gamesocket is None: return None

        try:
            s = self.gamesocket.recv(100)  # Try 100 bytes...
            if len(s) > 0:
                self.savedData += s
            elif len(s) == 0:
                raise Exception("Connection lost.")
            self.lastSocketError = 0
            
        except socket_error, reason:
            if reason[0] == EAGAIN:
                # EAGAIN means that there is no data.  This isn't
                # really an error for a non-blocking socket, so ignore
                # it.
                pass
            else:
                # Report other errors.
                raise

        # Not enough data for a message, try later.
        if len(self.savedData) < 16: return None

        startMarker = stringToInt(self.savedData)
        if startMarker != SC_MSG_START:
            self.savedData = ''
            raise SpellcastProtocolException("Bad start marker: %x" % startMarker)

        msgLen = stringToInt(self.savedData, 4)
        if msgLen > 100:
            self.savedData = ''
            raise SpellcastProtocolException("Message length too large (>100): %d" % msgLen)

        # Not enough data, try again later.
        if len(self.savedData) < msgLen:
            return None

        endMarker = stringToInt(self.savedData, msgLen-4)
        if endMarker != SC_MSG_END:
            self.savedData = ''
            raise SpellcastProtocolException("Bad end marker: %x" % endMarker)

        # Remove the available data out of the savedData stream before creating
        # the message so we don't get endless exceptions from the same source.
        messageData = self.savedData[0:msgLen]
        self.savedData = self.savedData[msgLen:]
        msg = SpellcastMessage(data=messageData)
        return msg


    def sendMessage(self, message):
        s = message.makeProtocolString()
        write(self.gamesocket.fileno(), s)


# Add stuff to exported symbols.
for key in locals().keys():
    if key.startswith('GST_') \
       or key.startswith('MSG_') \
       or key.startswith('CS_') \
       or key.startswith('SC_'):
        __all__.append(key)


# Local variables: ***
# mode:python ***
# py-indent-offset:4 ***
# fill-column:80 ***
# End: ***
### arch-tag: fce29b10-e083-4e6c-afe2-487a6d3524dc

