#!/usr/bin/env python2.3

# vim: set shiftwidth=4 tabstop=4 expandtab smartindent textwidth=80:

# Part of ecar, an entry for the LCA 2005 hackfest:
# http://lca2005.linux.org.au/hackfest/
# (C) 2005, Russell Steicke, <russells@adelie.cx>
# Licensed under GPL, see the file COPYING.


# Make us faster, if available.
try:
    import psyco
    psyco.full()
except ImportError, reason:
    pass
import sys
import gtk
import socket
from random import randint

# Debugging
df = False

def d(s):
    global df
    if df: print s

# Local stuff
from hflib.protocol import *
from hflib.Creature import Creature

__all__ = ['GestureImage', 'GestureSelectWindow', 'Player']

NumOldImages = 8

# Images for gestures.  This is a dictionary, not a tuple, so it can be modified
# when the user selects a different theme, and to make it resistant to changes
# in the enumeration order of the gestures.
leftGestureImageFileNamesDictionary = \
        { GST_NOTHING    : 'themes/default/None.png',
          GST_KNIFE      : 'themes/default/LeftKnife.png',
          GST_FINGER     : 'themes/default/LeftFingers.png',
          GST_PALM       : 'themes/default/LeftPalm.png',
          GST_SNAP       : 'themes/default/LeftSnap.png',
          GST_WAVE       : 'themes/default/LeftWave.png',
          GST_POINT      : 'themes/default/LeftPoint.png',
          GST_CLAP       : 'themes/default/LeftClap.png',
          GST_ANTISPELL  : 'themes/default/Antispell.png',
          GST_FOG        : 'themes/default/Fog.png',
        }
rightGestureImageFileNamesDictionary = \
        { GST_NOTHING    : 'themes/default/None.png',
          GST_KNIFE      : 'themes/default/RightKnife.png',
          GST_FINGER     : 'themes/default/RightFingers.png',
          GST_PALM       : 'themes/default/RightPalm.png',
          GST_SNAP       : 'themes/default/RightSnap.png',
          GST_WAVE       : 'themes/default/RightWave.png',
          GST_POINT      : 'themes/default/RightPoint.png',
          GST_CLAP       : 'themes/default/RightClap.png',
          GST_ANTISPELL  : 'themes/default/Antispell.png',
          GST_FOG        : 'themes/default/Fog.png',
        }



class GestureSelectWindow(gtk.Window):

    def __init__(self, right=False):
        gtk.Window.__init__(self, gtk.WINDOW_POPUP)
        if right:
            gestureImageFileNamesDictionary = rightGestureImageFileNamesDictionary
        else:
            gestureImageFileNamesDictionary = leftGestureImageFileNamesDictionary
        self.set_type_hint(gtk.gdk.WINDOW_TYPE_HINT_UTILITY)
        self.set_decorated(False)
        self.set_position(gtk.WIN_POS_MOUSE)
        self.set_modal(True)
        self.set_property("border-width", 3)
        self.spellSelectTable = gtk.Table()
        for gestureNumber in range(0,8):
            image = gtk.Image()
            image.set_from_file(gestureImageFileNamesDictionary[gestureNumber])
            eventBox = gtk.EventBox()
            eventBox.add(image)
            eventBox.connect("button-release-event",
                             self.gestureSelectWindowImagePressed, gestureNumber)

            xl = gestureNumber%4
            yt = (gestureNumber/4)%2
            #d("GestureSelectWindow attaching image %d at xl=%d yt=%d" % (gestureNumber,xl,yt))
            self.spellSelectTable.attach(eventBox, xl, xl+1, yt, yt+1, xpadding=3, ypadding=3)
        self.modify_bg(gtk.STATE_NORMAL, gtk.gdk.Color(20000,20000,20000))
        self.add(self.spellSelectTable)
        self.spellSelectTable.show_all()


    def setGestureImage(self, gestureImage):
        self.gestureImage = gestureImage


    def gestureSelectWindowImagePressed(self, w, e, gestureNumber):
        d("Selected gestureNumber==%d" % gestureNumber)
        self.gestureImage.setGestureNumber(gestureNumber)
        self.hide()


leftGestureSelectWindow = GestureSelectWindow(right=False)
rightGestureSelectWindow = GestureSelectWindow(right=True)


class GestureImage(gtk.EventBox):

    def __init__(self, selectable=False, right=False):
        gtk.EventBox.__init__(self)
        self.setRight(right)
        self.image = gtk.Image()
        self.image.set_from_file(self.gestureImageFileNamesDictionary[0])
        self.add(self.image)
        if selectable:
            self.connect("button-press-event", self.buttonPressed)
        self.gestureNumber = 0
        self.active = False


    def buttonPressed(self, widget, event):
        d("GestureImage.buttonPressed()")
        if self.active:
            self.selectWindow.setGestureImage(self)
            self.selectWindow.show()


    def setGestureNumber(self, gestureNumber):
        if gestureNumber >= len(self.gestureImageFileNamesDictionary)  or  gestureNumber < 0:
            self.image.set_from_file('ImageQuestion.png')
            self.gestureNumber = 0
        else:
            self.image.set_from_file(self.gestureImageFileNamesDictionary[gestureNumber])
            self.gestureNumber = gestureNumber


    def getGestureNumber(self):
        return self.gestureNumber


    def setActive(self, active):
        self.active = active


    def getActive(self):
        return self.active


    def setRight(self, right):
        if right:
            self.gestureImageFileNamesDictionary = rightGestureImageFileNamesDictionary
            self.selectWindow = rightGestureSelectWindow
        else:
            self.gestureImageFileNamesDictionary = leftGestureImageFileNamesDictionary
            self.selectWindow = leftGestureSelectWindow


class Player(Creature):

    def __init__(self, name):

        Creature.__init__(self, name)
        d("new Player(\"%s\")" % name)

        # EventBox is required for tooltips.
        self.tooltipEventBox = gtk.EventBox()
        self.frame = gtk.Frame()
        table = gtk.Table(homogeneous=False)

        self.leftHandImages = []
        self.rightHandImages = []

        y1 = NumOldImages
        y2 = NumOldImages+1
        #d("Attaching current image at %d,%d" % (y1,y2))
        self.leftHandImages.append(GestureImage(selectable=True, right=False))
        table.attach(self.leftHandImages[0], 0,1, y1, y2, xpadding=5, ypadding=5)
        self.rightHandImages.append(GestureImage(selectable=True, right=True))
        table.attach(self.rightHandImages[0], 1,2, y1, y2, xpadding=5, ypadding=5)

        for i in range(1, NumOldImages+1):
            y1 = NumOldImages-i
            y2 = NumOldImages-i+1
            #d("Attaching old images at %d,%d" % (y1,y2))
            lhi = GestureImage(right=False)
            self.leftHandImages.append(lhi)
            table.attach(self.leftHandImages[i], 0,1, y1, y2, xpadding=5, ypadding=5)
            rhi = GestureImage(right=True)
            self.rightHandImages.append(rhi)
            table.attach(self.rightHandImages[i], 1,2, y1, y2, xpadding=5, ypadding=5)

        self.gestureButton = gtk.Button("Gestures")
        table.attach(self.gestureButton, 0,2, 19,20, xpadding=5, ypadding=5)
        self.gestureButton.connect("clicked", self.gestureButtonClicked)

        self.healthLabel = gtk.Label("Health: ")
        table.attach(self.healthLabel, 0,2, 20,21, xpadding=5, ypadding=5)
        self.stateLabel = gtk.Label("Normal\n\n\n\n")
        self.stateLabel.set_use_markup(True)
        self.stateLabel.set_property("use-markup", True)
        self.stateLabel.set_markup('''<span size="small">Normal\n\n\n\n</span>''')
        table.attach(self.stateLabel, 0,2, 21,22, xpadding=5, ypadding=5)

        self.frame.add(table)

        self.tooltips = gtk.Tooltips()
        self.tooltips.enable()
        self.tooltipEventBox.add(self.frame)
        self.widget = self.tooltipEventBox
        self.setName(name)
        self.setNumber(-1)
        self.setState(0)

        self.gestureHandler = None


    def gesturesSeen(self,
                     next=(GST_NOTHING,GST_NOTHING),
                     new=GST_NOTHING):
        '''Move all the images up, so we are ready to start a new round.'''
        d("%s: new gestures: (%s,%s)" % (str(self),
                                         gestureTypeToName(next[0]),
                                         gestureTypeToName(next[1])))
        for i in range(NumOldImages-1, 0, -1):
            oldLeftGesture = self.leftHandImages[i].getGestureNumber()
            oldRightGesture = self.rightHandImages[i].getGestureNumber()
            self.leftHandImages[i+1].setGestureNumber(oldLeftGesture)
            self.rightHandImages[i+1].setGestureNumber(oldRightGesture)
        self.leftHandImages[1].setGestureNumber(next[0])
        self.rightHandImages[1].setGestureNumber(next[1])
        self.leftHandImages[0].setGestureNumber(new)
        self.rightHandImages[0].setGestureNumber(new)


    def gestureButtonClicked(self, button):
        d("gestureButton clicked")
        if self.gestureHandler is not None:
            self.gestureHandler( (self.leftHandImages[0].getGestureNumber(),
                                  self.rightHandImages[0].getGestureNumber()) )


    def setGestureHandler(self, gestureHandler):
        self.gestureHandler = gestureHandler


    def setName(self, newName):
        Creature.setName(self, newName)
        self.tooltips.set_tip(self.tooltipEventBox, "Player: "+newName)
        if len(newName) > 8:
            self.frame.set_label(newName[:7]+"...")
        else:
            self.frame.set_label(newName)


    def getWidget(self):
        return self.widget


    def getHealth(self):
        return self.health


    def setHealth(self, health):
        d("%s: setting health to %s" % (str(self), str(health)))
        self.health = health
        self.healthLabel.set_text("Health: "+str(health))


    def setState(self, state):
        '''Show the player state in its widget.  The state is show as text.'''
        Creature.setState(self, state)
        d("%s: setting state to %s" % (str(self), str(state)))
        # The state string should always have four newlines in it.  This number
        # allows for five different states to be shown without the size of the
        # label changing.  I guessed that this will be enough.
        if state == 0:
            statetext = "Normal\n\n\n\n"
        else:
            statetexts = []
            statetext = ""
            for n in range(32):
                cstype = state & (1<<n)
                if cstype != 0:
                    s = creatureStateTypeToName(cstype)
                    if s is not None:
                        # Here we add newlines _between_ the bits of text, so
                        # that there isn't a trailing newline, which would make
                        # the text take up one more line that would never have
                        # any text on it.
                        if len(statetexts) == 0:
                            statetexts.append(s)
                        else:
                            statetexts.append("\n" + s)
            # But here we add trailing newlines to take the total to five.  This
            # logic probably needs a good looking at.
            while len(statetexts) < 5:
                statetexts.append("\n")
            for s in statetexts:
                statetext = statetext + s
        statetext = '''<span size="small">'''+statetext+'''</span>'''
        self.stateLabel.set_markup(statetext)


    def setGettingGestures(self, getting):
        '''Set to true when we want the user to input the gestures.  Set to
        false afterwards so the user cannot input gestures until the next time.'''
        d("%s.setGettingGestures(%s)" % (str(self), str(getting)))
        self.leftHandImages[0].setActive(getting)
        self.rightHandImages[0].setActive(getting)
        self.gestureButton.set_sensitive(getting)


    def getGestures(self):
        '''Return a 2-tuple of left and right gesture numbers.'''
        return (self.leftHandImages[0].getGestureNumber(),
                self.rightHandImages[0].getGestureNumber())


    def __str__(self):
        return "<Player:%s>" % (self.name,)


# Local variables: ***
# mode:python ***
# py-indent-offset:4 ***
# fill-column:80 ***
# End: ***
### arch-tag: 72f5c829-5d9f-4496-a4e5-e9485f307564

