#define DEBUG
#define VERBOSE

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <time.h>

//#define SMP

#ifdef SMP
#define NUM_WORKERS 4
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <errno.h>
#include "message.h"
#endif

#include "gestures.h"
#include "spells.h"
#include "wizard.h"
#include "moves.h"
#include "lookup8.h"

#ifdef SMP
int max_fd = 0;
int workers[NUM_WORKERS];
#endif

void show_move(Move *move) {
  printf("move is %c%c",
         gesture_print[LH(move->gl) & 0x0F],
         gesture_print[RH(move->gl) & 0x0F]);
  if (move->paralysis_hand || move->paralysis_who) {
    printf(" p%d/%d", move->paralysis_hand, move->paralysis_who);
  }
  if (move->sp) {
    printf(" spells %s (%d) %s (%d)",
           spell_name[(move->sp >> 16) & 0xFF], move->sp >> 24,
           spell_name[move->sp & 0xFF], (move->sp >> 8) & 0xFF);
  }
  printf("\n");
}

void show_movelist(Move *movelist, int num_moves) {
  int i;
  unsigned short l, r;

  printf("%d moves:\n", num_moves);
  for (i = 0; i < num_moves; i++) {
    printf("%2d (", i);
    if (movelist[i].who) {
      printf("w%d ", movelist[i].who);
    }
    if (movelist[i].paralysis_hand || movelist[i].paralysis_who) {
      printf("p%d/%d ", movelist[i].paralysis_hand, movelist[i].paralysis_who);
    }
    printf("s%d): ", movelist[i].score);
    print_gesture_list(movelist[i].gl);
    l = movelist[i].sp >> 16;
    r = movelist[i].sp & 0xFFFF;
    printf("spell %s (%d) + %s (%d)\n",
           spell_name[l & 0xFF], l >> 8,
           spell_name[r & 0xFF], r >> 8);
  }
}

void show_move_short(Move *move) {
  printf("%c%c",
         gesture_print[LH(move->gl) & 0x0F],
         gesture_print[RH(move->gl) & 0x0F]);
  if (move->paralysis_hand) {
    printf(" p%d", move->paralysis_hand);
  }
  if (move->sp) {
    printf(" %d/%d", move->sp >> 16, move->sp & 0xFFFF);
  }
}

void show_movelist_short(Move *movelist, int num_moves) {
  int i;

  for (i = 0; i < num_moves; i++) {
    printf("%d:", i);
    show_move_short(&movelist[i]);
    printf(" ");
  }
  printf("\n");
}

/* - only apply resist heat to fire elemental
 * - only apply resist cold to ice elemental
 * - only apply charm monster to normal elemental
 */
int can_apply_spell_to_monster(int *monster_num, int spell_id) {
  if (spell_id == SPL_CHARM_MONSTER) {
    if (monster_num[0]) {
      return 1;
    }
  } else if (spell_id == SPL_RESIST_HEAT) {
    if (monster_num[1]) {
      return 1;
    }
  } else if (spell_id == SPL_RESIST_COLD) {
    if (monster_num[2]) {
      return 1;
    }
  } else if (monster_num[0] || monster_num[1] || monster_num[2]) {
    return 1;
  }
  return 0;
}

/* return a list (ended by -1) of spells that we might want to cast */
int *get_self_spell_targets(WizardState *ws, int num_wizards, int spell_id) {
  int targets[100];
  int *target_p = targets;
  int num_targets;
  int i;

  /* - if spell_who is 0 or spell_either is 1, start at wizard 0
   *   (otherwise start at wizard 1)
   * - if spell_who is 1 or spell_either is 1, finish after
   *   num_wizards (otherwise finish on wizard 0)
   */
  for (i = (((spell_who[spell_id] == 0) || spell_either[spell_id]) ? 0 : 1);
       i < ((spell_who[spell_id] || spell_either[spell_id]) ? num_wizards : 1);
       i++) {

    /* apply spell to wizards */
    if (spell_monster[spell_id] == 0 || spell_monster[spell_id] == 2) {
      *target_p++ = i;
    }

    /* apply spell to monsters */
    if (spell_monster[spell_id] == 1 || spell_monster[spell_id] == 2) {
      if (can_apply_spell_to_monster(ws[i].monster_num, spell_id)) {
        *target_p++ = i | (1 << 4);
      }
    }
  }

  /* if no spells applied, add SPL_NONE */
  if (target_p == targets) {
    *target_p++ = 0;
  }

  *target_p++ = -1;
  num_targets = target_p - targets;
  target_p = malloc(sizeof(int) * num_targets);
  memcpy(target_p, targets, sizeof(int) * num_targets);
  return target_p;
}

/* return a list (ended by -1) of spells that our opponent might cast */
int *get_other_spell_targets(WizardState *ws, int wizard_num, int spell_id) {
  int targets[100];
  int *target_p = targets;
  int num_targets;
  int i;

  /* always apply SPL_NONE to wizard 0 */
  if (spell_id == SPL_NONE) {
    target_p = malloc(sizeof(int) * 2);
    target_p[0] = SPL_NONE;
    target_p[1] = -1;
    return target_p;
  }

  /* similar to get_self_spell_targets, except i is either 0 or 1
   * 0: apply to wizard_num
   * 1: apply to us (0)
   */
  for (i = (((spell_who[spell_id] == 0) || spell_either[spell_id]) ? 0 : 1);
       i < ((spell_who[spell_id] || spell_either[spell_id]) ? 2 : 1);
       i++) {

    /* convert from 0 or 1 to wizard_num or 0 respectively */
    int target = (!i) * wizard_num;

    /* apply spell to wizards */
    if (spell_monster[spell_id] == 0 || spell_monster[spell_id] == 2) {
      *target_p++ = target;
    }

    /* apply spell to monsters
     * (don't apply charm monster to elementals)
     */
    if (spell_monster[spell_id] == 1 || spell_monster[spell_id] == 2) {
      if (can_apply_spell_to_monster(ws[target].monster_num, spell_id)) {
        *target_p++ = target | (1 << 4);
      }
    }
  }

  /* if no spells applied, add SPL_NONE */
  if (target_p == targets) {
    *target_p++ = 0;
  }

  *target_p++ = -1;
  num_targets = target_p - targets;
  target_p = malloc(sizeof(int) * num_targets);
  memcpy(target_p, targets, sizeof(int) * num_targets);
  return target_p;
}

static void get_movelist(
  WizardState *ws, int num_wizards, int wizard_num,
  int paralysis_hand, int paralysis_who,
  Move **movelist_out, int *num_moves_out
) {

  int state = ws[wizard_num].state;
  GestureList gl = ws[wizard_num].gl;

  int fear = state & CS_FEAR;
  GesturePair *gp;
  GesturePair poss_gp_bad[2] = { 0, 0 };

  /* list of possible moves (gestures + chosen spell) */
  Move movelist[1000];
  Move *movelist_p = movelist;

  /* list of possible SpellPairs that can result from the gesture
   * we're currently looking at
   */
  SpellPair sp[MAX_SPELL_LIST + 1];
  SpellPair *sp_p;

  /* if paralysed, need to consider more moves in case we're
   * paralysing a "nothing" gesture or a "half-clap" gesture which
   * isn't in poss_gp[]
   */
  if (state & CS_PARALYSIS) {
    gp = poss_gp_para;

  /* if amnesia, the only move is the most-recent one that was made */
  } else if (state & CS_AMNESIA) {
    GestureList last_gp = gl;
    while (((last_gp & 0xFF) == make_gp(GST_FOG, GST_FOG)) ||
           ((last_gp & 0xFF) == make_gp(GST_ANTISPELL, GST_ANTISPELL))) {
      last_gp >>= 8;
    }

    gp = poss_gp_bad;
    poss_gp_bad[0] = last_gp & 0xFF;
    fear = 0; /* just in case */

  } else if (state & CS_CONFUSION) {
    gp = poss_gp_confused;
  } else if (((gl & 0xFF) == 0) ||
             ((gl & 0xFF) == make_gp(GST_ANTISPELL, GST_ANTISPELL)) ||
             ((gl & 0xFF) == make_gp(GST_CLAP, GST_CLAP))) {
    gp = poss_gp_noswap;
  } else {
    gp = poss_gp;
  }

  for (; *gp; gp++) {
    GestureList new_gl;

    if (movelist_p - movelist > 1900) {
      printf("warning: getting close to end of movelist array\n");
    }

    /* if fearful, there are certain gestures that can't be made */
    if (fear && (
          (*gp == make_gp(GST_CLAP, GST_CLAP)) ||
          (LH(*gp) == GST_POINT) ||
          (RH(*gp) == GST_POINT) ||
          (LH(*gp) == GST_FINGER) ||
          (RH(*gp) == GST_FINGER) ||
          (LH(*gp) == GST_SNAP) ||
          (RH(*gp) == GST_SNAP)
       )) {
      continue;
    }

    /* put the new gesture onto the GestureList and see what spells we
     * can cast
     */
    new_gl = gl_push(gl, *gp);
    get_spellpair_list(new_gl, sp);

    /* can't cast any spells? */
    if (*sp == 0) {
      movelist_p->gl = new_gl;
      movelist_p->sp = 0;
      movelist_p->paralysis_hand = 0;
      movelist_p->paralysis_who = 0;
      movelist_p->who = wizard_num;
      movelist_p->score = 0;
      movelist_p++;
      continue;
    }

    /* can cast at least one spell; go through them all */
    for (sp_p = sp; *sp_p; sp_p++) {
      int score;
      int *left_target_list, *right_target_list;
      int left_target_count, right_target_count;

      assert(((*sp_p >> 16) & 0xFF) < SPL_FINAL_MARKER);
      assert((*sp_p & 0xFF) < SPL_FINAL_MARKER);

      score = spell_score[(*sp_p >> 16) & 0xFF] + spell_score[*sp_p & 0xFF];

      if (wizard_num == 0) {
        left_target_list =
            get_self_spell_targets(ws, num_wizards, (*sp_p >> 16) & 0xFF);
        right_target_list =
            get_self_spell_targets(ws, num_wizards, *sp_p & 0xFF);
      } else {
        left_target_list =
            get_other_spell_targets(ws, wizard_num, (*sp_p >> 16) & 0xFF);
        right_target_list =
            get_other_spell_targets(ws, wizard_num, *sp_p & 0xFF);
      }

      for (left_target_count = 0;
           left_target_list[left_target_count] != -1;
           left_target_count++) {
        int left_target = left_target_list[left_target_count];

        for (right_target_count = 0;
             right_target_list[right_target_count] != -1;
             right_target_count++) {
          int right_target = right_target_list[right_target_count];

          movelist_p->gl = new_gl;
          movelist_p->sp = *sp_p | (left_target << 24) | (right_target << 8);
          movelist_p->paralysis_hand = 0;
          movelist_p->paralysis_who = 0;
          movelist_p->who = wizard_num;
          movelist_p->score = score;
          movelist_p++;
        }
      }

      free(left_target_list);
      free(right_target_list);
    }
  }

  *num_moves_out = movelist_p - movelist;
  assert(*num_moves_out < 1000);

  /* this could happen if amnesia is cast on someone who just gestured
   * nn
   */
  if (*num_moves_out == 0) {
    movelist_p->gl = gl_push(gl, 0);
    movelist_p->sp = 0;
    movelist_p->paralysis_hand = 0;
    movelist_p->paralysis_who = 0;
    movelist_p->who = wizard_num;
    movelist_p->score = 0;
    movelist_p++;
    (*num_moves_out)++;
  }

  /* if we're not paralysing someone's hand or if we don't have a
   * choice about which hand to paralyse
   */
  if (paralysis_hand >= 0) {
    *movelist_out = malloc(sizeof(Move) * *num_moves_out);
    memcpy(*movelist_out, movelist, sizeof(Move) * *num_moves_out);

    /* if we are paralysing someone's hand, set that state */
    if (paralysis_hand != 0) {
      int i;
      for (i = 0, movelist_p = *movelist_out;
           i < *num_moves_out;
           i++, movelist_p++) {
        movelist_p->paralysis_hand = paralysis_hand;
        movelist_p->paralysis_who = paralysis_who;
      }
    }

  /* if we are paralysing someone's hand and we have a choice of which
   * hand, duplicate the movelist, setting paralysis_hand to 1 for the
   * original (left) and 2 for the copy (right)
   */
  } else {
    int i;

    *movelist_out = malloc(sizeof(Move) * *num_moves_out * 2);
    memcpy(*movelist_out, movelist,
           sizeof(Move) * *num_moves_out);
    memcpy(*movelist_out + *num_moves_out, movelist,
           sizeof(Move) * *num_moves_out);
    movelist_p = *movelist_out;
    for (i = 0; i < *num_moves_out; i++) {
      movelist_p++->paralysis_hand = 1;
    }
    for (i = 0; i < *num_moves_out; i++) {
      movelist_p++->paralysis_hand = 2;
    }
    *num_moves_out *= 2;
  }

//show_movelist(*movelist_out, *num_moves_out);
}

static void get_movelists(
  WizardState *ws, int num_ws,
  int us_paralysed,
  Move **movelist_out, int *num_moves_out
) {
  Move *movelist_tmp;
  int  movelist_tmp_moves;
  Move *movelist_combined = NULL;
  int i;
  int movelist_count = 0;

  for (i = 1; i < num_ws; i++) {
    get_movelist(ws, num_ws, i,
                 us_paralysed, 0,
                 &movelist_tmp, &movelist_tmp_moves);
    movelist_combined = realloc(movelist_combined,
                                (movelist_count + movelist_tmp_moves) *
                                    sizeof(Move));
    memcpy(&movelist_combined[movelist_count],
           movelist_tmp,
           movelist_tmp_moves * sizeof(Move));
    free(movelist_tmp);
    movelist_count += movelist_tmp_moves;
  }

  *movelist_out = movelist_combined;
  *num_moves_out = movelist_count;
}

static int compare_moves(const void *a, const void *b) {
  return ((Move *)a)->score < ((Move *)b)->score ? 1 :
         ((Move *)a)->score > ((Move *)b)->score ? -1 : 0;
}

static void sort_movelist(Move *movelist, int num_moves) {
  qsort(movelist, num_moves, sizeof(Move), &compare_moves);
}

static void sort_spells(unsigned short *spells) {
  int swapped, i;

  /* bubble biggest spell to the top */
  swapped = 0;
  for (i = 0; i < 3; i++) {
    if (spell_order[spells[i] & 0xFF] > spell_order[spells[i + 1] & 0xFF]) {
      unsigned short temp = spells[i];
      spells[i] = spells[i + 1];
      spells[i + 1] = temp;
      swapped = 1;
    }
  }
  if (!swapped) {
    return;
  }

  /* bubble smallest spell to the bottom */
  swapped = 0;
  for (i = 1; i >= 0; i--) {
    if (spell_order[spells[i] & 0xFF] > spell_order[spells[i + 1] & 0xFF]) {
      unsigned short temp = spells[i];
      spells[i] = spells[i + 1];
      spells[i + 1] = temp;
      swapped = 1;
    }
  }
  if (!swapped) {
    return;
  }

  /* swap middle two if necessary */
  if (spell_order[spells[1] & 0xFF] > spell_order[spells[2] & 0xFF]) {
    unsigned short temp = spells[1];
    spells[1] = spells[2];
    spells[2] = temp;
    swapped = 1;
  }
}

/* called on the first mirror found -- only applies to spells after it
 * in the spell list (spells[mirror_spell + 1]) because the list has
 * been sorted
 */
void handle_magic_mirror(unsigned short *spells, int mirror_spell,
                         int their_id) {
  int i;
  int mirror_target = spells[mirror_spell] >> 8;

  /* cancel any other mirrors (since Delayed Effect isn't implemented
   * and the Magic Mirror spell requires two hands, a spell can't be
   * invoked and bounce between two mirrors
   */
  for (i = mirror_spell + 1; i < 4; i++) {
    if ((spells[i] & 0xFF) == SPL_MAGIC_MIRROR) {
      spells[i] = 0;
    }
  }

  /* a mirror reflects any spell from any other person which is cast
   * at the mirror target except spells which are usually cast on self
   * (spell_who[spell] == 0) and SPL_MAGIC_MIRROR (already handled),
   * SPL_STAB, and SPL_REMOVE_ENCHANTMENT
   */
  for (i = mirror_spell + 1; i < 4; i++) {
    unsigned char spell = spells[i] & 0xFF;
    unsigned char spell_target = (spells[i] >> 8) & 0xFF;
    if (spell_target == mirror_target &&
        spell_who[spell] &&
        spell != SPL_STAB &&
        spell != SPL_REMOVE_ENCHANTMENT) {
      spell_target = spell_target ? 0 : their_id;
      spells[i] = (spells[i] & 0x00FF) | (spell_target << 8);
    }
  }
}

void apply_no_spells(WizardState *ws) {
  int who;

  for (who = 0; who < 2; who++) {
    ws[who].paralysed = 0;
    ws[who].state &= ~CS_ONCE_OFF;

    if (ws[who].state & CS_PROT_FROM_EVIL) {
      if (--ws[who].prot_from_evil <= 0) {
        ws[who].state &= ~CS_PROT_FROM_EVIL;
      }
    }
    if (ws[who].state & CS_DISEASED) {
      if (--ws[who].disease <= 0) {
        ws[who].state |= CS_DEAD;
      }
    }
    if (ws[who].state & CS_POISONED) {
      if (--ws[who].poison <= 0) {
        ws[who].state |= CS_DEAD;
      }
    }
    if (ws[who].state & CS_BLINDED) {
      if (--ws[who].blind <= 0) {
        ws[who].state &= (~CS_BLINDED);
      }
    }
    if (ws[who].state & CS_INVISIBILITY) {
      if (--ws[who].invisible <= 0) {
        ws[who].state &= (~CS_INVISIBILITY);
      }
    }
  }
}

void apply_spell_charm_monster(WizardState *ws, int who, int their_id) {
  /* the monster may have been killed by another spell before we charm it */
  if (!ws[who].monster_num[0]) {
    return;
  }

  /* guess how much a charmed monster is worth
   * ignore elementals -- no point charming them
   */
  int hp = ws[who].monster_hp[0] / ws[who].monster_num[0];
  int max_hp = ws[who].monster_max_hp[0] / ws[who].monster_num[0];
  int from;
  int to;

  /* someone else charming one of our monsters? */
  if (who == 0) {
    from = 0;
    to = their_id;

  /* us charming someone else's monsters */
  } else {
    from = who;
    to = 0;
  }

  ws[from].monster_num[0]--;
  ws[from].monster_hp[0] -= hp;
  ws[from].monster_max_hp[0] -= max_hp;
  ws[to].monster_num[0]++;
  ws[to].monster_hp[0] += hp;
  ws[to].monster_max_hp[0] += max_hp;
}

void inflict_monster_damage(WizardState *ws, int damage) {
  int type;

  for (type = 0; type < 3; type++) {
    if (!ws->monster_num[type]) {
      continue;
    }
    int hp = ws->monster_hp[type] / ws->monster_num[type];
    int max_hp = ws->monster_max_hp[type] / ws->monster_num[type];

    if (damage < hp) {
      ws->monster_hp[type] -= damage;
    } else {
      ws->monster_num[type]--;
      ws->monster_hp[type] -= hp;
      ws->monster_max_hp[type] -= max_hp;
    }
    break;
  }
}

void apply_spell_to_monster(WizardState *ws, int num_wizards, int their_id,
                            unsigned short *spells, int spell_num) {
  int spell = spells[spell_num] & 0xFF;
  int who = (spells[spell_num] >> 8) & 0x0F;

  switch (spell) {
    case SPL_MISSILE:
      inflict_monster_damage(&ws[who], 1);
      break;

    case SPL_LIGHTNING_BOLT:
    case SPL_LIGHTNING_BOLT1:
    case SPL_FIREBALL:
      inflict_monster_damage(&ws[who], 5);
      break;

    case SPL_CAUSE_LIGHT_WOUNDS:
      inflict_monster_damage(&ws[who], 2);
      break;

    case SPL_CAUSE_HEAVY_WOUNDS:
      inflict_monster_damage(&ws[who], 3);
      break;

    case SPL_CHARM_MONSTER:
      apply_spell_charm_monster(ws, who, their_id);
      break;

    case SPL_RESIST_HEAT:
      ws[who].monster_num[1] = 0;
      ws[who].monster_hp[1] = 0;
      ws[who].monster_max_hp[1] = 0;
      break;

    case SPL_RESIST_COLD:
      ws[who].monster_num[2] = 0;
      ws[who].monster_hp[2] = 0;
      ws[who].monster_max_hp[2] = 0;
      break;
  }
}

void apply_spell_to_wizard(WizardState *ws, int num_ws, int their_id,
                           unsigned short *spells, int spell_num,
                           int *multiple_enchantment) {
  int i;
  int spell = spells[spell_num] & 0xFF;
  int who = spells[spell_num] >> 8;
  int who_set = who ? 1 : 0;

  switch (spell) {
    case SPL_NONE:
      break;
    case SPL_MAGIC_MIRROR:
      handle_magic_mirror(spells, spell_num, their_id);
      break;

    case SPL_DISPEL_MAGIC:
      for (i = 0; i < num_ws; i++) {
        ws[i].state &= ~CS_REMOVE_ENCHANTMENTS;
        memset(ws[i].monster_num, 0, sizeof(ws[i].monster_num));
        memset(ws[i].monster_hp, 0, sizeof(ws[i].monster_hp));
        memset(ws[i].monster_max_hp, 0, sizeof(ws[i].monster_max_hp));
      }
      for (i = spell_num + 1; i < 4; i++) {
        spells[i] = SPL_NONE;
      }
      ws[who].state |= CS_SHIELD;
      break;

    case SPL_COUNTER_SPELL:
    case SPL_COUNTER_SPELL1:
      for (i = spell_num + 1; i < 4; i++) {
        /* nullify any spell pointing at the counter spell subject
         * except for finger of death (or dispel magic, but that has
         * already taken effect)
         */
        if ((who == (spells[i] >> 8)) &&
            ((spells[i] & 0xFF) != SPL_FINGER_OF_DEATH)) {
          spells[i] = SPL_NONE;
        }
      }
      /* Note: no break!
       * SPL_COUNTER_SPELL acts as a shield too
       */

    case SPL_SHIELD:
      ws[who].state |= CS_SHIELD;
      break;

    case SPL_STAB:
    case SPL_MISSILE:
      if (!(ws[who].state & (CS_SHIELD | CS_PROT_FROM_EVIL))) {
        ws[who].damage_inflicted++;
      }
      break;

    case SPL_SUMMON_GOBLIN:
      ws[who].monster_num[0]++;
      ws[who].monster_hp[0]++;
      ws[who].monster_max_hp[0]++;
      break;

    case SPL_SUMMON_OGRE:
      ws[who].monster_num[0]++;
      ws[who].monster_hp[0] += 2;
      ws[who].monster_max_hp[0] += 2;
      break;

    case SPL_SUMMON_TROLL:
      ws[who].monster_num[0]++;
      ws[who].monster_hp[0] += 3;
      ws[who].monster_max_hp[0] += 3;
      break;

    case SPL_SUMMON_GIANT:
      ws[who].monster_num[0]++;
      ws[who].monster_hp[0] += 4;
      ws[who].monster_max_hp[0] += 4;
      break;

    case SPL_SUMMON_FIRE_ELEMENTAL:
      ws[who].monster_num[1]++;
      ws[who].monster_hp[1] += 3;
      ws[who].monster_max_hp[1] += 3;
      break;

    case SPL_SUMMON_ICE_ELEMENTAL:
      ws[who].monster_num[1]++;
      ws[who].monster_hp[1] += 3;
      ws[who].monster_max_hp[1] += 3;
      break;

    case SPL_ANTI_SPELL:
      ws[who].gl = gl_push(ws[who].gl, make_gp(GST_ANTISPELL, GST_ANTISPELL));
      break;

    case SPL_CAUSE_LIGHT_WOUNDS:
      ws[who].damage_inflicted += 2;
      break;

    case SPL_CAUSE_HEAVY_WOUNDS:
      ws[who].damage_inflicted += 3;
      break;

    case SPL_CURE_LIGHT_WOUNDS:
      ws[who].damage_inflicted--;
      if (ws[who].damage_inflicted < 0) {
        ws[who].damage_inflicted = 0;
      }
      break;

    case SPL_CURE_HEAVY_WOUNDS:
      ws[who].damage_inflicted -= 2;
      if (ws[who].damage_inflicted < 0) {
        ws[who].damage_inflicted = 0;
      }
      ws[who].state &= ~CS_DISEASED;
      break;

    case SPL_PARALYSIS:
      if (ws[who].state & CS_ANY_ENCHANTMENT) {
        multiple_enchantment[who_set] = 1;
      }
      ws[who].state |= CS_PARALYSIS;
      ws[who].paralysed = -1;
      break;

    case SPL_FEAR:
      if (ws[who].state & CS_ANY_ENCHANTMENT) {
        multiple_enchantment[who_set] = 1;
      }
      ws[who].state |= CS_FEAR;
      break;

    case SPL_AMNESIA:
      if (ws[who].state & CS_ANY_ENCHANTMENT) {
        multiple_enchantment[who_set] = 1;
      }
      ws[who].state |= CS_AMNESIA;
      break;

    case SPL_PROTECTION_FROM_EVIL:
      ws[who].state |= CS_PROT_FROM_EVIL;
      ws[who].prot_from_evil = 3;
      break;

    case SPL_DISEASE:
      ws[who].state |= CS_DISEASED;
      ws[who].disease = 6;
      break;

    case SPL_POISON:
      ws[who].state |= CS_POISONED;
      ws[who].poison = 6;
      break;

    case SPL_BLINDNESS:
      ws[who].state |= CS_BLINDED;
      ws[who].blind = 3;
      break;

    case SPL_INVISIBILITY:
      ws[who].state |= CS_INVISIBILITY;
      ws[who].invisible = 3;
      break;

    case SPL_CONFUSION:
      ws[who].state |= CS_CONFUSION;
      break;

    case SPL_FINGER_OF_DEATH:
      ws[who].state |= CS_DEAD;
      break;

    case SPL_LIGHTNING_BOLT1:
    case SPL_LIGHTNING_BOLT:
      ws[who].damage_inflicted += 5;
      break;

    case SPL_FIREBALL:
      if (!(ws[who].state & CS_RESIST_HEAT)) {
        ws[who].damage_inflicted += 5;
      }
      break;

    case SPL_FIRESTORM:
      for (i = 0; i < num_ws; i++) {
        if (!(ws[i].state & CS_RESIST_HEAT)) {
          ws[i].damage_inflicted += 5;
        }
      }
      /* firestorm kills all monsters (except for heat-resistant ones,
       * but we don't handle that)
       */
      for (i = 0; i < num_ws; i++) {
        memset(ws[i].monster_num, 0, sizeof(ws[i].monster_num));
        memset(ws[i].monster_hp, 0, sizeof(ws[i].monster_hp));
        memset(ws[i].monster_max_hp, 0, sizeof(ws[i].monster_max_hp));
      }
      break;

    case SPL_ICESTORM:
      for (i = 0; i < num_ws; i++) {
        if (!(ws[i].state & CS_RESIST_COLD)) {
          ws[i].damage_inflicted += 5;
        }
        /* icestorm kills all monsters (except for cold-resistant
         * ones, but we don't handle that)
         */
        memset(ws[i].monster_num, 0, sizeof(ws[i].monster_num));
        memset(ws[i].monster_hp, 0, sizeof(ws[i].monster_hp));
        memset(ws[i].monster_max_hp, 0, sizeof(ws[i].monster_max_hp));
      }
      break;

    case SPL_RESIST_HEAT:
      ws[who].state |= CS_RESIST_HEAT;
      break;

    case SPL_RESIST_COLD:
      ws[who].state |= CS_RESIST_COLD;
      break;

    case SPL_CHARM_PERSON:
      /* treat like paralysis here for the moment -- when asked for
       * the gesture, we'll respond with a P on a random hand in the
       * hope that they surrender
       */
      if (ws[who].state & CS_ANY_ENCHANTMENT) {
        multiple_enchantment[who_set] = 1;
      }
      ws[who].state |= CS_PARALYSIS;
      ws[who].paralysed = -1;
      break;

    case SPL_CHARM_MONSTER:
      /* has no effect on wizards */
      break;

    case SPL_REMOVE_ENCHANTMENT:
      ws[who].state &= ~CS_REMOVE_ENCHANTMENTS;
      break;

    case SPL_RAISE_DEAD:
      /* ignore this -- there are easier ways to stop the
       * finger-of-death spell or to get back a monster
       */
      break;

    default:
      if (spell >= SPL_FINAL_MARKER) {
        printf("invalid spell %d\n", spell);
      } else {
        printf("unhandled spell %s\n", spell_name[spell]);
      }
  }
}

void apply_spells(WizardState *ws, int num_ws, int their_id,
                  SpellPair sp1, SpellPair sp2) {
  unsigned short spells[4];
  int i;
  int was_paralysed[2] = { ws[0].paralysed, ws[their_id].paralysed };
  int multiple_enchantment[2] = { 0, 0 };

  assert(their_id < num_ws);

  apply_no_spells(ws);

  if (ws[0].state & CS_DEAD) {
    return;
  }

  /* Split the spells from sp1 and sp2 into spells[] */
  spells[0] = sp1 >> 16;
  spells[1] = sp1 & 0xFFFF;
  spells[2] = sp2 >> 16;
  spells[3] = sp2 & 0xFFFF;

  for (i = 0; i < 4; i++) {
    if ((spells[i] & 0xFF) >= SPL_FINAL_MARKER) {
      printf("invalid spell %d/%d!\n", i, spells[i]);
    }
  }

  /* make sure SPL_NONE is always invoked on self */
  for (i = 0; i < 4; i++) {
    if ((spells[i] & 0xFF) == SPL_NONE) {
      spells[i] = SPL_NONE;
    }
  }

  /* ignore spells from us that are targetted at a different wizard */
  for (i = 0; i < 2; i++) {
    int target = (spells[i] >> 8) & 0x0F;

    assert(target < num_ws);

    if ((target > 0) && (target != their_id)) {
      spells[i] = 0;
    }
  }

  /* since we are pessimistic, we shouldn't find wizards targetting
   * other opponents of ours
   */
  for (i = 2; i < 4; i++) {
    int target = (spells[i] >> 8) & 0x0F;

    if (target >= num_ws) {
      printf("target is %d\n", target);
    }
    assert(target < num_ws);
    assert((target == 0) || (target == their_id));
  }

  /* simple sort on spells so we invoke them in the right order
   * note that sc-0.3 has the spells in a different order to SpellCast
   * so this will most likely be changing!
   */
  sort_spells(spells);

  for (i = 0; i < 4; i++) {
    if ((spells[i] >> 8) & 0x10) {
      apply_spell_to_monster(ws, num_ws, their_id, spells, i);
    } else {
      apply_spell_to_wizard(ws, num_ws, their_id, spells, i,
                            multiple_enchantment);
    }
  }

  for (i = 0; i < 2; i++) {
    int who_id = i ? their_id : 0;

    /* check for multiple enchantments*/
    if (multiple_enchantment[i]) {
      ws[who_id].state &= ~CS_ANY_ENCHANTMENT;
      ws[who_id].paralysed = 0;
    }

    /* if a hand was paralysed before, keep the same one paralysed */
    if (was_paralysed[i] && ws[who_id].paralysed) {
      ws[who_id].paralysed = was_paralysed[i];
    }
  }
}

static void check_deadness(WizardState *ws, int num_ws) {
  int who;

  for (who = 0; who < num_ws; who++) {
    if (ws[who].damage_inflicted >= ws[who].num_hit_points) {
      ws[who].state |= CS_DEAD;
    }
  }
}

static int static_score(WizardState *ws) {
#define QUICK
#ifdef QUICK
  return ws->monster_max_hp[0] + ws->monster_max_hp[1] + ws->monster_max_hp[2]
         - ws->damage_inflicted * 2
         + cs_score[ws->state];
#else
  int score = 0;
  int match_bonus;
  int i;

  for (i = 0; i < SPL_MATCH_MAX; i++) {
    GestureList this_spell_match = spell_match[i] >> 8;
    GestureList this_spell_match_mask = spell_match_mask[i] >> 8;
    match_bonus = 8;
    while (this_spell_match_mask != 0) {
      if ((ws->gl & this_spell_match_mask) == this_spell_match) {
        score += match_bonus;
        break;
      }
      this_spell_match >>= 8;
      this_spell_match_mask >>= 8;
      match_bonus--;
    }
  }
  return -(ws->damage_inflicted * 100) + score;
#endif
}

void apply_monsters(WizardState *ws, int num_ws) {
  int their_monster_max_hp = 0;
  int i;

  for (i = 1; i < num_ws; i++) {
    their_monster_max_hp = ws[i].monster_max_hp[0] +
                           ws[i].monster_max_hp[1] +
                           ws[i].monster_max_hp[2];
  }

  /* assume we get 1/(opponents - 1) of the damage (rounded up) */
  int opponents = num_ws - 1;
  if (!(ws[0].state & (CS_SHIELD | CS_PROT_FROM_EVIL))) {
    ws[0].damage_inflicted = (their_monster_max_hp + opponents - 1) / opponents;
  }

  int our_monster_max_hp, our_monster_per_opponent, our_monster_remainder;

  /* assume we distribute the monster HP evenly between opponents */
  our_monster_max_hp = ws[0].monster_max_hp[0] +
                       ws[0].monster_max_hp[1] +
                       ws[0].monster_max_hp[2];
  our_monster_per_opponent = our_monster_max_hp / opponents;
  our_monster_remainder = our_monster_max_hp % opponents;

  for (i = 1;
       (i < num_ws && (our_monster_per_opponent || our_monster_remainder));
       i++) {
    if (!(ws[i].state & (CS_SHIELD | CS_PROT_FROM_EVIL))) {
      ws[i].damage_inflicted += our_monster_per_opponent;
      if (our_monster_remainder) {
        ws[i].damage_inflicted++;
        our_monster_remainder--;
      }
    }
  }
}

void get_paralysis_value_mask(int hand, GesturePair g,
                              GestureList *paralysis_value,
                              GestureList *paralysis_mask) {
  if (hand == 1) {
    *paralysis_value = g & 0xF0;
    *paralysis_mask = 0xF0;
    switch (*paralysis_value) {
      case GST_CLAP << 8: *paralysis_value = GST_FINGER << 8; break;
      case GST_SNAP << 8: *paralysis_value = GST_POINT << 8;  break;
      case GST_WAVE << 8: *paralysis_value = GST_PALM << 8;   break;
    }
  } else if (hand == 2) {
    *paralysis_value = g & 0x0F;
    *paralysis_mask = 0x0F;
    switch (*paralysis_value) {
      case GST_CLAP: *paralysis_value = GST_FINGER; break;
      case GST_SNAP: *paralysis_value = GST_POINT;  break;
      case GST_WAVE: *paralysis_value = GST_PALM;   break;
    }
  } else {
    printf("get_paralysis_value_mask: hand is %d\n", hand);
    exit(1);
  }
}

static char *figure_paralysis_table(Move *ml1, int num_ml1,
                                    Move *ml2, int num_ml2) {
  char *paralysis_table = calloc(num_ml1 * num_ml2, sizeof(char));
  int i1, i2;
  char *p;

  /* are we paralysing them? */
  if (ml1[0].paralysis_hand) {

    /* go through each of our moves */
    for (p = paralysis_table, i1 = 0; i1 < num_ml1; i1++) {
      GestureList paralysis_value, paralysis_mask;

      /* go through each of their moves */
      for (i2 = 0; i2 < num_ml2; i2++, p++) {
        GestureList last_gp = ml2[i2].gl >> 8;
        while (((last_gp & 0xFF) == make_gp(GST_FOG, GST_FOG)) ||
               ((last_gp & 0xFF) == make_gp(GST_ANTISPELL, GST_ANTISPELL))) {
          last_gp >>= 8;
        }
        get_paralysis_value_mask(ml1[i1].paralysis_hand,
                                 last_gp & 0xFF,
                                 &paralysis_value,
                                 &paralysis_mask);

        if ((ml2[i2].gl & paralysis_mask) != paralysis_value) {
          *p = 1;
        }
      }
    }
  }

  /* are they paralysing us? */
  if (ml2[0].paralysis_hand) {

    /* go through each of our moves */
    for (p = paralysis_table, i1 = 0; i1 < num_ml1; i1++) {
      GestureList paralysis_value, paralysis_mask;

      /* go through each of their moves */
      for (i2 = 0; i2 < num_ml2; i2++, p++) {
        GestureList last_gp = ml1[i1].gl >> 8;
        while (((last_gp & 0xFF) == make_gp(GST_FOG, GST_FOG)) ||
               ((last_gp & 0xFF) == make_gp(GST_ANTISPELL, GST_ANTISPELL))) {
          last_gp >>= 8;
        }
        get_paralysis_value_mask(ml2[i2].paralysis_hand,
                                 last_gp & 0xFF,
                                 &paralysis_value,
                                 &paralysis_mask);

        if ((ml1[i1].gl & paralysis_mask) != paralysis_value) {
          *p = 1;
        }
      }
    }
  }

  return paralysis_table;
}

/* We want to maximise (with our move) the minimum score resulting
 * from any of their possible moves.
 * If we consider each of our moves in an outer loop, and each of
 * their moves in an inner loop, then if we encounter a score less
 * than our best maximum we can skip the rest of the inner loop.
 *
 * Pseudo-code:
 *
 * set max_score to -inf, bestmove to unknown
 * for each of our possible moves (ourmove):
 *   set min_score to +inf
 *   for each of their moves (theirmove):
 *     calculate score for ourmove + theirmove
 *     if score < max_score then go to our next move
 *     if score < min_score set min_score = score
 *   if min_score > max_score set max_score = min_score, bestmove = ourmove
 *
 * Best to order our moves from best (for us) to worst.
 * Best to order their moves from best (for them) to worst.
 *
 * Calculate score is a recursive call to find_move(depth - 1)
 * unless depth == 0 in which case it's a call to static_score.
 */

#define USEHASHTABLE
#ifdef USEHASHTABLE
struct wizard_moves {
  unsigned int lock;
  int          num_wizards;
  Move         *movelist[2];
  int          num_moves[2];
  int          in_use;
};

//#define HASHTABLE_BITS 20
#define HASHTABLE_BITS 16
#define HASHTABLE_SIZE (1 << HASHTABLE_BITS)
#define HASHTABLE_MASK (HASHTABLE_SIZE - 1)

struct wizard_moves hash_table[HASHTABLE_SIZE];

void init_hash(void) {
  memset(hash_table, 0, sizeof(hash_table));
}
#endif

void hash_put(WizardState *ws, int num_wizards,
              Move *movelist1, Move *movelist2,
              int num_moves_1, int num_moves_2) {
#ifdef USEHASHTABLE
  unsigned long long h = hash((void *)ws, sizeof(WizardState) * num_wizards, 0);
  struct wizard_moves *hash_table_entry = &hash_table[h & HASHTABLE_MASK];

//printf("storing %u\n", h);

  /* are the movelists already in there?  if so, decrement the in-use
   * counter
   */
  if (hash_table_entry->movelist[0] == movelist1) {
//  printf("already exists\n");
    hash_table_entry->in_use--;
    return;
  }

  /* is this entry already taken and in use further up the tree? */
  if (hash_table_entry->in_use) {
    /* unfortunately then, we can't keep it */
    free(movelist1);
    free(movelist2);
    return;
  }

//printf("lock is %u\n", h >> HASHTABLE_BITS);
  free(hash_table_entry->movelist[0]);
  free(hash_table_entry->movelist[1]);
  hash_table_entry->lock = h >> HASHTABLE_BITS;
  hash_table_entry->num_wizards = num_wizards;
  hash_table_entry->movelist[0] = movelist1;
  hash_table_entry->movelist[1] = movelist2;
  hash_table_entry->num_moves[0] = num_moves_1;
  hash_table_entry->num_moves[1] = num_moves_2;
#endif
}

int count_hit = 0;
int count_miss = 0;

int hash_get(WizardState *ws, int num_wizards,
             Move **movelist1, Move **movelist2,
             int *num_moves_1, int *num_moves_2) {
#ifdef USEHASHTABLE
  unsigned long long h = hash((void *)ws, sizeof(WizardState) * num_wizards, 0);
  struct wizard_moves *hash_table_entry = &hash_table[h & HASHTABLE_MASK];

//printf("looking up %u\n", h);
//printf("lock is %u vs %u\n", hash_table_entry->lock, h >> 20);
  if ((hash_table_entry->lock == ((h >> HASHTABLE_BITS) & UINT_MAX)) &&
      (hash_table_entry->num_wizards == num_wizards) &&
      hash_table_entry->movelist[0]) {
//  printf("hit\n");
    count_hit++;
    *movelist1 = hash_table_entry->movelist[0];
    *movelist2 = hash_table_entry->movelist[1];
    *num_moves_1 = hash_table_entry->num_moves[0];
    *num_moves_2 = hash_table_entry->num_moves[1];
    hash_table_entry->in_use++;
    return 1;
  }
//printf("miss\n");
  count_miss++;
  return 0;
#else
  return 0;
#endif
}

/* move the "to" move to before the "from" move */
static void shuffle_movelist(Move *movelist, int *scores, int from, int to) {
  Move temp_move;
  int temp_score;

  temp_move = movelist[from];
  memmove(&movelist[to + 1], &movelist[to], (from - to) * sizeof(Move));
  movelist[to] = temp_move;

  temp_score = scores[from];
  memmove(&scores[to + 1], &scores[to], (from - to) * sizeof(int));
  scores[to] = temp_score;
}

int find_move(WizardState *ws, int num_ws, int their_id,
              int depth, int max_score) {
  Move *movelist[2];
  int num_moves[2];
  int *scores[2];
  int our_move, their_move;
  WizardState *ws_new = malloc(num_ws * sizeof(WizardState));
  char *paralysis_table = NULL;

  if (!hash_get(ws, num_ws,
                &movelist[0], &movelist[1],
                &num_moves[0], &num_moves[1])) {
    get_movelist(ws, num_ws, 0,
                 ws[their_id].paralysed, their_id,
                 &movelist[0], &num_moves[0]);
    get_movelist(ws, num_ws, their_id,
                 ws[0].paralysed, 0,
                 &movelist[1], &num_moves[1]);
    sort_movelist(movelist[0], num_moves[0]);
    sort_movelist(movelist[1], num_moves[1]);
  }

  assert(num_moves[0]);
  assert(num_moves[1]);

  /* scores[0] is for the min score of all their moves indexed by our
   * move
   */
  scores[0] = calloc(num_moves[0], sizeof(int));

  /* scores[1] is for the individual scores of their moves combined
   * with our move indexed by their move
   */
  scores[1] = calloc(num_moves[1], sizeof(int));

  if (movelist[0][0].paralysis_hand || movelist[1][0].paralysis_hand) {
    paralysis_table = figure_paralysis_table(movelist[0], num_moves[0],
                                             movelist[1], num_moves[1]);
  }

  for (our_move = 0; our_move < num_moves[0]; our_move++) {
    int min_score = INT_MAX;
    int found_move = 0;
    for (their_move = 0; their_move < num_moves[1]; their_move++) {
      int score;

      if (paralysis_table &&
          paralysis_table[our_move * num_moves[1] + their_move]) {
        continue;
      }

      memcpy(ws_new, ws, num_ws * sizeof(WizardState));
      ws_new[0].gl = movelist[0][our_move].gl;
      ws_new[their_id].gl = movelist[1][their_move].gl;

      if (ws_new[0].paralysed == -1) {
        ws_new[0].paralysed = movelist[0][our_move].paralysis_hand;
      }
      if (ws_new[their_id].paralysed == -1) {
        ws_new[their_id].paralysed = movelist[1][their_move].paralysis_hand;
      }

      if (movelist[0][our_move].sp != 0 ||
          movelist[1][their_move].sp != 0) {
        apply_spells(ws_new, num_ws, their_id,
                     movelist[0][our_move].sp,
                     movelist[1][their_move].sp);
      }
      apply_monsters(ws_new, num_ws);
      check_deadness(ws_new, num_ws);

      /* us dead is very bad
       * in more moves (smaller depth) is better
       */
      if (ws_new[0].state & CS_DEAD) {
        score = -10000000 - depth;

      /* them dead is very good
       * in less moves (bigger depth) is better
       */
      } else if (ws_new[their_id].state & CS_DEAD) {
        score = 10000000 + depth;

      /* neither dead -- are we looking further ahead? */
      } else if (depth == 0) {

        int i;

        /* no; this is a leaf -- guess at how good this position is */
        score = static_score(&ws_new[0]);
        for (i = 1; i < num_ws; i++) {
          score -= static_score(&ws_new[i]);
        }

      /* yes; recurse to find the score */
      } else {
        score = find_move(ws_new, num_ws, their_id, depth - 1, max_score);
      }

      scores[1][their_move] = score;

      int move_to = their_move;

      /* if this is not the first move of theirs that we tried, and
       * this move scored better for them than the previous move, move
       * it closer to the beginning
       */
      if ((their_move != 0) && (score < scores[1][their_move - 1])) {

        /* need to move at least 1 position backwards */
        move_to--;

        /* look backwards further until we're at the beginning */
        while ((move_to > 0) && (score < scores[1][move_to - 1])) {
          move_to--;
        }

        /* now move the Move from their_move to move_to and shuffle
         * everything inbetween up one
         */
        shuffle_movelist(movelist[1], scores[1], their_move, move_to);
      }

      /* cutoff */
      if (score <= max_score) {
        found_move = 0;
        min_score = score;

        break;
      }

      if (score < min_score) {
        found_move = 1;
        min_score = score;
      }
    }

    if (!found_move) {
      scores[0][our_move] = INT_MIN;
    }

    if (found_move) {
      scores[0][our_move] = min_score;

      int move_to = our_move;

      /* if this is not the first move of ours that we tried, and this
       * move scored better than the previous move, move it closer to
       * the beginning
       */
      if ((our_move != 0) && (min_score > scores[0][our_move - 1])) {

        /* need to move at least 1 position backwards */
        move_to--;

        /* look backwards further until we're at the beginning */
        while ((move_to > 0) && (min_score > scores[0][move_to - 1])) {
          move_to--;
        }

        /* now move the Move from our_move to move_to and shuffle
         * everything inbetween up one
         */
        shuffle_movelist(movelist[0], scores[0], our_move, move_to);
      }

      if (min_score > max_score) {
        max_score = min_score;
      }

    }

  }

  free(ws_new);
  if (paralysis_table) {
    free(paralysis_table);
  }
  free(scores[0]);
  free(scores[1]);

  /* hash_put() will either free movelist[] or save it in the hash
   * table
   */
  hash_put(ws, num_ws,
           movelist[0], movelist[1],
           num_moves[0], num_moves[1]);

  return max_score;
}

#ifdef SMP
void worker(int fd) {
  int         our_move;
  int         their_move;
  int         num_ws;
  WizardState *ws;
  int         wizard_num;
  int         depth;
  int         max_score;

  while (1) {
    read_bytes(fd, &our_move, sizeof(int));
    read_bytes(fd, &their_move, sizeof(int));
    read_bytes(fd, &num_ws, sizeof(int));
    ws = malloc(num_ws * sizeof(WizardState));
    read_bytes(fd, ws, num_ws * sizeof(WizardState));
    read_bytes(fd, &wizard_num, sizeof(int));
    read_bytes(fd, &depth, sizeof(int));
    read_bytes(fd, &max_score, sizeof(int));
    int score = find_move(ws, num_ws, wizard_num, depth, max_score);
    write_bytes(fd, &our_move, sizeof(int));
    write_bytes(fd, &their_move, sizeof(int));
    write_bytes(fd, &score, sizeof(int));
    free(ws);
  }
}

void send_job_to_worker(int fd, int our_move, int their_move,
                        int num_ws, WizardState *ws,
                        int their_id, int depth, int max_score) {
  write_bytes(fd, &our_move, sizeof(int));
  write_bytes(fd, &their_move, sizeof(int));
  write_bytes(fd, &num_ws, sizeof(int));
  write_bytes(fd, ws, num_ws * sizeof(WizardState));
  write_bytes(fd, &their_id, sizeof(int));
  write_bytes(fd, &depth, sizeof(int));
  write_bytes(fd, &max_score, sizeof(int));
}

fd_set get_fd_set(int *workers_in_use) {
  fd_set fds;
  int    i;

  FD_ZERO(&fds);

  for (i = 0; i < NUM_WORKERS; i++) {
    if (workers_in_use[i]) {
      FD_SET(workers[i], &fds);
    }
  }

  return fds;
}

void find_move_to_compute(
  int num_our_moves, int num_their_moves,
  int *scores, int *scores_computed,
  int *min_scores, int *min_scores_computed,
  int *our_move_out, int *their_move_out
) {
  int i, j;

  /* complete top row first */
  if (min_scores_computed[0] == 0) {

    /* look left to right along the top row for any entry that we
     * don't have an answer for and haven't sent to a worker
     */
    for (i = 0; i < num_their_moves; i++) {
      if (scores_computed[i] == 0) {
        scores_computed[i] = -1;
        *our_move_out = 0;
        *their_move_out = i;
        return;
      }
    }
  }

  /* now do column-by-column... */
  for (i = 0; i < num_their_moves; i++) {

    /* ...top-to-bottom (skipping the top row) */
    for (j = 1; j < num_our_moves; j++) {

      /* have we already completed this row? */
      if (min_scores_computed[j] != 0) {
        continue;
      }

      /* have we not asked for an answer for this entry yet? */
      int move_id = j * num_their_moves + i;
      if (scores_computed[move_id] == 0) {
        scores_computed[move_id] = -1;
        *our_move_out = j;
        *their_move_out = i;
        return;
      }
    }
  }

  /* nothing left? */
  *our_move_out = -1;
}

int deal_with_worker(
  int worker_num, int num_our_moves, int num_their_moves,
  int *scores, int *scores_computed,
  int *min_scores, int *min_scores_computed,
  int *max_score
) {
  int fd = workers[worker_num];
  int our_move;
  int their_move;
  int score;

  read_bytes(fd, &our_move, sizeof(int));
  read_bytes(fd, &their_move, sizeof(int));
  read_bytes(fd, &score, sizeof(int));

  assert(our_move >= 0);
  assert(our_move < num_our_moves);
  assert(their_move >= 0);
  assert(their_move < num_their_moves);
  int move_id = our_move * num_their_moves + their_move;

  /* place this new entry into scores/scores_computed */
  scores[move_id] = score;
  scores_computed[move_id] = 1;

  /* check for cutoff -- if so, mark the row as irrelevant */
  if (scores[move_id] <= *max_score) {
    min_scores_computed[our_move] = -1;

  /* otherwise check to see if we've completed the row */
  } else {
    int i, row_complete = 1, min_score = INT_MAX;
    for (i = 0, move_id = our_move * num_their_moves;
         i < num_their_moves;
         i++, move_id++) {

      /* entry either not started, or started and not completed? */
      if (scores_computed[move_id] <= 0) {
        row_complete = 0;
        break;

      /* entry completed */
      } else if (scores_computed[move_id] == 1) {
        if (scores[move_id] < min_score) {
          min_score = scores[move_id];
        }

      /* otherwise the entry is irrelevant (eg. paralysis) */
      }
    }

    /* if the row is complete, place the entry into
     * min_scores/min_scores_computed
     */
    if (row_complete) {
      min_scores[our_move] = min_score;
      min_scores_computed[our_move] = 1;
      if (min_score > *max_score) {
        *max_score = min_score;
      }
    }
  }

  /* now check to see if we've completed the table */
  int i;
  for (i = 0; i < num_our_moves; i++) {
    if (min_scores_computed[i] == 0) {
      return 0;
    }
  }

  return 1;
}

int deal_with_workers(
  fd_set fds, int *num_workers_free, int *workers_in_use,
  int num_our_moves, int num_their_moves,
  int *scores, int *scores_computed,
  int *min_scores, int *min_scores_computed,
  int *max_score
) {
  int i;
  int table_complete;

  for (i = 0; i < NUM_WORKERS; i++) {
    if (workers_in_use[i] && FD_ISSET(workers[i], &fds)) {
      table_complete =
          deal_with_worker(i, num_our_moves, num_their_moves,
                           scores, scores_computed,
                           min_scores, min_scores_computed,
                           max_score);
      (*num_workers_free)++;
      workers_in_use[i] = 0;
      if (table_complete) {
        return 1;
      }
    }
  }

  return 0;
}

int get_free_worker(int *workers_in_use) {
  int i;

  for (i = 0; i < NUM_WORKERS; i++) {
    if (!workers_in_use[i]) {
      return i;
    }
  }
  printf("get_free_worker() called when no free workers\n");
  exit(1);
}

void find_move_toplevel_smp(WizardState *ws, int num_ws,
                            int depth, int stop_at, Move *best_move_out) {
  int max_score = INT_MIN;
  Move *movelist[2];
  int num_moves[2];
  int *scores, *scores_computed;
  int *min_scores, *min_scores_computed;
  int our_move, their_move;
  WizardState *ws_new = malloc(num_ws * sizeof(WizardState));
  char *paralysis_table = NULL;
  int i;

  if (hash_get(ws, num_ws,
               &movelist[0], &movelist[1],
               &num_moves[0], &num_moves[1])) {
    printf("move list was cached\n");
  } else {
    printf("move list not cached -- generating\n");
    int us_paralysed = ws[0].paralysed;
    int them_paralysed = 0;               /* 0 = no, otherwise wizard number */

    for (i = 1; i < num_ws; i++) {
      if (ws[i].paralysed) {
        them_paralysed = i;
        break;
      }
    }
    get_movelist(ws, num_ws, 0,
                 ws[them_paralysed].paralysed, them_paralysed,
                 &movelist[0], &num_moves[0]);
    get_movelists(ws, num_ws,
                  us_paralysed,
                  &movelist[1], &num_moves[1]);
    sort_movelist(movelist[0], num_moves[0]);
    sort_movelist(movelist[1], num_moves[1]);
  }

  for (i = 0; i < num_ws; i++) {
    show_ws(&ws[i]);
  }
  printf("Depth: %d\n", depth);
  show_movelist(movelist[0], num_moves[0]);
  show_movelist(movelist[1], num_moves[1]);
  printf("   ");
  for (their_move = 0; their_move < num_moves[1]; their_move++) {
    printf("%3d", their_move);
  }
  printf("\n");

  assert(num_moves[0]);
  assert(num_moves[1]);

  /* scores[our_move * num_moves[1] + their_move] is for the score
   * from that result
   * scores_computed[] is 0 for scores[] not valid, -1 for scores[]
   * currently being computed, 1 for scores[] valid, and 2 for
   * irrelevant (eg. paralysis)
   * min_scores[our_move] is for the minimum of:
   * scores[our_move * num_moves[1] ..
   *        (our_move + 1) * num_moves[1] - 1]
   * min_scores_computed[] is 0 for min_scores[] not valid, -1 for
   * irrelevant (eg. cutoff), or 1 for min_scores[] valid
   */
  scores = calloc(num_moves[0] * num_moves[1], sizeof(int));
  scores_computed = calloc(num_moves[0] * num_moves[1], sizeof(int));
  min_scores = calloc(num_moves[0], sizeof(int));
  min_scores_computed = calloc(num_moves[0], sizeof(int));

  paralysis_table = figure_paralysis_table(movelist[0], num_moves[0],
                                           movelist[1], num_moves[1]);

  int move_id;
  int table_is_paralysed = 1;
  for (our_move = 0, move_id = 0; our_move < num_moves[0]; our_move++) {
    int row_is_paralysed = 1;
    for (their_move = 0; their_move < num_moves[1]; their_move++, move_id++) {
      if (paralysis_table[move_id]) {
        scores_computed[move_id] = 2;
      } else {
        row_is_paralysed = 0;
      }
    }
    if (row_is_paralysed) {
      min_scores_computed[our_move] = -1;
    } else {
      table_is_paralysed = 0;
    }
  }
  free(paralysis_table);
  if (table_is_paralysed) {
    memcpy(best_move_out, &movelist[0][0], sizeof(Move));
    free(ws_new);
    free(scores);
    return;
  }

  /* number of workers that haven't been sent work */
  int num_workers_free = NUM_WORKERS;

  /* which workers have (1) or haven't (0) been sent work */
  int *workers_in_use = calloc(NUM_WORKERS, sizeof(int));

  /* are the any queries left to send off? */
  int moves_left = 1;

  /* have we completed the table? */
  int table_complete = 0;

  while (!table_complete && time(NULL) < stop_at) {
    /* give all our workers some work */
    while (num_workers_free > 0 && moves_left && time(NULL) < stop_at) {
      int our_move, their_move;

      find_move_to_compute(num_moves[0], num_moves[1],
                           scores, scores_computed,
                           min_scores, min_scores_computed,
                           &our_move, &their_move);
      if (our_move == -1) {
        moves_left = 0;
        break;
      }
      int worker_num = get_free_worker(workers_in_use);
      assert(worker_num >= 0);
      assert(worker_num < NUM_WORKERS);
      int their_id = movelist[1][their_move].who;

      memcpy(ws_new, ws, num_ws * sizeof(WizardState));
      ws_new[0].gl = movelist[0][our_move].gl;
      ws_new[their_id].gl = movelist[1][their_move].gl;

      if (ws_new[0].paralysed == -1) {
        ws_new[0].paralysed = movelist[0][our_move].paralysis_hand;
      }
      if (ws_new[their_id].paralysed == -1) {
        ws_new[their_id].paralysed = movelist[1][their_move].paralysis_hand;
      }

      if (movelist[0][our_move].sp != 0 ||
          movelist[1][their_move].sp != 0) {
        apply_spells(ws_new, num_ws, their_id,
                     movelist[0][our_move].sp,
                     movelist[1][their_move].sp);
      }
      apply_monsters(ws_new, num_ws);
      check_deadness(ws_new, num_ws);

      int score;

      /* us dead is very bad
       * in more moves (smaller depth) is better
       */
      if (ws_new[0].state & CS_DEAD) {
        score = -10000000 - depth;

      /* them dead is very good
       * in less moves (bigger depth) is better
       */
      } else if (ws_new[their_id].state & CS_DEAD) {
        score = 10000000 + depth;

      /* neither dead -- are we looking further ahead? */
      } else if (depth == 0) {

        /* no; this is a leaf -- guess at how good this position is */
        score = static_score(&ws_new[0]);
        for (i = 1; i < num_ws; i++) {
          score -= static_score(&ws_new[i]);
        }

      /* yes; pass to a worker */
      } else {
        send_job_to_worker(workers[worker_num],
                           our_move, their_move,
                           num_ws, ws_new,
                           their_id, depth - 1, max_score);
        num_workers_free--;
        workers_in_use[worker_num] = 1;

        continue;
      }

      int move_id = our_move * num_moves[1] + their_move;
      scores[move_id] = score;
      scores_computed[move_id] = -1;
    }

    /* wait for at least one worker to complete */
    fd_set fds = get_fd_set(workers_in_use);
    int num_ready = select(max_fd, &fds, NULL, NULL, NULL);

    if (num_ready < 0) {
      if (errno != EINTR) {
        perror("select");
        exit(1);
      }
    }

    /* at least one worker has a result for us -- fill in the table
     * and see if we're done
     */
    table_complete =
        deal_with_workers(fds, &num_workers_free, workers_in_use,
                          num_moves[0], num_moves[1],
                          scores, scores_computed,
                          min_scores, min_scores_computed,
                          &max_score);
  }

#if 0
      int move_to = their_move;

      /* if this is not the first move of theirs that we tried, and
       * this move scored better for them than the previous move, move
       * it closer to the beginning
       */
      if ((their_move != 0) && (score < scores[1][their_move - 1])) {

        /* need to move at least 1 position backwards */
        move_to--;

        /* look backwards further until we're at the beginning */
        while ((move_to > 0) && (score < scores[1][move_to - 1])) {
          move_to--;
        }

        /* now move the Move from their_move to move_to and shuffle
         * everything inbetween up one
         */
        shuffle_movelist(movelist[1], scores[1], their_move, move_to);
      }

      /* cutoff */
      if (score <= max_score) {
        found_move = 0;
        min_score = score;

        break;
      }

      if (score < min_score) {
        found_move = 1;
        min_score = score;
      }
    }

    if (!found_move) {
      scores[0][our_move] = INT_MIN;
    }

    if (found_move) {
      scores[0][our_move] = min_score;

      int move_to = our_move;

      /* if this is not the first move of ours that we tried, and this
       * move scored better than the previuos move, move it closer to
       * the beginning
       */
      if ((our_move != 0) && (min_score > scores[0][our_move - 1])) {

        /* need to move at least 1 position backwards */
        move_to--;

        /* look backwards further until we're at the beginning */
        while ((move_to > 0) && (min_score > scores[0][move_to - 1])) {
          move_to--;
        }

        /* now move the Move from our_move to move_to and shuffle
         * everything inbetween up one
         */
        shuffle_movelist(movelist[0], scores[0], our_move, move_to);
      }

      if (min_score > max_score) {
        max_score = min_score;
        printf("  new best move");
      }

    }

    printf("  min was %d\n", min_score);
  }

  show_move(&movelist[0][0]);
  printf("\n");
  memcpy(best_move_out, &movelist[0][0], sizeof(Move));
#endif

  free(ws_new);
  free(scores);

  /* hash_put() will either free movelist[] or save it in the hash
   * table
   */
  hash_put(ws, num_ws,
           movelist[0], movelist[1],
           num_moves[0], num_moves[1]);
}
#endif

void find_move_toplevel(WizardState *ws, int num_ws,
                        int depth, int stop_at, Move *best_move_out) {
#ifdef SMP
  if (depth != 0) {
    return find_move_toplevel_smp(ws, num_ws, depth, stop_at, best_move_out);
  }
#endif

  int max_score = INT_MIN;
  Move *movelist[2];
  int num_moves[2];
  int *scores[2];
  int our_move, their_move;
  WizardState *ws_new = malloc(num_ws * sizeof(WizardState));
  char *paralysis_table = NULL;
  int out_of_time = 0;
  int i;

  if (hash_get(ws, num_ws,
               &movelist[0], &movelist[1],
               &num_moves[0], &num_moves[1])) {
    printf("move list was cached\n");
  } else {
    printf("move list not cached -- generating\n");
    int us_paralysed = ws[0].paralysed;
    int them_paralysed = 0;               /* 0 = no, otherwise wizard number */

    for (i = 1; i < num_ws; i++) {
      if (ws[i].paralysed) {
        them_paralysed = i;
        break;
      }
    }
    get_movelist(ws, num_ws, 0,
                 ws[them_paralysed].paralysed, them_paralysed,
                 &movelist[0], &num_moves[0]);
    get_movelists(ws, num_ws,
                  us_paralysed,
                  &movelist[1], &num_moves[1]);
    sort_movelist(movelist[0], num_moves[0]);
    sort_movelist(movelist[1], num_moves[1]);
  }

  for (i = 0; i < num_ws; i++) {
    show_ws(&ws[i]);
  }
  printf("Depth: %d\n", depth);
  show_movelist(movelist[0], num_moves[0]);
  show_movelist(movelist[1], num_moves[1]);
  printf("   ");
  for (their_move = 0; their_move < num_moves[1]; their_move++) {
    printf("%3d", their_move);
  }
  printf("\n");

  assert(num_moves[0]);
  assert(num_moves[1]);

  /* scores[0] is for the min score of all their moves indexed by our
   * move
   */
  scores[0] = calloc(num_moves[0], sizeof(int));

  /* scores[1] is for the individual scores of their moves combined
   * with our move indexed by their move
   */
  scores[1] = calloc(num_moves[1], sizeof(int));

  paralysis_table = figure_paralysis_table(movelist[0], num_moves[0],
                                           movelist[1], num_moves[1]);

  for (our_move = 0; our_move < num_moves[0] && !out_of_time; our_move++) {
    printf("%3d", our_move);
    int min_score = INT_MAX;
    int found_move = 0;
    for (their_move = 0; their_move < num_moves[1]; their_move++) {
      int their_id = movelist[1][their_move].who;
      int score;

      if (time(NULL) >= stop_at) {
        printf("  Out of time");
        found_move = 0;
        out_of_time = 1;
        break;
      }

      if (paralysis_table[our_move * num_moves[1] + their_move]) {
        printf(" --");
        continue;
      }

      memcpy(ws_new, ws, num_ws * sizeof(WizardState));
      ws_new[0].gl = movelist[0][our_move].gl;
      ws_new[their_id].gl = movelist[1][their_move].gl;

      if (ws_new[0].paralysed == -1) {
        ws_new[0].paralysed = movelist[0][our_move].paralysis_hand;
      }
      if (ws_new[their_id].paralysed == -1) {
        ws_new[their_id].paralysed = movelist[1][their_move].paralysis_hand;
      }

      if (movelist[0][our_move].sp != 0 ||
          movelist[1][their_move].sp != 0) {
        apply_spells(ws_new, num_ws, their_id,
                     movelist[0][our_move].sp,
                     movelist[1][their_move].sp);
      }
      apply_monsters(ws_new, num_ws);
      check_deadness(ws_new, num_ws);

      /* us dead is very bad
       * in more moves (smaller depth) is better
       */
      if (ws_new[0].state & CS_DEAD) {
        score = -10000000 - depth;

      /* them dead is very good
       * in less moves (bigger depth) is better
       */
      } else if (ws_new[their_id].state & CS_DEAD) {
        score = 10000000 + depth;

      /* neither dead -- are we looking further ahead? */
      } else if (depth == 0) {

        /* no; this is a leaf -- guess at how good this position is */
        score = static_score(&ws_new[0]);
        for (i = 1; i < num_ws; i++) {
          score -= static_score(&ws_new[i]);
        }

      /* yes; recurse to find the score */
      } else {
        score = find_move(ws_new, num_ws, their_id, depth - 1, max_score);
      }

      printf("%3d", score);
      fflush(stdout);

      scores[1][their_move] = score;

      int move_to = their_move;

      /* if this is not the first move of theirs that we tried, and
       * this move scored better for them than the previous move, move
       * it closer to the beginning
       */
      if ((their_move != 0) && (score < scores[1][their_move - 1])) {

        /* need to move at least 1 position backwards */
        move_to--;

        /* look backwards further until we're at the beginning */
        while ((move_to > 0) && (score < scores[1][move_to - 1])) {
          move_to--;
        }

        /* now move the Move from their_move to move_to and shuffle
         * everything inbetween up one
         */
        shuffle_movelist(movelist[1], scores[1], their_move, move_to);
      }

      /* cutoff */
      if (score <= max_score) {
        found_move = 0;
        min_score = score;

        break;
      }

      if (score < min_score) {
        found_move = 1;
        min_score = score;
      }
    }

    if (!found_move) {
      scores[0][our_move] = INT_MIN;
    }

    if (found_move) {
      scores[0][our_move] = min_score;

      int move_to = our_move;

      /* if this is not the first move of ours that we tried, and this
       * move scored better than the previuos move, move it closer to
       * the beginning
       */
      if ((our_move != 0) && (min_score > scores[0][our_move - 1])) {

        /* need to move at least 1 position backwards */
        move_to--;

        /* look backwards further until we're at the beginning */
        while ((move_to > 0) && (min_score > scores[0][move_to - 1])) {
          move_to--;
        }

        /* now move the Move from our_move to move_to and shuffle
         * everything inbetween up one
         */
        shuffle_movelist(movelist[0], scores[0], our_move, move_to);
      }

      if (min_score > max_score) {
        max_score = min_score;
        printf("  new best move");
      }

    }

    printf("  min was %d\n", min_score);
  }

  show_move(&movelist[0][0]);
  printf("\n");
  memcpy(best_move_out, &movelist[0][0], sizeof(Move));

  free(ws_new);
  free(paralysis_table);
  free(scores[0]);
  free(scores[1]);

  /* hash_put() will either free movelist[] or save it in the hash
   * table
   */
  hash_put(ws, num_ws,
           movelist[0], movelist[1],
           num_moves[0], num_moves[1]);
}

/* take a list of WizardStates and remove any duplicates within
 * ws[1..num_ws - 1] (ie. don't consider ws[0] (ourself) when removing
 * duplicates)
 */
WizardState *squeeze_ws(WizardState *ws, int num_ws, int *new_num_ws,
                        int *squeezed_num_map) {
  int i, j;

  assert(num_ws > 2);

  WizardState *new_ws = malloc(2 * sizeof(WizardState));
  memcpy(new_ws, ws, 2 * sizeof(WizardState));

  *new_num_ws = 2;

  /* go through each of the remaining entries in ws[]: 2..num_ws - 1 */
  for (i = 2; i < num_ws; i++) {
    int found = 0;

    /* check if we have already stored this in new_ws */
    for (j = 1; j < *new_num_ws; j++) {
      if (!memcmp(&ws[i], &new_ws[j], sizeof(WizardState))) {
        found = 1;
        break;
      }
    }

    /* not already stored -- copy into new_ws */
    if (!found) {
      new_ws = realloc(new_ws, ((*new_num_ws) + 1) * sizeof(WizardState));
      memcpy(&new_ws[*new_num_ws], &ws[i], sizeof(WizardState));
      squeezed_num_map[*new_num_ws] = i;
      (*new_num_ws)++;
    }
  }

  return new_ws;
}

void fix_sp_nums(Move *m, int *squeezed_num_map) {
  unsigned char spell_who;

  /* fix the LH spell target */
  spell_who = (m->sp & 0x0F000000) >> 24;
  m->sp = (m->sp & 0xF0FFFFFF) | (squeezed_num_map[spell_who] << 24);

  /* fix the RH spell target */
  spell_who = (m->sp & 0x00000F00) >> 8;
  m->sp = (m->sp & 0xFFFFF0FF) | (squeezed_num_map[spell_who] << 8);
}

void find_move_id(WizardState *ws, int num_ws,
                 int timeout, Move *best_move_out) {
  int d = 0;
  time_t stop_at;
  WizardState *ws_squeezed;
  int         num_ws_squeezed;
  int         *squeezed_num_map = malloc(num_ws * sizeof(int));

  stop_at = time(NULL) + timeout;

  /* create ws_squeezed which doesn't have any duplicate wizard states
   * in it
   */
  squeezed_num_map[0] = 0;
  squeezed_num_map[1] = 1;
  if (num_ws <= 2) {
    ws_squeezed = ws;
    num_ws_squeezed = num_ws;
  } else {
    ws_squeezed = squeeze_ws(ws, num_ws, &num_ws_squeezed, squeezed_num_map);
  }

  do {
    printf("find_move_toplevel(%d)\n", d);
    find_move_toplevel(ws_squeezed, num_ws_squeezed, d,
                       (d == 0 ? INT_MAX : stop_at),
                       best_move_out);
    d++;
  } while (time(NULL) < stop_at);

  /* unsqueeze the spell target numbers in best_move_out */
  fix_sp_nums(best_move_out, squeezed_num_map);

  if (num_ws > 2) {
    free(ws_squeezed);
  }
  free(squeezed_num_map);
}

void cleanup_moves(void) {
  int i;

  for (i = 0; i < HASHTABLE_SIZE; i++) {
    if (hash_table[i].movelist[0]) {
      free(hash_table[i].movelist[0]);
      free(hash_table[i].movelist[1]);
    }
  }
}

#ifdef SMP
void start_workers(void) {
  int i;

  for (i = 0; i < NUM_WORKERS; i++) {
    int sv[2];

    if (socketpair(AF_UNIX, SOCK_STREAM, 0, sv) < 0) {
      perror("socketpair");
    }

    if ((sv[0] + 1) > max_fd) {
      max_fd = sv[0] + 1;
    }
    if ((sv[1] + 1) > max_fd) {
      max_fd = sv[1] + 1;
    }

    int pid = fork();

    /* fork error */
    if (pid < 0) {
      perror("fork");
      exit(1);
    }

    /* child */
    if (pid == 0) {
      close(sv[1]);
      worker(sv[0]);
      exit(0);
    }

    /* parent */
    close(sv[0]);
    workers[i] = sv[1];
  }
}
#endif

void init_moves(void) {
#ifdef USEHASHTABLE
  init_hash();
  atexit(cleanup_moves);
#endif
#ifdef SMP
  start_workers();
#endif
}
