/*
 *  Many parts taken from GTetrinet
 *  Copyright (C) 1999, 2000, 2001, 2002, 2003  Ka-shu Wong (kswong@zip.com.au)
 *  Copyright (C) 2003 Chris Yeoh (cyeoh@samba.org)
 *  Copyright (C) 2003 Jeremy Kerr (jeremy@redfishsoftware.com.au)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <netinet/in.h>

#include <sys/types.h>
#include <sys/socket.h>


#include <stdlib.h>

#define _GNU_SOURCE
#include <stdio.h>

#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <time.h>
#include <sys/time.h>
#include <netdb.h>

#include "client.h"
#include "comm_client.h"
#include "tetris.h"
#include "log.h"

#define PORT 31457

static void process_response(char *buf, int len);

static void sendfield(int reset, struct game *newgame);

static void send_player_is_dead(void);

struct timeval last_move_time;
static char *CurrentChannel = NULL;
static char *DesiredChannel = "#tetribot0";
static char *DesiredChannelTemplate = "#tetribot%i";
static int CurrentChannelNum=0;
int AmInGame = 0;

int main(int argc, char **argv)
{
	int sd;
	struct sockaddr_in addr;
	struct in_addr server_address;
	/* todo: max message size */
	unsigned char buf[4096];
	int buf_index = 0;
	struct timeval tv, curtime; /* , movetime; */
	fd_set fdset;
	struct hostent *tetrinet_server_host;

	int serverSocket, clientSocket;
	struct sockaddr_in clientAcceptAddress;
	int tmp_int;
	signed char option;
	int do_fork = 1;
	int reset_timer;

	/* We seed the random number generator before we fork 
		 and so all the clients should get the same blocks but
		 still be randomly selected */
	srand(time(NULL));

	while ( (option=getopt(argc, argv, "dh")) > 0)
	{
		switch (option)
		{
		case 'h':
			printf("Usage: tetribot [-hd] [<tetrinet_server_name>]\n");
			exit(0);

		case 'd':
			do_fork = 0;
			break;

		default:
		}

		option = getopt(argc, argv, "dh");
	}

	/* Players with null name don't exist */
	for (tmp_int=0; tmp_int<6; tmp_int++)
	{
		strcpy(game.players[tmp_int].name, "");
	}

	serverSocket = createServerSocket(9467);
	tmp_int = sizeof(clientAcceptAddress);

	while (1)
	{
		printf("Waiting for client connection\n");

		if ( (clientSocket = accept(serverSocket, 
																(struct sockaddr *)&clientAcceptAddress, 
																&tmp_int)) < 0)
		{
			perror("client accept failed\n");
			exit(1);
		}
		else
		{
			if (do_fork)
			{
				int pid;
				printf("Got client connection\n");
				pid = fork();
				if (pid==-1)
				{
					perror("fork failed");
					exit(1);
				}
				else if (pid==0)
				{
					/* child */
					close(serverSocket);
					tbt_setpid();
					break;
				}
				else
				{
					/* parent */
					close(clientSocket);
				}
			}
			else
			{
				/* We don't fork but still close the
					 server socket so other teribots can start up */
				close(serverSocket);
				break;
			}
		}
	}

	processClientMessage(clientSocket, &SendField, &reset_timer);

	if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("socket");
		return EXIT_FAILURE;
	}

	server_address.s_addr = 0;
	if (argc>optind)
	{
		tetrinet_server_host = gethostbyname(argv[optind]);
		if (tetrinet_server_host)
		{
			memcpy(&server_address, tetrinet_server_host->h_addr_list[0],
						 tetrinet_server_host->h_length);
		}
		else
		{
			printf("Tetrinet server %s does not exist\n", argv[optind]);
			exit(1);
		}
	}


	memset ((char *)&addr, 0, sizeof(addr));
        addr.sin_family = AF_INET;
        addr.sin_port   = htons(PORT);
        memcpy(&(addr.sin_addr), &server_address, sizeof(server_address));

	if (connect(sd, (struct sockaddr *)&addr, sizeof(addr))) {
		perror("connect");
		close(sd);
		return EXIT_FAILURE;
	}

 	client_init(sd, getUsername());

	while (1) {
		int len, i, ret, array_i;
		unsigned char *bufptr;
		

		if (game.gameon && ClientState==CC_CLIENT_READY && AmInGame)
		{
			struct TC_GameData *gd;
			int num_players = 0, i;

			/* Set players state */
			tetris_set_all_players_alive();

			/* Setup block frequencies */
			tetris_set_block_frequencies(game.piecefrequencyarray);
			tetris_set_block_special_frequencies(game.specialfrequencyarray);
			tetris_set_num_specials(game.specialcapacity);
			tetris_clear_all_specials();

			/* Create a pending block */
			tetris_make_pending_block();


			for (i=0; i<6; i++)
			{
				if (strcmp(game.players[i].name, "")!=0) 
				{
					num_players++;
					tbt_log(TBT_LOG_NOISY, "Found player %s\n", game.players[i].name);
				}
			}
			tbt_log(TBT_LOG_MEDIUM, "Found %i players\n", num_players);

			gd = malloc(sizeof(struct TC_GameData) 
									+ num_players*sizeof(struct TC_PlayerData));
			gd->MaxNumSpecials = game.specialcapacity;
			gd->NumberOfPlayers = num_players;
			gd->PlayerNumber = game.playernum-1;

			tbt_log(TBT_LOG_MEDIUM, "Own player number is %i\n", gd->PlayerNumber);

			for (i=0, array_i=0; array_i<num_players; i++)
			{
				if (strcmp(game.players[i].name, "")!=0)
				{
					gd->Players[array_i].PlayerNumber = i;
					asprintf(&gd->Players[array_i].PlayerName, "%s", 
									 game.players[i].name);
					asprintf(&gd->Players[array_i].TeamName, "%s", 
									 game.players[i].team);
					printf("--- Added player %s (%i), index %i\n", 
								 gd->Players[array_i].PlayerName,
								 array_i, 
								 gd->Players[array_i].PlayerNumber);
					array_i++;
				}
			}

			startClient(clientSocket, gd);
			ClientState = CC_CLIENT_PLAYING;
			free(gd);

		}

		gettimeofday(&curtime, NULL);

		if (game.gameon && !tetris_get_player_state(game.playernum-1))
		{
			/* Copy field */
			memcpy(&GameState, &game, sizeof(struct game));

			/* Calculate timeout 
				 We want to drop the block no sooner than
				 one second since the last drop */
			curtime.tv_sec -= 1;
/* 			printf("curr time: %i, %i\nlast_time: %i %i\n", */
/* 						 (int)curtime.tv_sec, (int)curtime.tv_usec, */
/* 						 (int)last_move_time.tv_sec, (int)last_move_time.tv_usec); */
			if (curtime.tv_sec>last_move_time.tv_sec
					|| (curtime.tv_sec==last_move_time.tv_sec
							&& curtime.tv_usec>=last_move_time.tv_usec))
			{
				/* More than one second has elapsed */
				tv.tv_sec = 0;
				tv.tv_usec = 0;
			}
			else
			{
				tv.tv_sec = 0;
				if (last_move_time.tv_sec>curtime.tv_sec)
				{
					tv.tv_usec = 1000000 - curtime.tv_usec + last_move_time.tv_usec;
				}
				else
				{
					tv.tv_usec = last_move_time.tv_usec - curtime.tv_usec;
				}
			}
		}
		else
		{
			/* Change channel if we can to what we want */
			if ( (CurrentChannel==NULL || strcmp(CurrentChannel, DesiredChannel))
					 && game.playernum!=0)
			{
				/* Try to change channel to #tetribot */
				char *message;

				tbt_log(TBT_LOG_NOISY, 
								"Current channel is '%s' and Desired Channel is '%s'\n",
								CurrentChannel, DesiredChannel);
				if (CurrentChannel && !strcmp(CurrentChannel, DesiredChannel))
				{
					tbt_log(TBT_LOG_HIGH, "Channel names same and trying to change?\n");
				}
				asprintf(&message, "pline %i /join %s", game.playernum,
								 DesiredChannel);
 				client_send(message, strlen(message));
				printf("foo\n");
				free(message);

			}
			
			/* Set timeout to be about 10 seconds */
			tv.tv_sec = 10;
			tv.tv_usec = 0;
		}

		FD_ZERO(&fdset);
		FD_SET(sd, &fdset);
		FD_SET(clientSocket, &fdset);

		
#ifdef DEBUG
		printf("timeout in %d sec %d\n", (int)tv.tv_sec, (int)tv.tv_usec);
#endif

		ret = select(sd > clientSocket ? sd+1 : clientSocket+1, 
								 &fdset, NULL, NULL, 
 								 !tetris_get_player_state(game.playernum-1) ? &tv : NULL);


		if (ret==0 && !game.gameon)
		{
			/* Timeout outside of game */
			/* Find out if there are any other players in the game */
			int i, other_player = 0;
			for (i=0; i<6; i++)
			{
				if ( (i!=game.playernum-1) && strcmp(game.players[i].name, "") )
				{
					other_player = 1;
					break;
				}
			}

			if (other_player)
			{
				/* Lets try to start the game */
				char *msg;
				printf("Auto starting game\n");
				asprintf(&msg, "startgame 1 %i", game.playernum);
				client_send(msg, strlen(msg));
				free(msg);
			}
			else
			{
				/* No other players */
				/* If we're not in the first tetribot channel then see if there
					 is space in any of the other channels */
				if (CurrentChannelNum!=0)
				{
					CurrentChannelNum=0;
					asprintf(&DesiredChannel, DesiredChannelTemplate, CurrentChannelNum);
				}
			}
		}

		if (ret==0 && game.gameon && !tetris_get_player_state(game.playernum-1)
			&& AmInGame)
		{
			/* Timeout - so should move block */
			if (get_current_block()<0)
			{
				if (tetris_makeblock(tetris_pending_block(), 
														 tetris_pending_block_orient(),
														 GameState.players[GameState.playernum-1].field)
						== 1)
				{
					/* player has died */
					send_player_is_dead();
				}
				else
				{
					tetris_make_pending_block();
				}
			}
			else
			{
				if (tetris_blockdown(GameState.players[GameState.playernum-1].field)==-1)
				{
					if (tetris_solidify(GameState.players[GameState.playernum-1].field)
							== 0)
					{
						int lines_cleared, i, j;
						char specials[200];
						char *spc;
						lines_cleared = tetris_removelines(
							specials, 
							GameState.players[GameState.playernum-1].field);
						tetrinet_addsbtofield(
							lines_cleared, 
							GameState.players[GameState.playernum-1].field);
						i=0;
						while (specials[i]!=0)
						{
							for (j=0; j<lines_cleared; j++)
								tetris_add_special(specials[i]);
							i++;
						}
						/* send extra lines to other people */
						switch (lines_cleared)
						{
						case 4:
							spc = "cs4";
							break;
						case 3:
							spc = "cs2";
							break;
						case 2:
							spc = "cs1";
							break;
						default:
							spc = NULL;
						}
						if (spc) send_special(0, spc, GameState.playernum);
					}
					else
					{
						/* player is dead */
						send_player_is_dead();
					}

					SendField = 1;
				}
			}

			/* Reset last time move */
			gettimeofday(&last_move_time, NULL);
			
		}

		if (FD_ISSET(sd, &fdset)) 
		{
			int lastFF_pos = 0;
			len = client_receive(buf + buf_index, 1024 - buf_index);
			len += buf_index;
			bufptr = buf;
			for (i = 0; i < len; i++) {
				if (*(buf + i) == 0xff) {
					lastFF_pos = i;
					process_response(bufptr, i - (bufptr - buf));
					bufptr = buf + i + 1;
				}
			}
			if (lastFF_pos<len-1)
			{
				/* We have stuff left over that we haven't yet interpreted */
				memmove(buf, buf+lastFF_pos+1, len-lastFF_pos);
				buf_index = len-lastFF_pos;
				tbt_log(TBT_LOG_LOW, "Saved %i bytes, (%s)\n", buf_index, buf);
			}
			else
			{
				buf_index = 0;
			}
		}

		if (FD_ISSET(clientSocket, &fdset))
		{
			if (processClientMessage(clientSocket, &SendField, &reset_timer))
			{
				if (reset_timer)
					gettimeofday(&last_move_time, NULL);
			}
			else
			{
				printf("Tetrinet AI has disconnected\n");

				/* Player dies if the AI disconnects */
				send_player_is_dead();

				printf("Exiting\n");
				exit(0);
			}
			if (tetris_get_player_state(GameState.playernum-1)) AmInGame = 0;
		}

		if (SendField)
		{
			sendfield(0, &GameState);
			SendField = 0;
		}

	}
	
	/* parse args */

	return 0;
}

static inline int safe_strcmp(const char *buf, const char *str,
                              unsigned int bufl)
{
	unsigned int strl = strlen(str);
	return !strncmp(buf, str, strl < bufl ? strl : bufl);
}

static char blocks[] = "012345acnrsbgqo";

static void print_field(struct player *p)
{
	int x, y;

	printf("field %s --\n", p->name);
	for (y = 0; y < FIELD_HEIGHT; y++) {
		for (x = 0; x < FIELD_WIDTH; x++) {
			int i = p->field[y][x];
			printf("%c", i < strlen(blocks) ? blocks[i] : 'x');
		}
		printf("\n");
	}
}

/**
 * modified from gtetrinet/tetrinet.c
 */
void sendfield(int reset, struct game *newgame)
{
  int x, y, i, d = 0; /* d is the number of differences */
  char buf[1024], *p;
  char *field = (char *)(newgame->players[game.playernum-1].field);

  char diff_buf[15][(FIELD_WIDTH + 1) * FIELD_HEIGHT * 2] = {{0}};

  int row_count[15] = {1,1,1,1,1, 1,1,1,1,1, 1,1,1,1,1};

#ifdef DEBUG
  printf("sending field %i:\n", game.playernum);
	for (y = 0; y < FIELD_HEIGHT; y++) {
		for (x = 0; x < FIELD_WIDTH; x++) {
			int i = *(field + y * FIELD_WIDTH + x);
			printf("%c", i < strlen(blocks) ? blocks[i] : 'x');
		}
		printf("\n");
	}
#endif

  sprintf(buf, "f %d ", game.playernum);

  if(!reset) {
    /* Find out which blocks changed, how, and how many */
    for (y = 0; y < FIELD_HEIGHT; y ++) {
      for (x = 0; x < FIELD_WIDTH; x ++) {

	const char block = *(field + y * FIELD_WIDTH + x);

        if ((block < 0) || (block >= 15))
        {
          tbt_log(TBT_LOG_HIGH, 
									"sendfield shouldn't reach here, block=%d\n", block);
          continue;
        }
        
	if (block != game.players[game.playernum-1].field[y][x]) {
	  diff_buf[(int)(block)][row_count[(int)block]++] = x + '3';
	  diff_buf[(int)(block)][row_count[(int)block]++] = y + '3';
	  d += 2;
	}
      }
    }
    if (d == 0) return; /* no differences */

    for (i = 0; i < 15; ++i)
      if (row_count[i] > 1)
        ++d; /* add an extra value for the '!'+i at the start */
  }
  
  if (reset || d >= (FIELD_HEIGHT*FIELD_WIDTH)) {
    /* sending entire field is more efficient */
    p = buf + 4;
    for (y = 0; y < FIELD_HEIGHT; y ++)
      for (x = 0; x < FIELD_WIDTH; x ++)
				*(p++) = blocks[(int)*(field + y*FIELD_WIDTH + x)];
    *p = 0;
  }
  else { /* so now we need to create the buffer strings */
    for(i = 0; i < 15; ++i) {
      if(row_count[i] > 1) {
	diff_buf[i][0] = '!' + i;
	strcat(buf, diff_buf[i]);
      }
    }
  }

  client_send(buf, strlen(buf));
  /* update the one in our memory */
  memcpy(game.players[game.playernum-1].field,
  	 newgame->players[game.playernum-1].field,
         sizeof(char) * FIELD_WIDTH * FIELD_HEIGHT);
}

char translateblock(char c)
{
    int i;
    for (i = 0; blocks[i]; i ++)
        if (c == blocks[i]) return i;
    return 0;
}

static void process_response(char *buf, int len)
{
	char *tmp, *op, *delim = " ";

	*(buf + len) = 0;

/* 	printf("len: %d, string: %s\n", len, buf); */
	op = strtok(buf, delim);

	tbt_log(TBT_LOG_LOW, "op: %s\n", op);

	if (!op)
		return;

	if (safe_strcmp(op, "playernum", len)) {
		char *return_message;
		tmp = strtok(NULL, delim);
		assert(tmp);
		game.playernum = atoi(tmp);
		printf("player num: %d\n", game.playernum);

		/* It appears tetrinetx requires that we send the team number
			 back straight away */
		asprintf(&return_message, "team %i ", game.playernum);
		client_send(return_message, 7);
		free(return_message);

		
	} else if (safe_strcmp(buf, "winlist", len)) {
		printf("winlist\n");

	} else if (safe_strcmp(buf, "playerjoin", len)) {
		int pn;
		
		tmp = strtok(NULL, delim);
		assert(tmp);

		pn = atoi(tmp);
		assert(pn > 0 && pn <= 6);

		tmp = strtok(NULL, delim);
		assert(tmp);

		strncpy(game.players[pn-1].name, tmp, NAMELEN - 1);

	} else if (safe_strcmp(buf, "team", len)) {
		int pn;
		
		tmp = strtok(NULL, delim);
		assert(tmp);

		pn = atoi(tmp);
		printf("pn %i, gpn %i\n", pn, game.playernum);
//		assert(pn > 0 && pn != game.playernum && pn < 7);

		tmp = strtok(NULL, delim);

		if (tmp) {
			strncpy(game.players[pn-1].team, tmp, NAMELEN - 1);
			tbt_log(TBT_LOG_MEDIUM, "Team for pn %i is %s\n", pn, tmp);
		} else {
			game.players[pn-1].team[0] = '\0';
			tbt_log(TBT_LOG_MEDIUM, "No team for pn %i\n", pn);
		}


	} else if (safe_strcmp(buf, "newgame", len)) {
		int i;

		assert(tmp = strtok(NULL, delim));
		/* stack height ? */

		assert(tmp = strtok(NULL, delim));
		game.initiallevel = atoi(tmp);

		assert(tmp = strtok(NULL, delim));
		game.linesperlevel = atoi(tmp);
		
		assert(tmp = strtok(NULL, delim));
		game.levelinc = atoi(tmp);
		
		assert(tmp = strtok(NULL, delim));
		game.speciallines = atoi(tmp);

		assert(tmp = strtok(NULL, delim));
		game.specialcount = atoi(tmp);

		assert(tmp = strtok(NULL, delim));
		game.specialcapacity = atoi(tmp);

		assert(tmp = strtok(NULL, delim));
		while (*tmp) {
			i = *tmp - '1';
			if (i >= 0 && i < 7)
				game.piecefrequencyarray[i]++;
			tmp++;
		}
		for (i=1; i<7; i++)
		{
			game.piecefrequencyarray[i] += game.piecefrequencyarray[i-1];
		}
		if (game.piecefrequencyarray[6]<100)
			game.piecefrequencyarray[6] = 100;


		assert(tmp = strtok(NULL, delim));
		while (*tmp) {
			i = *tmp - '1';
			if (i >= 0 && i < 9)
				game.specialfrequencyarray[i]++;
			tmp++;
		}
		for (i=1; i<9; i++)
		{
			game.specialfrequencyarray[i] += game.specialfrequencyarray[i-1];
		}
		if (game.specialfrequencyarray[8]<100)
			game.specialfrequencyarray[8] = 100;


		assert(tmp = strtok(NULL, delim));
		game.levelavg = atoi(tmp);

		assert(tmp = strtok(NULL, delim));
		game.oldmode = atoi(tmp);

		for (i = 0; i < 6; i++) {
			memset(&game.players[i].field, 0,
			       FIELD_HEIGHT * FIELD_WIDTH);
			game.players[i].level = game.initiallevel;
		}

		sendfield(1, &game);

		game.gameon = 1;
		gettimeofday(&last_move_time, NULL);
		AmInGame = 1;

	} else if (safe_strcmp(buf, "f", len)) {
		int pn;
		
		assert(tmp = strtok(NULL, delim));
		pn = atoi(tmp) - 1;

		assert(tmp = strtok(NULL, delim));

		if (*tmp >= '0') {
			char *p = (char *)game.players[pn].field;
			for (; *tmp; tmp ++, p ++)
                    		*p = translateblock(*tmp);
		} else {
			char block = 0;
			int x, y;
			while (*tmp) {
				if (*tmp < '0' && *tmp >= '!') {
					block = *tmp - '!';
				} else {
					x = *tmp - '3';
					y = *(++tmp) - '3';
					game.players[pn].field[y][x] = block;
				}
				tmp++;
			}
		}
#ifdef DEBUG
		print_field(&game.players[pn]);
#endif
	} else if (safe_strcmp(buf, "sb", len))
	{
		/* incoming special block */
		int toplayer, fromplayer;
		char *specialtype;
		int player_died = 0;

		tmp = strtok(NULL, delim);
		assert(tmp);
		toplayer = atoi(tmp);

		tmp = strtok(NULL, delim);
		assert(tmp);
		specialtype = strdup(tmp);
		
		tmp = strtok(NULL, delim);
		assert(tmp);
		fromplayer = atoi(tmp);

#ifdef DEBUG
		printf("Received special %s from player %i (to %i)\n",
					 specialtype, fromplayer, toplayer);
#endif
		if ( (toplayer==GameState.playernum || toplayer==0) && AmInGame)
		{
			if (toplayer==0) toplayer = GameState.playernum;
			SendField = 1;
			if (strcmp(specialtype, "a")==0)
			{
				player_died = tetris_addlines(1, GameState.players[toplayer-1].field, 
																			1);
			}
			else if (strcmp(specialtype, "cs1")==0)
			{
				player_died = tetris_addlines(1, GameState.players[toplayer-1].field,
																			2);
			}
			else if (strcmp(specialtype, "cs2")==0)
			{
				player_died = tetris_addlines(2, GameState.players[toplayer-1].field,
																			2);
			}
			else if (strcmp(specialtype, "cs4")==0)
			{
				player_died = tetris_addlines(4, GameState.players[toplayer-1].field,
																			2);
			}
			else if (strcmp(specialtype, "c")==0)
			{
				tetris_clearline(GameState.players[toplayer-1].field);
			}
			else if (strcmp(specialtype, "n")==0)
			{
				memset(GameState.players[toplayer-1].field, 0, 
							 FIELD_WIDTH*FIELD_HEIGHT);
			}
			else if (strcmp(specialtype, "r")==0)
			{
				tetris_clear_random(GameState.players[toplayer-1].field);
			}
			else if (strcmp(specialtype, "b")==0)
			{
				tetris_clear_specials(GameState.players[toplayer-1].field);
			}
			else if (strcmp(specialtype, "g")==0)
			{
				tetris_block_gravity(GameState.players[toplayer-1].field);
			}
			else if (strcmp(specialtype, "q")==0)
			{
				tetris_block_quake(GameState.players[toplayer-1].field);
			}
			else if (strcmp(specialtype, "o")==0)
			{
				tetris_block_bomb(GameState.players[toplayer-1].field);
			}
			else if (strcmp(specialtype, "s")==0)
			{
				tetris_switch_fields(GameState.players[toplayer-1].field,
														 GameState.players[fromplayer-1].field);
			}
			else
			{
				tbt_log(TBT_LOG_HIGH, "Unknown special %s\n", specialtype);
			}
			
			if (player_died)
				send_player_is_dead();
			else if (AmInGame)
				tetris_removelines(NULL, 
													 GameState.players[GameState.playernum-1].field);
		}


	} else if (safe_strcmp(buf, "playerlost", len))
	{
		int player;

		tmp = strtok(NULL, delim);
		assert(tmp);
		player = atoi(tmp);

		printf("Player %i has died\n", player);
		tetris_set_player_state(player-1, 1);
		tbt_log(TBT_LOG_MEDIUM, "Player state is %i %i %i %i %i %i\n",
						tetris_get_player_state(0),
						tetris_get_player_state(1),
						tetris_get_player_state(2),
						tetris_get_player_state(3),
						tetris_get_player_state(4),
						tetris_get_player_state(5));


	} else if (safe_strcmp(buf, "endgame", len))
	{
		printf("Game has ended\n");
		game.gameon = 0;
		if (ClientState==CC_CLIENT_PLAYING) ClientState = 0;
		AmInGame = 0;

	} else if (safe_strcmp(buf, "playerwon", len))
	{
		int player;
						 
		tmp = strtok(NULL, delim);
		assert(tmp);
		player = atoi(tmp);

		printf("Player %i won\n", player);

		tetris_set_player_state(player-1, 2);

	} else if (safe_strcmp(buf, "pline", len))
	{
		int from_player;
		tmp = strtok(NULL, delim);
		assert(tmp);
		from_player = atoi(tmp);

		tmp = strtok(NULL, "");
		assert(tmp);

		tbt_log(TBT_LOG_LOW, "Message from player %i\n  %s\n", from_player, tmp);

		if (from_player==0)
		{
			/* Message from server */
			char *channel_change = "Joined existing Channel - ";
			if (strstr(tmp, channel_change)!=NULL)
			{
				tmp += strlen(channel_change)+2;
				/* Remove last character as its not part of the channel name */
				tmp[strlen(tmp)-1] = 0;
				tbt_log(TBT_LOG_LOW, "------ Just joined channel %s\n", tmp);
				if (CurrentChannel) free(CurrentChannel);
				asprintf(&CurrentChannel, "%s", tmp);
			}
			else if (strstr(tmp, "That channel is ") && strstr(tmp, "FULL"))
			{
				CurrentChannelNum++;
				asprintf(&DesiredChannel, DesiredChannelTemplate,
								 CurrentChannelNum);
			}
		}

	} else if (safe_strcmp(buf, "playerleave", len))
	{
		int player_leaving;

		tmp = strtok(NULL, "");
		player_leaving = atoi(tmp);

		tbt_log(TBT_LOG_MEDIUM, "Player %i has left\n", player_leaving);

		strcpy(game.players[player_leaving-1].name, "");

	} else if (safe_strcmp(buf, "ingame", len))
	{
		tbt_log(TBT_LOG_MEDIUM, "Game already in progress\n");
		game.gameon = 1;

	} else {
		tmp = strtok(NULL, "");
		tbt_log(TBT_LOG_MEDIUM, "unknown opcode %s\n", op);
		tbt_log(TBT_LOG_MEDIUM, "Message was - len: %d, string: %s\n", len, tmp);

	}
}

void send_player_is_dead()
{
	char *msg;
	printf("** Player is dead\n");
	tetris_set_player_state(GameState.playernum-1, 1);
	tetris_create_dead_field(
		GameState.players[GameState.playernum-1].field);
	sendfield(0, &GameState);
	asprintf(&msg, "playerlost %i", GameState.playernum);
	client_send(msg, strlen(msg));
	free(msg);
	SendField = 1;
	AmInGame = 0;
}
