/*
   Copyright (C) 2003 Christopher Yeoh <cyeoh@samba.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA. 
*/
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <netdb.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <string.h>

#include "tclient.h"
#include "tcommon.h"



static int ClientSocket = -1;
#define ERROR_STRING_LENGTH 100
char *tc_error_string = NULL;


int connectToServer(const char *Hostname, int Port, const char *Username,
										char *TeamName, char *Channel)
{
	struct sockaddr_in remote_address;
	struct hostent *remote_host;
	int tmp_int;

	if (tc_error_string==NULL)
	{
		if ( (tc_error_string = malloc(ERROR_STRING_LENGTH+1)) == NULL )
		{
			fprintf(stderr, "Out of memory\n");
			abort();
		}
		tc_error_string[ERROR_STRING_LENGTH] = 0;
	}

	if ( (ClientSocket = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	{
		snprintf(tc_error_string, ERROR_STRING_LENGTH, 
						 "Failed to create socket");
		return -1;
	}

	remote_host = gethostbyname(Hostname);
	if (!remote_host)
	{
		snprintf(tc_error_string, ERROR_STRING_LENGTH,
						 "Host %s does not exist", Hostname);
		return -1;
	}

	bzero(&remote_address, sizeof(remote_address));
	remote_address.sin_family = AF_INET;
	remote_address.sin_port = htons(Port);
	memcpy(&remote_address.sin_addr.s_addr, remote_host->h_addr_list[0],
				 remote_host->h_length);
	
	if (connect(ClientSocket, (struct sockaddr *)&remote_address, 
							sizeof(remote_address))<0)
	{
		snprintf(tc_error_string, ERROR_STRING_LENGTH, 
						 "Could not connect to server %s", Hostname);
		return -1;
	}

	tmp_int = htonl(_TC_CONNECT);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
	tmp_int = htonl(TCOMMON_PROTOCOL_VERSION);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
	writeString(ClientSocket, Username);
	if (writeString(ClientSocket, TeamName)==-1)
	{
		printf("write failed\n");
	}
	writeString(ClientSocket, Channel);

	readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
	printf("Connection code error %i\n", tmp_int);
	if (tmp_int==0)
	{
		return 0;
	}
	else
	{
		char *error;
		error = readString(ClientSocket);
		if (error) 
		{
			strncpy(tc_error_string, error, ERROR_STRING_LENGTH);
		}
		return -1;
	}
}

int Start(struct TC_GameData **Data)
{
	int i, tmp_int;

	tmp_int = htonl(_TC_START);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
	
	readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));

	if (tmp_int==0)
	{
		int num_players;

		/* Get number of players */
		readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
		num_players = ntohl(tmp_int);

		*Data = (struct TC_GameData *)malloc(sizeof(struct TC_GameData)
						 + num_players*sizeof(struct TC_PlayerData));
		if (!Data)
		{
			fprintf(stderr, "Out of memory");
			abort();
		}

		/* Get player number */
		readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
		(*Data)->PlayerNumber = ntohl(tmp_int);

		/* Get number of specials */
		(*Data)->NumberOfPlayers = num_players;
		readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
		(*Data)->MaxNumSpecials = ntohl(tmp_int);

		/* Get player data */
		for (i=0; i < (*Data)->NumberOfPlayers; i++)
		{
			/* Get player number */
			readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
			(*Data)->Players[i].PlayerNumber = ntohl(tmp_int);

			/* Get player name */
			(*Data)->Players[i].PlayerName = readString(ClientSocket);
			 
			/* Get Team name */
			(*Data)->Players[i].TeamName = readString(ClientSocket);

			if ( (*Data)->Players[i].PlayerName==NULL
					 || (*Data)->Players[i].TeamName==NULL )
			{
				fprintf(stderr, "Out of memory");
				abort();
			}
		}

		return 0;
		
	}
	else
	{
		char *error;
		error = readString(ClientSocket);
		strncpy(tc_error_string, error, ERROR_STRING_LENGTH);
		return -1;
	}
}

int MoveBlock(int MoveCommand)
{
	int tmp_int;

	tmp_int = htonl(_TC_COMMAND_MOVE);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));

	tmp_int = htonl(MoveCommand);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
	
	readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));

	tmp_int = ntohl(tmp_int);
	
	return tmp_int;
}

int UseSpecial(int PlayerNum)
{
	int tmp_int;

	tmp_int = htonl(_TC_COMMAND_USE_SPECIAL);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));

	tmp_int = htonl(PlayerNum);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
	
	readNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));

	tmp_int = ntohl(tmp_int);
	
	return tmp_int;
}

void DiscardSpecial()
{
	int tmp_int;

	tmp_int = htonl(_TC_COMMAND_DISCARD_SPECIAL);
	writeNumBytes(ClientSocket, &tmp_int, sizeof(tmp_int));
}

char *GetField(int PlayerNum)
{
	int tmp_int;
	char *field;

	writeInt32(ClientSocket, _TC_COMMAND_GET_FIELD);
	writeInt32(ClientSocket, PlayerNum);

	tmp_int = readInt32(ClientSocket);

	if (tmp_int==0)
	{
		field = malloc(TC_FIELD_SIZE);
		if (field==NULL) return NULL;
		/* A bit misleading to say the player is dead if we're
			 out of memory. oh well */

		readNumBytes(ClientSocket, field, TC_FIELD_SIZE);
		return field;
	}
	else
	{
		return NULL;
	}
}

void GetSpecials(char *Specials)
{
	int num_specials;

	writeInt32(ClientSocket, _TC_COMMAND_GET_SPECIALS);

	num_specials = readInt32(ClientSocket);
	if (num_specials)
	{
		readNumBytes(ClientSocket, Specials, num_specials);
	}
	Specials[num_specials] = 0;
}

int GetCurrentBlock(int *BlockType, int *XPosition, int *YPosition)
{
	int status;

	writeInt32(ClientSocket, _TC_COMMAND_GET_CURRENT_BLOCK);

 	status = readInt32(ClientSocket);
	*BlockType = readInt32(ClientSocket);
	*XPosition = readInt32(ClientSocket);
	*YPosition = readInt32(ClientSocket);

	return status;
}

void GetNextBlock(int *BlockType)
{
	writeInt32(ClientSocket, _TC_COMMAND_GET_NEXT_BLOCK);
	*BlockType = readInt32(ClientSocket);
}

int GetMessage(int *FromPlayer, char **Message)
{
	int have_message;

	writeInt32(ClientSocket, _TC_COMMAND_GET_MESSAGE);

	have_message = readInt32(ClientSocket);
	if (!have_message)
	{
		return 1;
	}
	else
	{
		*FromPlayer = readInt32(ClientSocket);
		*Message = readString(ClientSocket);
		return 0;
	}
}

void SendMessage(int ToPlayer, char *Message)
{
	writeInt32(ClientSocket, _TC_COMMAND_SEND_MESSAGE);
	writeInt32(ClientSocket, ToPlayer);
	writeString(ClientSocket, Message);
}

int GetNumPlayers(struct TC_GameData *Data)
{
	return Data->NumberOfPlayers;
}

struct TC_PlayerData *GetPlayer(struct TC_GameData *Data, int Player)
{
	if (Player >= Data->NumberOfPlayers)
	{
		return NULL;
	}
	else
	{
		return &Data->Players[Player];
	}
}

int GetMaxSpecials(struct TC_GameData *Data)
{
	return Data->MaxNumSpecials;
}

int GetOwnPlayerNumber(struct TC_GameData *Data)
{
	return Data->PlayerNumber;
}

void FreeGameData(struct TC_GameData *Data)
{
	free(Data);
}

int GetPlayerNumber(struct TC_PlayerData *Player)
{
	return Player->PlayerNumber;
}

char *GetPlayerName(struct TC_PlayerData *Player)
{
	return Player->PlayerName;
}

char *GetPlayerTeam(struct TC_PlayerData *Player)
{
	return Player->TeamName;
}
