/*
 *
 * Copyright (C) 2004, Laszlo Peter
 *
 * Nautilus is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * Nautilus is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Authors: Laszlo Peter <laca@sun.com>
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <termios.h>
#include <values.h>
#include <string.h>

#include <tclient.h>
#include "blocks.h"
#include "lacabot.h"
#include "debug.h"

static struct TC_GameData *gdata;

static const char *default_username = "laca"; /* name of the bot */
static char *username = NULL;
static char *servername = NULL;               /* server to connect to */
#ifdef WITH_DEBUG
static int debug = 0;
#endif
static int my_number = 0;                     /* my player number */
static field_stats my_stat;                   /* statistics of my field */
static float my_score = 0.0;                  /* score assigned to my field */
static char *my_specials = NULL;              /* my special blocks */

static int num_players = 0;
static int max_specials;
static float score_of_empty = 0.0;
static int do_delay = 1;

/* Utility functions */

/* return a pseudo-random integer between 0 and i-1 */
static inline int my_rand (int i) {
    return ((int) ((float)i * rand() / (RAND_MAX + 1.0)));
}

static inline float fsqr (const float f) {
    return f*f;
}

static inline int imax (const int a, const int b) {
    return (a > b) ? a  : b;
}

static inline float fmax (const float a, const float b) {
    return (a > b) ? a  : b;
}

static inline int has_specials () {
    return my_specials ? my_specials[0] : 0;
}

/* Calculate various statistics of a field and store them in stat
   Details about field_stats in lacabot.h */
static void count_field_statistics (FIELD f0, field_stats *stat) {
    int i, j, k, row_is_full;
    int hole_count = 0;
    int specials_in_row = 0;
    int ndh = 0;
    FIELD field;

    bzero (stat, sizeof (field_stats));
    /* work on a copy of the field, since we're going to change it */
    bcopy (f0, field, TC_FIELD_WIDTH * TC_FIELD_HEIGHT * sizeof (char));

    /* clear the full lines */
    for (j = TC_FIELD_HEIGHT - 1; j >= stat->full_rows; j--) {
	row_is_full = 1;
	specials_in_row = 0;
	for (i = 0; i < TC_FIELD_WIDTH; i++) {
	    if (!field[j - stat->full_rows][i]) {
		row_is_full = 0;
		break;
	    }
	}
	if (field[j][i] >= TC_SPECIAL_FIRST_SPECIAL) {
	    specials_in_row++;
	}
	if (row_is_full) {
	    stat->full_rows++;
	    stat->specials_earned += specials_in_row;
	}
	if (stat->full_rows) {
	    for (i = 0; i < TC_FIELD_WIDTH; i++) {
		field[j][i] = field[j - stat->full_rows][i];
	    }
	}
	if (row_is_full) {
	    j++;
	}
    }

    stat->avg_hole_weight = 0.0;
    for (j = 0; j < stat->full_rows; j++) {
	for (i = 0; i < TC_FIELD_WIDTH; i++) {
	    field[j][i] = 0;
	}
    }

    for (j = 0; j < TC_FIELD_HEIGHT; j++) {
	for (i = 0; i < TC_FIELD_WIDTH; i++) {
	    if (field[j][i]) {
		stat->row_weight[j]++;
		stat->column_weight[i]++;
		if (!stat->column_height[i]) {
		    stat->column_height[i] = TC_FIELD_HEIGHT - j;
		}
		if (field[j][i] >= TC_SPECIAL_FIRST_SPECIAL) {
		    stat->specials_in_row[j]++;
		}
		if (field[j][i] == TC_SPECIAL_BLOCK_BOMB) {
		    stat->num_block_bombs++;
		}
	    } else {
		if (j && field[j - 1][i]) {
		    field[j][i] = 126; /* means it's blocked */
		    stat->row_blocked_ratio [j] += 1.0;
		    stat->field_blocked_ratio += 1.0;
		    hole_count++;
		    for (k = j - 1; k; k--) {
			stat->avg_hole_weight += 1.0;
			if((!field[k - 1][i]) || (field[k - 1][i] == 126)) {
			    break;
			}
			
		    }
		}
	    }
	}
    }

    if (hole_count) {
	stat->avg_hole_weight /= hole_count;
    }
    j = 0;
    stat->column_height_variance = 0.0;
    stat->field_density = 0.0;
    stat->num_deep_holes = 0;
    for (i = 0; i < TC_FIELD_WIDTH; i++) {
	if (stat->column_height[i]) {
	    stat->column_density [i] = 
		(float)(stat->column_weight[i]) / 
		(float)(stat->column_height[i]);
	} else {
	    stat->column_density [i] = 1.0;
	}
	j += stat->column_height[i];
	stat->field_density += stat->column_weight[i];
	if (stat->column_height[i] > stat->max_column_height) {
	    stat->max_column_height = stat->column_height[i];
	}
    }
    if (j) {
	stat->field_density /= j;
	stat->field_blocked_ratio /= j;
    } else {
	stat->field_density = 1.0;
	stat->field_blocked_ratio = 0.0;
    }
    stat->avg_col_height = (float)j / TC_FIELD_WIDTH;
    if (stat->column_height[0] < stat->column_height[1]) {
	stat->hole_depths[0] = stat->column_height[1] - stat->column_height[0];
    }
    if (stat->column_height[TC_FIELD_WIDTH - 1] < stat->column_height[TC_FIELD_WIDTH - 2]) {
	stat->hole_depths[TC_FIELD_WIDTH - 1] = 
	    stat->column_height[TC_FIELD_WIDTH - 2] 
	    - stat->column_height[TC_FIELD_WIDTH - 1];
    }
    for (i = 0; i < TC_FIELD_WIDTH; i++) {
	stat->column_height_variance += 
	    fsqr (stat->avg_col_height - stat->column_height[i]);
	if (i && (i < TC_FIELD_WIDTH - 1)) {
	    j = imax (
		stat->column_height[i - 1] - stat->column_height[i],
		stat->column_height[i + 1] - stat->column_height[i]);
	    stat->hole_depths[i] = (j > 0) ? j : 0;
	}
	if (stat->hole_depths[i] > 2) {
	    stat->num_deep_holes++;
	}
	if (stat->hole_depths[i] > 1) {
	    ndh++;
	}
	stat->avg_hole_depth += stat->hole_depths[i];
    }
    stat->column_height_variance /= TC_FIELD_WIDTH;
    stat->avg_hole_depth /= TC_FIELD_WIDTH;
    if (stat->num_deep_holes == 0) {
	stat->num_deep_holes = ndh;
    }

    for (j = 0; j < TC_FIELD_HEIGHT; j++) {
	stat->row_density[j] = 
	    (float)(stat->row_weight[j]) / TC_FIELD_WIDTH;
	stat->row_blocked_ratio[j] /= TC_FIELD_WIDTH;
	if (stat->row_blocked_ratio[j] > 0.0) {
	    stat->num_rows_with_blocked_holes++;
	}
	stat->num_specials += stat->specials_in_row[j];
    }

}

inline static float is_low (const float f) {
    if (f <= 0.0) {
	return 1.0;
    }
    if (f >= 1.0) {
	return 0.0;
    }
    return (1.0 - f);
}

inline static float is_high (const float f) {
    return is_low (1.0 - f);
}

/* rule weights define the relative importance of each rule,
   so this basically defines the strategy */
#define NUM_RULES 11
const static float rule_weights [NUM_RULES] = {
    4.5, 1.5, 1.0, 2.0, 4.0, 1.0, 1.0, 2.0, 2.0, 3.0, 1.0
};

/* assign a score to field */
static float get_field_score (FIELD field) {
    field_stats stat;
    float score = 0.0;

    count_field_statistics (field, &stat);

    /* Rule #0: number of full rows should be high */
    score += rule_weights[0] * 
	is_high (fsqr (stat.full_rows / 4));

    /* Rule #1: density should be high  */
    score += rule_weights[1] *
	is_high(fsqr (stat.field_density));

    /* Rule #2: column height variance should be low */
    score += rule_weights[2] *
	is_low (stat.column_height_variance / TC_FIELD_HEIGHT);

    /* Rule #3: average hole weight should be low */
    score += rule_weights[3] *
	is_low (stat.avg_hole_weight / TC_FIELD_HEIGHT);

    if (has_specials ()) {
	/* Rule #4: one deep (>2) hole is good, more than one is bad */
	score += rule_weights[4] *
	    is_low((stat.num_deep_holes - 1.0) / 2);
	score += rule_weights[4] * (stat.num_deep_holes == 1) * 
	    is_low (stat.avg_col_height / TC_FIELD_HEIGHT);
    }

    /* Rule #5: average hole depth should be low */
    score += rule_weights[5] *
	is_low (stat.avg_hole_depth / TC_FIELD_HEIGHT);

    /* Rule #6: average column height should be low */
    score += rule_weights[6] *
	is_low ((stat.avg_col_height - 3.0) / (TC_FIELD_HEIGHT - 3.0));

    /* Rule #7: earning specials is good */
    score += rule_weights[7] *
	(float)stat.specials_earned / TC_FIELD_WIDTH;

    /* Rule #8: the number of rows with blocked holes should be low */
    score += rule_weights[8] *
	is_low ((float)stat.num_rows_with_blocked_holes / TC_FIELD_HEIGHT);

    /* Rule #9: ratio of blocked holes should be low */
    score += rule_weights[9] *
	is_low (stat.field_blocked_ratio);

    /* Rule #10: the highest column should be low */
    score += rule_weights[10] *
	is_low (stat.max_column_height / TC_FIELD_HEIGHT);

#ifdef WITH_DEBUG
    if ((debug > 2) && stat.full_rows) {
	print_field_with_stats (field, &stat);
	fprintf (stderr, "score = %f\n", score);
    }
#endif

    return score;
}

/* try to place block in field at column x and store the resulting
   new field in f_new. The block is currently in row y.
   Return 0 if succeeded 1 if failed */
static int try_place_block (FIELD field, int block, int x, int y,
			    FIELD *f_new) {
    int top [4] = {-1, -1, -1, -1};
    int i, j;
    int new_y, test_y;

/* |       |    FFF      
   |       |     F
   |       |   -010 bottoms
   |    FFF|
   | F  F  |
   |FF FF  |
   | FF FF |
   |EEEEE__|
    5465333   tops
*/

#ifdef WITH_DEBUG
    if (debug > 3) {
	fprintf (stderr, "try_place_block (..., %d, %d, %d, ...)\n",
		 block, x, y);
    }
#endif
		 
    new_y = TC_FIELD_HEIGHT;
    for (i = 0; i < 4; i++) {
	if (x + i < 0) {
	    if (bottoms[block][i] != -1) {
		return 1;
	    } else {
	    continue;
	    }
	}
	if (x + i >= TC_FIELD_WIDTH) {
	    if (bottoms[block][i] != -1) {
		return 1;
	    } else {
	    continue;
	    }
	}
	if (bottoms[block][i] == -1) {
	    continue;
	}
	top[i] = TC_FIELD_HEIGHT;
	for (j = y; j < TC_FIELD_HEIGHT; j++) {
	    if (field[j][x + i]) {
		top[i] = j;
		break;
	    }
	}
	test_y = top[i] - bottoms[block][i] - 1;
	if (test_y < new_y) {
	    new_y = test_y;
	}
    }
    if (y > new_y) {
	return -1;
    }

    bcopy (field, *f_new, TC_FIELD_WIDTH * TC_FIELD_HEIGHT * sizeof (char));

    for (i = 0; i < 4; i++) {
	if (x + i < 0) {
	    if (bottoms[block][i] != -1) {
		return 2;
	    } else {
	    continue;
	    }
	}
	if (x + i >= TC_FIELD_WIDTH) {
	    if (bottoms[block][i] != -1) {
		return 2;
	    } else {
	    continue;
	    }
	}
	for (j = 0; j < 4; j++) {
	    if (blocks[block][j][i]) {
		(*f_new)[j + new_y][x + i] = block;
	    }
	}
    }

    return 0;
}

static const int block_mapping[8] = {
    1, 2, 4, 6, 8, 12, 16, 1
};

#if 0
/* this didn't seem to do any good */
static float guess_best_next_next_score (FIELD field) {
    FIELD f_new;
    float this_score;
    float max_score;
    int i, r, block;
    int rotated_block;

    block = block_mapping[my_rand (7)];
    
    max_score = -MAXFLOAT;
    
    for (i = -2; i < TC_FIELD_WIDTH - 2; i++) {
	if (!try_place_block (field, block, i, 0, &f_new)) {
	    this_score = get_field_score (f_new);
	    if (this_score > max_score) {
		max_score = this_score;
	    }
	}
    }
    rotated_block = block;
    for (r = 1; r <= max_rotations[block]; r++) {
	rotated_block = rotate_cw[rotated_block];
	for (i = -2; i < TC_FIELD_WIDTH - 2; i++) {
	    if (!try_place_block (field, rotated_block, i, 0, &f_new)) {
		this_score = get_field_score (f_new);
		if (this_score > max_score) {
		    max_score = this_score;
		}
	    }
	}
    }

#ifdef WITH_DEBUG
    if (debug > 2) {
	fprintf (stderr, "max_next_score=%f\n", max_score);
    }
#endif
    return max_score * 0.9; 
}
#endif

/* try to place the next block and return the best possible score */
static float get_best_next_score (FIELD field, int block) {
    FIELD f_new;
    float this_score;
    float max_score;
    int i, r;
    int rotated_block;

    max_score = -MAXFLOAT;

    for (i = -2; i < TC_FIELD_WIDTH - 2; i++) {
	if (!try_place_block (field, block, i, 0, &f_new)) {
	    this_score = //fmax (guess_best_next_next_score (f_new),
			       get_field_score (f_new);
	    if (this_score > max_score) {
		max_score = this_score;
	    }
	}
    }
    rotated_block = block;
    for (r = 1; r <= max_rotations[block]; r++) {
	rotated_block = rotate_cw[rotated_block];
	for (i = -2; i < TC_FIELD_WIDTH - 2; i++) {
	    if (!try_place_block (field, rotated_block, i, 0, &f_new)) {
		this_score = //fmax (guess_best_next_next_score (f_new),
				   get_field_score (f_new);
		if (this_score > max_score) {
		    max_score = this_score;
		}
	    }
	}
    }

    return max_score;
}

/* find the position for the current block that yields the highest score */
static float get_best_move (FIELD field, int block1, int block2,
			   int y, int *pos, int *rot) {
    FIELD f_new;
    float this_score, next_score, combined_score;
    float max_score;
    int i, r;
    int max_i = 0, max_r = 0;
    int rotated_block;

    max_score = -MAXFLOAT;

#ifdef WITH_DEBUG
    if (debug > 3) {
	fprintf (stderr, "get_best_move(..., b1 = %d, b2 = %d, y = %d, ...)\n",
		 block1, block2, y);
    }
#endif

    for (i = -2; i < TC_FIELD_WIDTH - 2; i++) {
	if (!try_place_block (field, block1, i, y, &f_new)) {
	    if (block2) {
		this_score = get_field_score (f_new);
		next_score = get_best_next_score (f_new, block2);
		combined_score = this_score + 0.8 * next_score;
		if (combined_score > max_score) {
		    max_score = combined_score;
		    max_r = 0;
		    max_i = i;
		} else if ((combined_score == max_score) &&
			   (my_rand (2) == 1)) {
		    max_r = 0;
		    max_i = i;
		}
	    } else {
		this_score = get_field_score (f_new);
		if (this_score > max_score) {
		    max_score = this_score;
		    max_r = 0;
		    max_i = i;
		} else if ((this_score == max_score) &&
			   (my_rand (2) == 1)) {
		    max_score = this_score;
		    max_r = 0;
		    max_i = i;
		}
	    }
	}
    }

    rotated_block = block1;
    for (r = 1; r <= max_rotations[block1]; r++) {
	rotated_block = rotate_cw[rotated_block];
	for (i = -2; i < TC_FIELD_WIDTH - 2; i++) {
	    if (!try_place_block (field, rotated_block, i, y, &f_new)) {
		if (block2) {
		    this_score = get_field_score (f_new);
		    next_score = get_best_next_score (f_new, block2);
		    combined_score = this_score + 0.8 * next_score;
		    if (combined_score > max_score) {
			max_score = combined_score;
			max_r = r;
			max_i = i;
		    }
		} else {
		    this_score = get_field_score (f_new);
		    if (this_score > max_score) {
			max_score = this_score;
			max_r = r;
			max_i = i;
		    }
		}
	    }
	}
    }

    if (rot) { *rot = max_r; }
    if (pos) { *pos = max_i; }

    return max_score;
}

/* use the specials if we have some */
static void use_specials () {
    int i, s;
    int pl;
    FIELD other_players_field;
    field_stats other_players_stat;
    int chosen_p = -1;
    float chosen_score = -MAXFLOAT;
    float sc;
    static long special_age = 0; /* the # of rounds the current special
				    remained unused */

    special_age++;
    
    if (!(s = my_specials[0])) {
	special_age = 0;
	return;
    }

#ifdef WITH_DEBUG
    if (debug) {
	fprintf (stderr, "DEBUG: current special: %s\n",
		 debug_get_spec_name (s));
    }
#endif

    switch (s) {
	case TC_SPECIAL_ADD_LINE: 
	    /* use this on a random player */
	    pl = my_rand (num_players);
	    if (pl != my_number) {
		UseSpecial (pl);
		special_age = 0;
	    }
	    return;
	case TC_SPECIAL_CLEAR_LINE:
	    /* use this if the bottom row has blocked holes in it */
	    if (my_stat.row_blocked_ratio[TC_FIELD_HEIGHT - 1] > 0.0) {
		UseSpecial (my_number);
		special_age = 0;
	    }
	    return;
	case TC_SPECIAL_CLEAR_SPECIALS:
	    /* clear our my specials if I have block bombs in the field --
	       they are just nasty... */
	    if (my_stat.num_block_bombs > 0) {
		UseSpecial (my_number);
		special_age = 0;
	    }
	case TC_SPECIAL_BLOCK_GRAVITY:
	    /* use gravity on myself if I have many holes and the avg
	       height is over the 3rd of the field */
	    if ((my_stat.field_density < 0.95) &&
		(my_stat.avg_col_height > (TC_FIELD_HEIGHT/3))) {
		UseSpecial (my_number);
		special_age = 0;
	    }
	    return;
	case TC_SPECIAL_NUKE:
	    /* nuke my field if it has holes or block bombs in it
	       and my score would be higher if the field was empty
	       and the avg column height it over 1/4 */
	    if (((my_stat.field_density < 0.95) || 
		 (my_stat.num_block_bombs > 1)) && 
		(my_stat.avg_col_height > (TC_FIELD_HEIGHT/4)) &&
		(my_score < score_of_empty)) {
		UseSpecial (my_number);
		special_age = 0;
	    }
	    return;
    }

    chosen_p = -1;
    chosen_score = -MAXFLOAT;
    /* when to use specials on other players */
    for (i = 0; (i < num_players) && (chosen_p == -1); i++) {
	if (i == my_number) {
	    continue;
	}
	if (!GetFieldP (i, (char *)other_players_field)) {
	    continue;
	}
	count_field_statistics (other_players_field, &other_players_stat);

	switch (s) {
	    case TC_SPECIAL_CLEAR_SPECIALS:
		/* Clear their specials if they have more than 2.
		   Choose the one with the most number of specials */
		if ((other_players_stat.num_specials > 2) &&
		    (other_players_stat.num_specials > chosen_score)) {
		    chosen_score = other_players_stat.num_specials;
		    chosen_p = i;
		}
		break;
	    case TC_SPECIAL_NUKE:
		/* Nuke their field if they have more then 5 specials and
		   the average height of the columns is less than 1/4
		   of the field.
		   Chosen the one with the most number of specials */
		if ((other_players_stat.num_specials > 5) &&
		    (other_players_stat.num_specials > chosen_score) &&
		    (other_players_stat.avg_col_height <
		     (TC_FIELD_HEIGHT / 4))) {
		    chosen_score = other_players_stat.num_specials;
		    chosen_p = i;
		}
		break;
	    case TC_SPECIAL_BLOCK_BOMB:
		/* use the block bomb against the player who has the
		   most block bombs in their field */
		sc = get_field_score (other_players_field);
		if (other_players_stat.num_block_bombs &&
		    (other_players_stat.num_block_bombs > chosen_score)) {
		    chosen_score = sc;
		    chosen_p = i;
		}
		break;
	    case TC_SPECIAL_BLOCK_QUAKE:
		/* block quake the player with the highest score out of
		   those who have more that 1/4 avg column height */
		sc = get_field_score (other_players_field);
		if ((sc > chosen_score) &&
		    (other_players_stat.avg_col_height > 
		     (TC_FIELD_HEIGHT / 4))) {
		    chosen_score = sc;
		    chosen_p = i;
		}
		break;
	    case TC_SPECIAL_CLEAR_RANDOM:
		/* clear random on the player with the highest score out of
		   those who have more that 1/4 avg column height */
		sc = get_field_score (other_players_field);
		if ((sc > chosen_score) &&
		    (other_players_stat.avg_col_height > 
		     (TC_FIELD_HEIGHT / 4))) {
		    chosen_score = sc;
		    chosen_p = i;
		}
		break;
	    case TC_SPECIAL_SWITCH_FIELDS:
		/* switch field with the player with the best score
		   if they don't have too many blocks in their field */
		sc = get_field_score (other_players_field);
		if ((sc > my_score) &&
		    (sc > chosen_score) &&
		    (other_players_stat.avg_col_height < 
		     my_stat.avg_col_height)) {
		    chosen_score = sc;
		    chosen_p = i;
		}
		break;
	}
    }

    if (chosen_p != -1) {
	UseSpecial (chosen_p);
	special_age = 0;
    }

    /* discard the current special if it remained unused for 10 rounds
       and is not the last one */
    if ((special_age > 10) && (my_specials[1])) {
	DiscardSpecial ();
	special_age = 0;
#ifdef WITH_DEBUG
	if (debug) {
	    fprintf (stderr, "DEBUG: discarding special %s after 10 rounds\n", 
		     debug_get_spec_name (s));
	}
#endif
	return;
    }

    /* discard the current special if it remained unused for 5 rounds
       and I already have too many  */
    for (i = 0; my_specials[i]; i++);
    if ((special_age > 5) && ((max_specials - i) < 2)) {
	DiscardSpecial ();
	special_age = 0;
#ifdef WITH_DEBUG
	if (debug) {
	    fprintf (stderr, "DEBUG: discarding special %s after 5 rounds\n", 
		     debug_get_spec_name (s));
	}
#endif
	return;
    }
}

static void play_game () {
    int status;
    int blocktype, xpos, ypos;
    int next_blocktype;
    int i, x, r;
    FIELD field;
    float score;

    while (1) {
	status = GetCurrentBlock(&blocktype, &xpos, &ypos);
	GetNextBlock (&next_blocktype);
	
	if (status == 1) { /* bot lost the game */
#ifdef WITH_DEBUG
	    if (debug) {
		printf ("DEBUG: I lost ):\n");
	    }
#endif
	    break;
	}
	
	if (status == 2) { /* bot won */
#ifdef WITH_DEBUG
	    if (debug) {
		printf ("DEBUG: I won! (:\n");
	    }
#endif
	    break;
	}

	if (blocktype == TC_BLOCK_NONE) 
	{
	    if (do_delay) {
		usleep(50000);
	    } else {
		usleep(1000);
	    }
	    continue;
	}

	if (!GetFieldP(my_number, (char *)field)) {
	    return;
	}	
	GetSpecials (my_specials);

	score = get_best_move (field, blocktype, next_blocktype, ypos, &x, &r);
#ifdef WITH_DEBUG
	if (debug > 1) {
	    fprintf (stderr, "DEBUG: MOVE: xpos->%d, rotate->%d, score=%f\n",
		     x, r, score);
	}
#endif

	if (r == 1) {
	    MoveBlock (TC_ROTATE_ANTICLOCKWISE);
	} else if (r == 2) {
	    MoveBlock (TC_ROTATE_ANTICLOCKWISE);
	    MoveBlock (TC_ROTATE_ANTICLOCKWISE);
	} else if (r == 3) {
	    MoveBlock (TC_ROTATE_CLOCKWISE);
	}

	if (x < xpos) {
	    for (i = 0; i < xpos - x; i++) {
		MoveBlock (TC_MOVE_LEFT);
	    }
	} else {
	    for (i = 0; i < x - xpos; i++) {
		MoveBlock (TC_MOVE_RIGHT);
	    }
	}

	MoveBlock (TC_MOVE_DROP);

	if (!GetFieldP(my_number, (char *)field)) {
	    return;
	}
	count_field_statistics (field, &my_stat);
	my_score = get_field_score (field);
#ifdef WITH_DEBUG
	if (debug > 1) {
	    print_field_with_stats (field, &my_stat);
	    fprintf (stderr, "score = %f\n", my_score);
	    fflush (stderr);
	}
#endif

	use_specials ();
    }

}

static void process_args (int argc, char **argv) {
    int i = 1;

    servername = "localhost";
    username = (char *)default_username;
    while (i < argc) {
	if (!strcmp (argv[i], "-u")) {
	    if (i + 1 >= argc) {
		fprintf (stderr, "ERROR: argument expected after -u\n");
		exit (1);
	    }
	    username = argv[i + 1];
	    ++i;
	} else if (!strcmp (argv[i], "-h")) {
	    if (i + 1 >= argc) {
		fprintf (stderr, "ERROR: argument expected after -h\n");
		exit (1);
	    }
	    servername = argv[i + 1];
	    ++i;
#ifdef WITH_DEBUG
	} else if (!strcmp (argv[i], "-d")) {
	    debug++;
#endif
	} else {
	    fprintf (stderr, "ERROR: unknown option %s\n", argv[i]);
	    exit (1);
	}
	++i;
    }

    if (getenv ("TETRIFAST") != NULL) {
	do_delay = 0;
    }
}

static void connect_to_server () {
    if (connectToServer (servername, 9467, username, "", "")) {
	fprintf (stderr, "ERROR: %s\n", tc_error_string);
	exit (1);
    }
}

static void start_game () {
    FIELD f;

    bzero (f, sizeof (FIELD));
    score_of_empty = get_field_score (f);

    if (Start (&gdata) != 0) {
	fprintf (stderr, "ERROR: Game start failed\n");
	exit (1);
    } else {
#ifdef WITH_DEBUG
	if (debug) {
	    printf ("DEBUG: Game started\n");
	    fflush (stdout);
	}
#endif
    }

    my_number = GetOwnPlayerNumber (gdata);
    num_players = GetNumPlayers (gdata);
    max_specials = GetMaxSpecials (gdata);
    if (my_specials) {
	free (my_specials);
    }
    my_specials = malloc (max_specials * sizeof (char));
}

static void end_game () {
    FreeGameData  (gdata);
}

int main (int argc, char **argv) {

    process_args (argc, argv);
    connect_to_server ();
    while (1) {
	start_game ();
	play_game ();
	end_game ();
    }
}
