/*
Kim Oldfield   lca2004@oldfield.wattle.id.au

$Id: kimo.c,v 1.29 2004/01/16 01:10:51 kim Exp kim $

Copyright 2003-2004 by Kim Oldfield.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not please download a copy from
   http://www.gnu.org/licenses/gpl.txt

ToDo:
- include number of specials in calcscore()
- allow rotating of blocks anywhere on field
  - include optimisation of move searching
- revise calcscore() - how can it be improved?
- check block lookahead - some of the moves are very dumb
  - maybe they aren't too bad considering both blocks, but the second block
    doesn't always end up in the corect position
- watch a game, looking for "wrong" moves, and work out how to fix them

The next line is used by "make dist":
$Revision: 1.29 $
*/

/* documentation strings */

char const version[] = "Kim Oldfield's tetinet bot\n\
RCS revision: $Id: kimo.c,v 1.29 2004/01/16 01:10:51 kim Exp kim $\n";

static char const helptext[] = "Kim Oldfield's tetinet bot\n\
Options:\n\
 -H       print this help text\n\
 -v       print version\n\
 -d[#]    print debugging information, see below for values of #\n\
 -f       flush output for each piece\n\
 -s       single block lookup only - disable 2 piece lookahead\n\
 -l file  load a field from the named file\n\
 -h host  connect to tetribot on this host\n\
 -p port  connect to tetribot on this port\n\
 -u user  connect as this user\n\
\n\
Debug levels:\n\
0  don't print anything (default)\n\
1  print details about major events or errors only (eg game ended)\n\
2  print field for each block being played\n\
3  as for 2, with some extra information\n\
4  print score for each final position considered\n\
5  print a breakdown of score for each possable position\n\
6  random extra stuff which was useful at some stage\n\
";

/******************************************************************/
/* includes */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <assert.h>
#include <ctype.h>
#include <getopt.h>
#include <stdarg.h>
#include <limits.h>
#include <math.h>
#include <sys/time.h>
#include <errno.h>

#include "tclient.h"
/* #include "tcommon.h" */

/******************************************************************/
/* #defines */

#define W 		TC_FIELD_WIDTH
#define H 		TC_FIELD_HEIGHT
#define fieldxy(x,y)	field[(y)*W+(x)]

#define BLOCKSIZE	4
#define cellxy(x,y)	cell[x + y * BLOCKSIZE]
#define NUMBLOCKS	(sizeof(blocks)/sizeof(blocks[0]))

#define NOMOVE		-10
#define NEWX		4	/* where new blocks appear */
#define NEWY		0	/* where new blocks appear */

#define sqr(n) ((n)*(n))

#define MAXPLAYERS	20

/******************************************************************/
/* data structure definitions */

struct blockstruct {
	int index;
	char name[4], *cellprint, cell[16];
	int	left,	/* position of left most cell */
		right,	/* position of right most cell */
		height,	/* vertical height of entire block */
		bottom[BLOCKSIZE], /* distance below top of each column */
		numrot; /* number of rotations for this piece */
	struct blockstruct *rotlist[4];
};

/******************************************************************/
/* global variables */

struct TC_GameData *gdata;
static int debug = 0, flush = 0, singlelookahead = 0;
static int numplayers;
static struct blockstruct blocks[] = {
	{ TC_BLOCK_NONE, "N", /* 0 */
		"...."
		"...."
		"...."
		"...." },
	{ TC_BLOCK_O, "O", /* 1 */
		".xx."
		".xx."
		"...."
		"...." },
	{ TC_BLOCK_I_1, "I1", /* 2 */
		"xxxx"
		"...."
		"...."
		"...." },
	{ TC_BLOCK_I_2, "I2", /* 3 */
		"..x."
		"..x."
		"..x."
		"..x." },
	{ TC_BLOCK_S_1, "S1", /* 4 */
		"..xx"
		".xx."
		"...."
		"...." },
	{ TC_BLOCK_S_2, "S2", /* 5 */
		".x.."
		".xx."
		"..x."
		"...." },
	{ TC_BLOCK_Z_1, "Z1", /* 6 */
		".xx."
		"..xx"
		"...."
		"...." },
	{ TC_BLOCK_Z_2, "Z2", /* 7 */
		"..x."
		".xx."
		".x.."
		"...." },
	{ TC_BLOCK_L_1, "L1", /* 8 */
		".xxx"
		".x.."
		"...."
		"...." },
	{ TC_BLOCK_L_2, "L2", /* 9 */
		".x.."
		".x.."
		".xx."
		"...." },
	{ TC_BLOCK_L_3, "L3", /* 10 */
		"...x"
		".xxx"
		"...."
		"...." },
	{ TC_BLOCK_L_4, "L4", /* 11 */
		".xx."
		"..x."
		"..x."
		"...." },
	{ TC_BLOCK_J_1, "J1", /* 12 */
		".xxx"
		"...x"
		"...."
		"...." },
	{ TC_BLOCK_J_2, "J2", /* 13 */
		".xx."
		".x.."
		".x.."
		"...." },
	{ TC_BLOCK_J_3, "J3", /* 14 */
		".x.."
		".xxx"
		"...."
		"...." },
	{ TC_BLOCK_J_4, "J4", /* 15 */
		"..x."
		"..x."
		".xx."
		"...." },
	{ TC_BLOCK_T_1, "T1", /* 16 */
		".xxx"
		"..x."
		"...."
		"...." },
	{ TC_BLOCK_T_2, "T2", /* 17 */
		".x.."
		".xx."
		".x.."
		"...." },
	{ TC_BLOCK_T_3, "T3", /* 18 */
		"..x."
		".xxx"
		"...."
		"...." },
	{ TC_BLOCK_T_4, "T4", /* 19 */
		"..x."
		".xx."
		"..x."
		"...." },
};

/******************************************************************/
/* function prototypes */

static double calcscore(char *field);
static inline void dprintf(int v, char *fmt, ...) __attribute__ ((format(printf, 2, 3)));
static char specialchar(int special);

/* my definitions of the tclient functions which can be changed to block or
 * not block */
#define USEOLD

static inline int myGetCurrentBlock(int *BlockType, int *XPosition, int *YPosition) {
#ifdef USEOLD
	return GetCurrentBlock(BlockType, XPosition, YPosition);
#else
	txGetCurrentBlock();
	return rxGetCurrentBlock(BlockType, XPosition, YPosition);
#endif
}

static inline void myGetNextBlock(int *BlockType) {
#ifdef USEOLD
	GetNextBlock(BlockType);
#else
	txGetNextBlock();
	rxGetNextBlock(BlockType);
#endif
}

static inline char *myGetFieldP(int PlayerNum, char *Field) {
#ifdef USEOLD
	return GetFieldP(PlayerNum, Field);
#else
	txGetField(PlayerNum);
	return rxGetField(Field);
#endif
}

static inline void myGetSpecials(char *Specials) {
#ifdef USEOLD
	GetSpecials(Specials);
#else
	txGetSpecials();
	rxGetSpecials(Specials);
#endif
}

/******************************************************************/
/* initialisation functions */

static void blockinit() {
	/* Calculates the extra fields for the block array */
	int i, j, x, y, start, num;
	struct blockstruct *b;

	for(i = 0; i < NUMBLOCKS; ++i) {
		b = blocks + i;
		assert(b->index == i);

		/* fill in block.cell[*] */
		for(x = 0; x < BLOCKSIZE; ++x)
			for(y = 0; y < BLOCKSIZE; ++y)
				b->cellxy(x, y) = (b->cellprint[x + y * BLOCKSIZE] == '.') ? 0 : 'A' + i;
				
		/* find left edge */
		b->left = 0;
		for(x = 0; x < BLOCKSIZE; ++x)
			for(y = 0; y < BLOCKSIZE; ++y)
				if (b->cellxy(x, y)) {
					b->left = x;
					goto endleft;
				}
		endleft:

		/* find right edge */
		b->right = 0;
		for(x = BLOCKSIZE - 1; x >= 0; --x)
			for(y = 0; y < BLOCKSIZE; ++y)
				if (b->cellxy(x, y)) {
					b->right = x;
					goto endright;
				}
		endright:

		/* find bottom */
		b->height = 0;
		for(x = b->left; x <= b->right; ++x) {
			for(y = BLOCKSIZE; y >= 0; --y) {
				if (b->cellxy(x, y)) {
					b->bottom[x] = y;
					break;
				}
			}
			if (b->bottom[x] > b->height) {
				b->height = b->bottom[x];
			}
		}

		/* rotlist[] array */
		for(start = i; start >= 0 && blocks[i].name[0] == blocks[start].name[0]; --start)
			; /* find start */
		++start;
		for(num = 0; num < NUMBLOCKS && blocks[i].name[0] == blocks[start+num].name[0]; ++num)
			; /* find end */
		assert(num == 1 || num == 2 || num == 4);
		blocks[i].numrot = num;
		for(j = 0; j < 4; ++j) {
			blocks[i].rotlist[j] = blocks+(((i+j-start)%num)+start);
		}
	}
}

static char *playername(int player) {
	return GetPlayerName(GetPlayer(gdata, player));
}

/******************************************************************/
/* diagnostic and debug functions */

#if 0
/* this is faster */
#define dprintf(v, ...)	/* nothing */
#else
static inline void dprintf(int v, char *fmt, ...) {
	/* debug printf */
	va_list args;
	if (debug >= v) {
		va_start(args, fmt);
		vprintf(fmt, args);
		va_end(args);
	}
}
#endif


static void printblock(struct blockstruct *b) {
	int x, y;
	assert(b);
	printf("%2i %-2s", b->index, b->name);
	for(y = 0; y < BLOCKSIZE; ++y) {
		putchar(' ');
		for(x = 0; x < BLOCKSIZE; ++x) {
			putchar(b->cellprint[x + y * BLOCKSIZE]);
		}
	}
	printf(" %i %i %i [%i %i %i %i] %i [%2i %2i %2i %2i]\n",
		b->left, b->right, b->height,
		b->bottom[0], b->bottom[1], b->bottom[2], b->bottom[3],
		b->numrot,
		b->rotlist[0]->index, b->rotlist[1]->index,
		b->rotlist[2]->index, b->rotlist[3]->index);
}

static void printallblocks() {
	int i;
	printf(" # name cells             l r h bottom  rot nrot\n");
	for(i = 0; i < NUMBLOCKS; ++i) {
		printblock(blocks+i);
	}
}

static void printblockline(struct blockstruct *b, int y) {
	/* used by printfield to print the block */
	assert(b >= blocks && b <= blocks + NUMBLOCKS);
	if (y == 5) {
		printf("%2i  ", b->index);
	} else if (y == 6) {
		printf("%-4s", b->name);
	} else {
		printf("%4.4s", b->cellprint + ((y - 7) * BLOCKSIZE));
	}
}

static void printfield(char *field, char *specials, struct blockstruct **block, int blockx, int blocky) {
	/* prints the field, block, and specials list */
	char c;
	int x, y;
	double score = calcscore(field);
	assert(field);
	/* last element in block[] array should be blocks */
	if (block) {
		assert(block[0] && block[0] >= blocks && block[0] < blocks + NUMBLOCKS);
		assert(block[0] == blocks || (block[1] && block[1] >= blocks && block[1] < blocks + NUMBLOCKS));
		assert(block[1] == blocks || block[2] == blocks);
	}
	for(y = 0; y < H; ++y) {
		printf("%2i ", y);
		for(x = 0; x < W; ++x) {
			c = fieldxy(x, y);
			if (c == 0) {
				putchar('.');
			} else if (c < TC_SPECIAL_FIRST_SPECIAL) {
				putchar(c + 'A' - 1);
			} else if (c <= TC_SPECIAL_LAST_SPECIAL) {
				putchar(c + '0' - TC_SPECIAL_FIRST_SPECIAL);
			} else {
				putchar('?');
			}
		}
		if (y == 0) {
			printf("  Score: %.2f", score);
		} else if (specials && y == 2) {
			printf("  %i specials:", strlen(specials));
		} else if (specials && y == 3) {
			printf(" ");
			for( ; specials[0]; ++specials) {
				printf(" %c", specialchar(specials[0]));
			}
		} else if (block && y > 4 && y < BLOCKSIZE + 6) {
			printf("  ");
			printblockline(block[0], y);
			printf("  ");
			printblockline(block[1], y);
		} else if (y == BLOCKSIZE + 7) {
			printf("  (%i, %i)", blockx, blocky);
		}
		putchar('\n');
	}
}

static char specialstring[20];
static char const *specialname(int special) {
	switch(special) {
		case TC_SPECIAL_ADD_LINE:
			return "ADD_LINE";
		case TC_SPECIAL_CLEAR_SPECIALS:
			return "CLEAR_SPECIALS";
		case TC_SPECIAL_CLEAR_LINE:
			return "CLEAR_LINE";
		case TC_SPECIAL_BLOCK_GRAVITY:
			return "BLOCK_GRAVITY";
		case TC_SPECIAL_NUKE:
			return "NUKE";
		case TC_SPECIAL_BLOCK_BOMB:
			return "BLOCK_BOMB";
		case TC_SPECIAL_BLOCK_QUAKE:
			return "BLOCK_QUAKE";
		case TC_SPECIAL_CLEAR_RANDOM:
			return "CLEAR_RANDOM";
		case TC_SPECIAL_SWITCH_FIELDS:
			return "SWITCH_FIELDS";
		default:
			snprintf(specialstring, sizeof(specialstring), "Unknown(%i)", special);
			return specialstring;
	}
}
static char specialchar(int special) {
	switch(special) {
		case TC_SPECIAL_ADD_LINE:
			return 'A';
		case TC_SPECIAL_CLEAR_SPECIALS:
			return 'B';
		case TC_SPECIAL_CLEAR_LINE:
			return 'C';
		case TC_SPECIAL_BLOCK_GRAVITY:
			return 'G';
		case TC_SPECIAL_NUKE:
			return 'N';
		case TC_SPECIAL_BLOCK_BOMB:
			return 'B';
		case TC_SPECIAL_BLOCK_QUAKE:
			return 'Q';
		case TC_SPECIAL_CLEAR_RANDOM:
			return 'R';
		case TC_SPECIAL_SWITCH_FIELDS:
			return 'S';
		default:
			return '?';
	}
}

static void printmove(int move) {
	switch(move) {
		case TC_MOVE_LEFT:
			printf("MOVE_LEFT");
			break;
		case TC_MOVE_RIGHT:
			printf("MOVE_RIGHT");
			break;
		case TC_ROTATE_CLOCKWISE:
			printf("ROTATE_CLOCKWISE");
			break;
		case TC_ROTATE_ANTICLOCKWISE:
			printf("ROTATE_ANTICLOCKWISE");
			break;
		case TC_MOVE_DOWN:
			printf("MOVE_DOWN");
			break;
		case TC_MOVE_DROP:
			printf("MOVE_DROP");
			break;
		default:
			printf("Unknown(%i)", move);
			break;
	}
}

static void printgamedetails(void) {
	int i;

	printf("\nGame details:\n");
	printf("MaxNumSpecials = %i\n", GetMaxSpecials(gdata));
	printf("numplayers = %i\n", numplayers);
	printf("I am player number %i\n", GetOwnPlayerNumber(gdata));
	for(i = 0; i < numplayers; ++i) {
		printf("PlayerNumber = %i  ", i);
		printf("PlayerName = %-10s   ", playername(i));
		printf("TeamName = %s\n", GetPlayerTeam(GetPlayer(gdata, i)));
	}
	printf("End of game details.\n\n");
}

static void printallplayers(void) {
	int player;
	char field[H*W];

	printf("There are %i players, I'm player %i.\n\n",
		numplayers, GetOwnPlayerNumber(gdata));
	for(player = 0; player < numplayers; ++player) {
		printf("Player %i, %s\n", player, playername(player));
		if (GetFieldP(player, field)) {
			printfield(field, NULL, NULL, -1, -1);
		} else {
			printf("Player %i has no field.\n", player);
		}
		printf("\n");
	}
}

/******************************************************************/
/* misc utility functions */

static int timediff(struct timeval *t1, struct timeval *t2) {
	int u = t2->tv_usec - t1->tv_usec;
	long s = t2->tv_sec - t1->tv_sec;
	if (u < 0) {
		u += 1000000;
		s -= 1;
	}
	return u + s * 1000000;
}

static int randbool(void) {
	/* y/n random function */
	return ! (rand() & 0x800); /* use a higher order bit */
}

static void minisleep(void) {
	/* wait for a small amount of time */
	if (flush)
		fflush(stdout);
	usleep(100000); /* 0.1 sec */
}

static int fieldsame(char *field, char *field2) {
	/* returns 1 if field and field2 are the same
	 * returns 0 if they are different */
	int i;
	if (field == field2)
		return 1;
	if (!field || !field2)
		return 0;
	for(i = 0; i < H * W; ++i) {
		if (field[i] != field2[i]) {
			return 0;
		}
	}
	return 1;
}

/******************************************************************/
/* field manipulation functions */

static int findbestplayer(double *scorep) {
	/* returns the number of the player with the best score */
	int player, bestplayer = -1;
	double score, bestscore = -1e30;
	char field[H*W];

	bestplayer = -1;
	for(player = 0; player < numplayers; ++player) {
		if (player == GetOwnPlayerNumber(gdata))
			continue; /* do not check my own field */
		if (!GetFieldP(player, field)) {
			/* player has died */
			continue;
		}
		score = calcscore(field);
		if (score > bestscore || (score == bestscore && randbool())) {
			bestscore = score;
			bestplayer = player;
		}
	}
	if (scorep && bestplayer >= 0)
		*scorep = bestscore;
	return bestplayer;
}

static int specialweight(int special) {
	switch(special) {
		case TC_SPECIAL_NUKE:
		case TC_SPECIAL_BLOCK_GRAVITY:
		case TC_SPECIAL_SWITCH_FIELDS:
			return 5;
		case TC_SPECIAL_CLEAR_SPECIALS:
		case TC_SPECIAL_BLOCK_BOMB:
			return 3;
		case TC_SPECIAL_BLOCK_QUAKE:
		case TC_SPECIAL_CLEAR_RANDOM:
			return 2;
		case TC_SPECIAL_CLEAR_LINE:
		case TC_SPECIAL_ADD_LINE:
			return 1;
		default:
			dprintf(1, "Invalid special: specialweight(%i).\n", special);
			return 0;
	}
}

static int findmostspecials(int *count) {
	/* returns the number of the player with the most specials
	 * where some specials have been given more weight */
	int i, player, mostplayer = -1;
	int specialcount, mostspecials = 0;
	char field[H*W];

	for(player = 0; player < numplayers; ++player) {
		if (player == GetOwnPlayerNumber(gdata))
			continue; /* do not check my own field */
		if (!GetFieldP(player, field))
			continue; /* player has died */
		specialcount = 0;
		for(i = 0; i < H * W; ++i) {
			if (field[i] > TC_SPECIAL_FIRST_SPECIAL) {
				specialcount += specialweight(field[i]);
			}
		}
		if (specialcount > mostspecials || (specialcount == mostspecials && randbool())) {
			mostspecials = specialcount;
			mostplayer = player;
		}
	}
	if (count && mostplayer >= 0)
		*count = mostspecials;
	return mostplayer;
}

static int findmostblocks(int *count) {
	/* returns the number of the player with the most blocks on their
	 * field */
	int i, player, mostplayer = -1;
	int blockcount, mostblocks = 0;
	char field[H*W];

	for(player = 0; player < numplayers; ++player) {
		if (player == GetOwnPlayerNumber(gdata))
			continue; /* do not check my own field */
		if (!GetFieldP(player, field))
			continue; /* player has died */
		blockcount = 0;
		for(i = 0; i < H * W; ++i) {
			if (field[i]) {
				++blockcount;
			}
		}
		if (blockcount > mostblocks || (blockcount == mostblocks && randbool())) {
			mostblocks = blockcount;
			mostplayer = player;
		}
	}
	if (count && mostplayer >= 0)
		*count = mostblocks;
	return mostplayer;
}

static int findmostblockbombs() {
	/* returns the number of the player with the most blockbombs */
	int i, player, mostplayer = -1;
	int specialcount, mostspecials = 0;
	char field[H*W];

	mostplayer = -1;
	for(player = 0; player < numplayers; ++player) {
		if (player == GetOwnPlayerNumber(gdata))
			continue; /* do not check my own field */
		if (!GetFieldP(player, field))
			continue; /* player has died */
		specialcount = 0;
		for(i = 0; i < H * W; ++i) {
			if (field[i] == TC_SPECIAL_BLOCK_BOMB) {
				++specialcount;
			}
		}
		if (specialcount > mostspecials || (specialcount == mostspecials && randbool)) {
			mostspecials = specialcount;
			mostplayer = player;
		}
	}
	return mostplayer;
}

static int mustplayspecial(char *field) {
	/* check if there is a block in the top V of the field */
	int i, x, y;
#define VSIZE (W/2)
#if VSIZE * 2 > W
#error VSIZE is too big. You must fix the code below.
#endif
	for(i = -VSIZE; i <= VSIZE; ++i) {
		x = W / 2 + i;
		for(y = 0; y <= VSIZE - abs(VSIZE - i); ++y) {
			if (fieldxy(x, y)) {
				return 1;
			}
		}
	}
	return 0;
}

static int playspecial(int playnow) {
	/* play the next special block
	 * playnow = 1, must play now
	 * playnow = 0, work out to play or not
	 * returns:
	 *   < 0 : nothing played
	 *   0-n : player on which special was played
	 * */
	int player = -1, otherplayer, count, x;
	char specials[GetMaxSpecials(gdata)];
	char field[H*W];
	double myscore, otherscore;
	double urgent; /* 0 to 1000 */

	if (playnow) {
		dprintf(2, "playspecial() called with playnow=1\n");
	}
	GetSpecials(specials);
	if (!specials[0])
		return -1;
	if (!GetFieldP(GetOwnPlayerNumber(gdata), field)) { /* no field */
		return -1;
	}
	myscore = calcscore(field);
	if (mustplayspecial(field)) {
		dprintf(2, "Field is too full - must play special.\n");
		playnow = 1;
	}
	if (playnow) {
		urgent = 1000;
	} else {
		double tmp1 = 800 * strlen(specials) / GetMaxSpecials(gdata);
		double tmp2 = 900 - myscore;
		urgent = (tmp1 > tmp2) ? tmp1 : tmp2;
	}
	dprintf(3, "playspecial(), urgent = %0.2f\n", urgent);
	if (urgent < 0) urgent = 0;
	if (urgent > 1000) urgent = 1000;
	if (debug >= 2) {
		dprintf(2, "Working out what to do with special %s (%i), playnow = %i:\n", specialname(specials[0]), specials[0], playnow);
		if (debug >= 3) {
			printallplayers();
		}
	}
	switch (specials[0]) {
	case 0:
		dprintf(2, "No special to play.\n");
		return -1;
	case TC_SPECIAL_CLEAR_LINE:
		if (playnow) {
			player = GetOwnPlayerNumber(gdata);
		} else {
			for(x = 0; x < W; ++x)
				if (fieldxy(x, H-1) >= TC_SPECIAL_FIRST_SPECIAL)
					count += specialweight(fieldxy(x, H-1));
			if (urgent > (500 - myscore) + 50 * count)
				player = GetOwnPlayerNumber(gdata);
		}
		break;
	case TC_SPECIAL_NUKE:
	case TC_SPECIAL_BLOCK_GRAVITY:
		if (urgent > 600) {
			player = GetOwnPlayerNumber(gdata);
		}
		break;
	case TC_SPECIAL_CLEAR_SPECIALS:
		/* find other player with most specials and play on them */
		otherplayer = findmostspecials(&count);
		if (otherplayer >= 0 && urgent * count >= 1000)
			player = otherplayer;
		break;
	case TC_SPECIAL_SWITCH_FIELDS:
		otherplayer = findbestplayer(&otherscore);
		if (otherplayer >= 0 && otherscore - myscore > 1000 - urgent)
			player = otherplayer;
		break;
	case TC_SPECIAL_BLOCK_BOMB:
		/* find player with most block bombs */
		otherplayer = findmostblockbombs(&count);
		if (otherplayer >= 0 && count * 100 > 1000 - urgent)
			player = otherplayer;
		break;
	case TC_SPECIAL_BLOCK_QUAKE:
	case TC_SPECIAL_CLEAR_RANDOM:
		/* play these on the players with lots of blocks */
	case TC_SPECIAL_ADD_LINE:
		otherplayer = findmostblocks(&count);
		if (count * 20 > (1000 - urgent))
			player = otherplayer;
		break;
	default:
		dprintf(1, "Error: unknown special %i !\n", specials[0]);
		player = -1;
		break;
	}
	if (player == GetOwnPlayerNumber(gdata)) {
		dprintf(2, "playing special %s on me (%i, %s).\n", specialname(specials[0]), player, playername(player));
		UseSpecial(player);
	} else if (player >= 0) {
		dprintf(2, "playing special %s on player %i, %s.\n", specialname(specials[0]), player, playername(player));
		UseSpecial(player);
	} else {
		if (playnow || urgent >= 1000) {
			dprintf(2, "no one to play special %s on, discarding.\n", specialname(specials[0]));
			DiscardSpecial();
		} else {
			dprintf(3, "Decided not to play special %s.\n", specialname(specials[0]));
		}
	}
	return player;
}

/******************************************************************/
static int playblock(char *field, struct blockstruct *block, int posx, int posy) {
	/* insert block in field, modify field in-place,
	 * return >= 0 for number of rows cleared
	 * -1 for nothing to rest block on, ie position is not final
	 * < -10 for error playing block */
	char const emptyrow[W] = {0};
	int x, y, rest = 0, rowscleared = 0;

	dprintf(6, "Trying to play block %i at (%i, %i).\n", block->index, posx, posy);
	if (posy < 0 || posy + block->height > H) {
		/* invalid vertical position */
		dprintf(3, "Invalid vertical position to play: %i\n", posy);
		return -11;
	}
	if (posx + block->left < 0 || posx + block->right >= W) {
		/* invalid horizontal position */
		dprintf(3, "Invalid horizontal position to play: %i\n", posx);
		return -12;
	}
	for(x = block->left; x <= block->right; ++x) {
		if (posy+block->bottom[x] == H ||
				fieldxy(posx+x, posy+block->bottom[x]+1))
			++rest;
		for(y = 0; y < BLOCKSIZE; ++y) {
			if (block->cellxy(x, y)) {
				if (fieldxy(posx+x, posy+y)) {
					/* block already in this cell, can't
					 * play this block here */
					dprintf(6, "Can't play on top of existing block (%i, %i)+(%i, %i)\n", posx, posy, x, y);
					return -13;
				}
				fieldxy(x+posx, y+posy) = block->cellxy(x, y);
			}
		}
	}
	if (! rest && posy + block->height != H - 1) {
		dprintf(7, "Nothing to rest block on at (%i, %i).\n", posx, posy);
		return -1;
	}
	/* remove complete rows */
	for(y = posy; y <= posy + block->height; ++y) {
		for(x = 0; x < W; ++x) {
			if (! fieldxy(x, y)) {
				break;
			}
		}
		if (x == W) { /* delete this row */
			++rowscleared;
			memmove(field + W, field, y * W);
			memcpy(field, emptyrow, W);
		}
	}
	dprintf(6, "Done playblock, posx = %i, posy = %i\n", posx, posy);
	return rowscleared;
}

#define PRINTSCOREKEY	dprintf(5, "%-8s %8s %8s %8s %8s %8s %8s %8s\n", "( x,  y)", "height", "maxh", "h-edge", "v-edge", "smooth", "h-comp", "total");

/******************************************************************/
static double calcscore(char *field) {
	/* calculate a "score" for the given field */
	/* higher scores are better */
	double score = 100, pscore, dist;
	int x, y, height[W], maxheight = H, previous, lasttop, c;

	/* heights of each column, gaps in columns */
	for(x = 0; x < W; ++x) {
		for(y = 0; y < H && ! fieldxy(x, y); ++y)
			;
		/* distance from top of column to top centre of field */
		dist = sqrt(sqr(x - (W-1)/2.0) + sqr(y + W/3));
		/* make further distances less important, and closer
		 * distances more important */
		score += 40 - 200 / (dist + 5);
		height[x] = y;
		if (maxheight < y)
			y = maxheight;
		previous = 0;
		lasttop = y;
		for( ; y < H; ++y) {
			if (previous && ! fieldxy(x, y)) {
				/* score -= 5; *//* top of gap is bad */
				score -= 10 + (y - lasttop);
				previous = fieldxy(x, y);
			} else if (! previous && fieldxy(x, y)) {
				lasttop = y;
				previous = fieldxy(x, y);
			}
		}
	}
	dprintf(5, "%8.2f", score);
	score += 2 * maxheight; /* try to minimise height */
	dprintf(5, " %8.2f", (double) maxheight);

	/* edges - try to avoid them */
	/* compare blocks horizontally */
	pscore = 0.0;
	for(y = 0; y < H; ++y) {
		if (!fieldxy(0, y))
			pscore -= 1;
		for(x = 0; x < W - 1; ++x)
			if (!fieldxy(x, y) ^ !fieldxy(x+1, y))
				pscore -= 1;
		if (!fieldxy(W - 1, y))
			pscore -= 1;
	}
	pscore /= 2.0;
	score += pscore;
	dprintf(5, " %8.2f", pscore);

	/* compare blocks vertically */
	pscore = 0.0;
	for(x = 0; x < W; ++x) {
		if (fieldxy(x, 0))
			pscore -= 1;
		for(y = 0; y < H - 1; ++y)
			if (!fieldxy(x, y) ^ !fieldxy(x, y + 1))
				pscore -= 1;
		if (!fieldxy(x, H - 1))
			pscore -= 1;
	}
	score += pscore;
	dprintf(5, " %8.2f", pscore);

	/* "smooth"ness */
	pscore = 0.0;
	for(x = 0; x < W-1; ++x) {
		/* too bumpy is bad */
		dist = abs(height[x+1] - height[x]);
#if 0
		if (dist >= 1)
			--dist;
#endif
		/* prefer up at edged, and lower in the middle */
		if ((x <= W / 3 && dist > 0) || (x >= W * 2 / 3 && (dist < 0))) {
			score -= dist;
		} else {
			score -= dist / 2.0;
		}
#if 0
		score -= abs(height[x] - height[x+1]);
		score -= 5 - 25 / (5 + abs(height[x] - height[x+1]));
#endif
	}
	score += pscore;
	dprintf(5, " %8.2f", pscore);

	/* horizontal completeness */
	pscore = 0.0;
	for(y = 0; y < H - 1; ++y) {
		c = 0;
		for(x = 0; x < W; ++x) {
			if (fieldxy(x, y)) {
				++c;
			}
		}
		pscore -= 3.0 * (1 - c / ((double) H));
	}
	score += pscore;
	dprintf(5, " %8.2f", pscore);

	dprintf(5, " %8.2f\n", score);
	return score;
}

#define rowbonus(rows)	((rows)*(rows)*(rows))

static int findmove(char *field, struct blockstruct **blocklist, int initialx, int initialy, int *movelist, double *returnscore) {
	/* Finds the best way to play the blocks in blocklist
	 * returns 0 for OK, > 0 for error
	 * returns in *movelist the backwards list of moves to play block[0],
	 * returns in returnscore the potential score achieved by playing
	 *   movelist */
	struct blockstruct *block /* = blocks + blocklist[0] */;
	int x, y, bestx = NOMOVE, besty = NOMOVE, rowscleared, rotation;
#define OFFSET	2
	int foundmove = 1, movelen, i;
	int rplaystatus[4][H*W] = {{0}}, *playstatus;
#define UNKNOWN	0	/* not checked yet */
#define INVALID	1	/* invalid position to play piece */
#define PARTIAL	2	/* OK, but nothing to rest piece on, ie not final */
#define FINAL	3	/* OK, could be final place to play piece */
	int rplaymove[4][H*W] = {{0}}, *playmove;
		/* contains the move which will end up in that position */
	double rplayscore[4][H*W], *playscore;
	double bestscore = -1e30, score;
	char fieldcopy[W*H];
	int bestrotation = -1;

	/* get field and block */
	if (!field) {
		dprintf(1, "No field - what are you trying to do to me?\n");
		return 2;
	}
	if (blocklist[0] == blocks) {
		dprintf(1, "No block, skipping.\n");
		return 1;
	}
	if (debug >= 4) {
		printf("findmove() called with:\n");
		printfield(field, NULL, blocklist, initialx, initialy);
	}

	/* block rotation loop */
	for(rotation = 0; rotation < blocklist[0]->numrot; ++rotation) {
		block = blocklist[0]->rotlist[rotation];
		dprintf(6, "findmove() rotation loop, rot=%i\n", rotation);
		foundmove = 0;
		playstatus = rplaystatus[rotation] + OFFSET;
		playmove = rplaymove[rotation] + OFFSET;
		/* OFFSET (+2) adjusts for the fact that blocks can actually be
		 * played at x=-2, and never more than H-3 */
		/* where piece appears */
		memcpy(fieldcopy, field, H * W);
		rowscleared = playblock(fieldcopy, block, initialx, initialy);
		if (rowscleared < -10) {
			playstatus[initialx + initialy * W] = INVALID;
			/* can't play here, or even rotate further */
			break;
		} else if (rowscleared == -1) {
			playstatus[initialx + initialy * W] = PARTIAL;
			playmove[initialx + initialy * W] = TC_MOVE_DOWN;
		} else {
			assert(rowscleared >= 0);
			playstatus[initialx + initialy * W] = FINAL;
			playmove[initialx + initialy * W] = TC_MOVE_DOWN;
		}
		for(y = initialy; y < H - block->height; ++y) {
			dprintf(6, "findmove() y loop, y=%i\n", y);
			dprintf(6, "y = %i\n", y);
			/* try moving block down */
			if (y > initialy) { /* skip first row */
				dprintf(6, "Considering move down.\n");
				for(x = - block->left; x < W - block->right; ++x) {
					assert(playstatus[x+y*W] == UNKNOWN);
					switch(playstatus[x+(y-1)*W]) {
						case UNKNOWN:
						case INVALID:
							/* can't move down from either status */
							continue;
						case FINAL:
							playstatus[x+y*W] = INVALID;
							continue;
						case PARTIAL: /* check this */
							break;
						default:
							abort(); /* shouldn't get here */
					}
					memcpy(fieldcopy, field, H * W);
					rowscleared = playblock(fieldcopy, block, x, y);
					if (rowscleared < -10) {
						playstatus[x+y*W] = INVALID;
					} else if (rowscleared == -1) {
						playstatus[x+y*W] = PARTIAL;
						playmove[x+y*W] = TC_MOVE_DOWN;
						foundmove = 1;
					} else {
						assert(rowscleared >= 0);
						playstatus[x+y*W] = FINAL;
						playmove[x+y*W] = TC_MOVE_DOWN;
					}
				}
			}
			/* try moving block right */
			dprintf(6, "Considering move right.\n");
			for(x = 1 - block->left; x < W - block->right; ++x) {
				if (playstatus[x+y*W]) {
					/* no point in moving to right - we
					 * already have checked that
					 * position */
					continue;
				}
				switch (playstatus[x-1+y*W]) {
					case UNKNOWN:
					case INVALID:
						/* no position to move from */
						continue;
				}
				memcpy(fieldcopy, field, H * W);
				rowscleared = playblock(fieldcopy, block, x, y);
				if (rowscleared < -10) {
					playstatus[x+y*W] = INVALID;
				} else if (rowscleared == -1) {
					playstatus[x+y*W] = PARTIAL;
					playmove[x+y*W] = TC_MOVE_RIGHT;
					foundmove = 1;
				} else {
					assert(rowscleared >= 0);
					playstatus[x+y*W] = FINAL;
					playmove[x+y*W] = TC_MOVE_RIGHT;
				}
			}
			/* try moving block left */
			dprintf(6, "Considering move left.\n");
			for(x = W - 1 - block->right; x >= - block->left; --x) {
				if (playstatus[x+y*W]) {
					/* no point in moving left - we
					 * already have checked that
					 * position */
					continue;
				}
				switch (playstatus[x+1+y*W]) {
					case UNKNOWN:
					case INVALID:
						/* no position to move from */
						continue;
				}
				memcpy(fieldcopy, field, H * W);
				rowscleared = playblock(fieldcopy, block, x, y);
				if (rowscleared < -10) {
					playstatus[x+y*W] = INVALID;
				} else if (rowscleared == -1) {
					playstatus[x+y*W] = PARTIAL;
					playmove[x+y*W] = TC_MOVE_LEFT;
					foundmove = 1;
				} else {
					assert(rowscleared >= 0);
					playstatus[x+y*W] = FINAL;
					playmove[x+y*W] = TC_MOVE_LEFT;
				}
			}
		}
	}
	dprintf(6, "Finished moving, calculating scores...\n");
	/* calculate the score for all FINAL positions */
	for(rotation = 0; rotation < blocklist[0]->numrot; ++rotation) {
		block = blocklist[0]->rotlist[rotation];
		playstatus = rplaystatus[rotation] + OFFSET;
		playmove = rplaymove[rotation] + OFFSET;
		playscore = rplayscore[rotation] + OFFSET;
		if (debug >= 5) { /* print playstatus and playmove arrarys */
			printf("rotation %i, block %i, playstatus:\n", rotation, block->index);
			for(y = 0; y < H; ++y) {
				printf("%2i:", y);
				for(x = - block->left; x < W - block->right; ++x) {
					printf(" %2i", playstatus[x+y*W]);
				}
				printf("\n");
			}
			printf("playmove:\n");
			for(y = 0; y < H; ++y) {
				printf("%2i:", y);
				for(x = - block->left; x < W - block->right; ++x) {
					printf(" %2i", playmove[x+y*W]);
				}
				printf("\n");
			}
		}
		PRINTSCOREKEY;
		for(y = 0; y < H; ++y) {
			for(x = - block->left; x < W - block->right; ++x) {
				if (playstatus[x+y*W] == FINAL) {
					memcpy(fieldcopy, field, H * W);
					rowscleared = playblock(fieldcopy, block, x, y);
					if (blocklist[1] == blocks) {
						/* last block in list to play */
						dprintf(5, "(%2i, %2i) ", x, y);
						score = calcscore(fieldcopy) + rowbonus(rowscleared);
					} else {
						/* recusivly call for following block(s) */
						dprintf(4, "Recursive having played %s at (%i, %i)\n", block->name, x, y);
						score = 0.0;
						if (findmove(fieldcopy, blocklist + 1, NEWX, NEWY, NULL, &score) == 0) {
							score = (3 * score + calcscore(fieldcopy)) / 4;
							score += rowbonus(rowscleared);
						} else {
							score = -1e30;
						}
					}
					if (score > bestscore) {
						bestscore = score;
						bestrotation = rotation;
						bestx = x;
						besty = y;
					}
					playscore[x+y*W] = score;
				} else {
					playscore[x+y*W] = -1e30;
				}
			}
		}
		if (debug >= 4) {
			dprintf(4, "scores for playing %s\n", block->name);
			for(y = 0; y < H; ++y) {
				dprintf(4, "%2i:", y);
				for(x = - block->left; x < W - block->right; ++x) {
					if (playscore[x+y*W] < -1e25) {
						dprintf(4, " %5s", "-");
					} else {
						dprintf(4, " %5.1f", playscore[x+y*W]);
					}
				}
				dprintf(4, "\n");
			}
		}
	}
	if (bestx == NOMOVE) {
		assert(besty == NOMOVE);
		dprintf(1, "No valid move found.\n");
		return 3;
	}
	dprintf(3, "Best move: score=%.1f rot=%i, (%i, %i)\n",
		bestscore, bestrotation, bestx, besty);
	*returnscore = bestscore;
	if (movelist) {
		/* generate the list of moves (backwards) */
		playstatus = rplaystatus[bestrotation] + OFFSET;
		playmove = rplaymove[bestrotation] + OFFSET;
		x = bestx;
		y = besty;
		dprintf(4, "Printing moves backwards:\n");
		for(movelen = 0; y != initialy || x != initialx; ++movelen) {
			if (debug >= 4) {
				printf("Moves: (%i, %i), %i ", x, y, playmove[x+y*W]);
				printmove(playmove[x+y*W]);
				printf(".\n");
			}
			assert(y >= 0 && y < H);
			assert(x >= - OFFSET && x < W - OFFSET);
			assert(playstatus[x+y*W]);
			switch (movelist[movelen] = playmove[x+y*W]) {
				case TC_MOVE_LEFT:
					++x;
					break;
				case TC_MOVE_RIGHT:
					--x;
					break;
				case TC_MOVE_DOWN:
					--y;
					break;
				default:
					/* all moves should have been listed above */
					abort();
			}
		}
		for(rotation = bestrotation; rotation; --rotation) {
			movelist[movelen++] = TC_ROTATE_ANTICLOCKWISE;
		}
		movelist[movelen] = 0; /* end of move list */
		/* optimse multiple TC_MOVE_DOWNs at end to a single
		 * TC_MOVE_DROP */ 
		for(i = 0; movelist[i] == TC_MOVE_DOWN; ++i)
			; /* find end move move down */
		if (i > 0) {
			memmove(movelist+1, movelist+i, (movelen - i + 1) * sizeof(*movelist));
			movelist[0] = TC_MOVE_DROP;
			dprintf(4, "Optimised %i MOVE_DOWNs into a MOVE_DROP.\n", i);
		}
	}
	if (debug >= 4) {
		printf("End findmove() for blocks:");
		for(i = 0; blocklist[i] != blocks; ++i) {
			printf(" %s", blocklist[i]->name);
		}
		printf("\n");
	}
	return 0;
}

static int playgame(void) {
	/* find the best way to play block in the field and then play it
	 * Returns:
	 *  1  won
	 *  2  lost
	 *  3  no field (shouldn't happen)
	 */
	char field[H*W], fieldcheck[H*W],
		lastfield[H*W] = {3, 2, 1, 4}; /* random stuff to be different */
	int status;
	int blockcheck;
	struct blockstruct *lastblock = NULL;
	int xpos, ypos, xpos2, ypos2, i, movelist[H*W];
	double score;
	int blocklistint[3] = {0};
	struct blockstruct *blocklist[3] = {blocks, blocks, blocks};
	char specials[GetMaxSpecials(gdata)];
	struct timeval t1, t2;

	/* get field and block */
	while(1) {
		if (flush)
			fflush(stdout);
		status = myGetCurrentBlock(blocklistint, &xpos, &ypos);
		myGetNextBlock(blocklistint+1);
		blocklist[0] = blocks + blocklistint[0];
		blocklist[1] = blocks + blocklistint[1];
		assert(blocklistint[2] == 0);
		assert(blocklist[2] == blocks);
		if (status == 0) {
			/* normal game in progress */
		} else if (status == 2) {
			dprintf(1, "I won :)\n");
			return 1;
		} else if (status == 1) {
			dprintf(1, "I died :(\n");
			return 2;
		} else {
			dprintf(1, "What does GetNextBlock()=%i mean?\n", status);
		}
		if (!myGetFieldP(GetOwnPlayerNumber(gdata), field)) {
			dprintf(1, "No field. I'm gone :(\n");
			continue;
			return 3;
		}
	 	myGetSpecials(specials);
		dprintf(3, "Working out what to do with block %i %s at (%i, %i).\n",
			blocklist[0]->index, blocklist[0]->name, xpos, ypos);
		if (debug >= 2) {
			if (!fieldsame(field, lastfield) || blocklist[0] != lastblock) {
				printf("\nplaygame(): Considering new field:\n");
				printfield(field, specials, blocklist, xpos, ypos);
				printf("\n");
				memcpy(lastfield, field, H * W);
				lastblock = blocklist[0];
			}
		}
		gettimeofday(&t1, NULL);
		if (blocklist[0] == blocks) {
			/* check it is possable to play the next block */
			i = findmove(field, blocklist+1, NEWX, NEWY, NULL, &score);
		} else {
			if (singlelookahead) {
				blocklist[1] = blocks;
			}
			i = findmove(field, blocklist, xpos, ypos, movelist, &score);
		}
		/* find best way to play block */
		gettimeofday(&t2, NULL);
		if (debug >= 3)
			printf("findmove() took %i usec\n", timediff(&t1, &t2));
		if (i != 0 || blocklist[0] == blocks) {
			if (i > 0) {
				dprintf(1, "Failed to find a move, trying to play special.\n");
				playspecial(1);
			} else {
				if (playspecial(0) < 0) {
					minisleep();
				}
			}
			continue;
		}
		if (!GetFieldP(GetOwnPlayerNumber(gdata), fieldcheck)) {
			/* did we die ? */
			dprintf(2, "Field dissapeared!\n");
			continue;
		}
		if (!fieldsame(field, fieldcheck)) {
			dprintf(2, "Field changed.\n");
			if (debug >= 3) {
				printf("Previous field:\n");
				printfield(field, NULL, NULL, -1, -1);
				printf("New field:\n");
				printfield(fieldcheck, NULL, NULL, -1, -1);
			}
			continue;
		}
		status = GetCurrentBlock(&blockcheck, &xpos2, &ypos2);
		if (blocklist[0]->index != blockcheck) {
			dprintf(1, "Block changed from %s to %s.\n", blocklist[0]->name, blocks[blockcheck].name);
			continue;
		}
		if (xpos != xpos2 || ypos != ypos2) {
			dprintf(1, "Block moved from (%i, %i) to (%i, %i).\n", xpos, ypos, xpos2, ypos2);
			continue;
		}
		for(i = 0; movelist[i]; ++i)
			; /* find end of move list */
		dprintf(3, "Need to make %i moves...\nMoving:", i);
		for(--i ; i >= 0; --i) {
			if (debug >= 3) {
				printf(" ");
				printmove(movelist[i]);
			}
			if (MoveBlock(movelist[i])) {
				if (debug >= 1) {
					printf(" Move ");
					printmove(movelist[i]);
					printf(" failed!\n");
				}
				break;
			}
		}
		dprintf(3, "\n");
		minisleep();
	}
}

static struct blockstruct *readblock(FILE *fh) {
	/* used by loadfield() */
	char buffer[80], *r;
	int i;

	r = fgets(buffer, sizeof(buffer), fh);
	if (!r) {
		printf("EOF while looking for block.\n");
		return blocks;	/* null block */
	}
	for(i = strlen(buffer) - 1; i >= 0; ++i) {
		if (isspace(buffer[i])) {
			buffer[i] = 0;
		} else {
			break;
		}
	}
	printf("block: \"%s\"\n", buffer);
	for(i = 1; i < NUMBLOCKS; ++i) {
		if (!strcmp(buffer, blocks[i].name) || atoi(buffer) == i) {
			return blocks + i;
		}
	}
	printf("Unknown block: \"%s\"\n", buffer);
	return blocks;	/* null block */
}

static int loadfield(char *fieldfilename) {
	/* load a field and analyse it, file format is simliar to output
	 * from printfield() */
	char field[W*H] = {0};
	char buffer[W+80];
	char *r, *c;
	FILE *fh;
	int i, x, y;
	struct blockstruct *blocklist[3] = {blocks, blocks, blocks};
	int movelist[H*W] = {0}, *movep;
	struct blockstruct *block;
	double score = -1e30;

	fh = fopen(fieldfilename, "r");
	if (! fh) {
		printf("Error opening \"%s\".\n%s\n", fieldfilename, strerror(errno));
		return 1;
	}
	for(i = 0; i < H; ++i) {
		r = fgets(buffer, sizeof(buffer), fh);
		if (! r) {
			printf("Unexpected EOF while reading \"%s\".\n", fieldfilename);
			return 1;
		}
		strncpy(field + i * W, buffer + 3, W);
	}
	for(i = 0; i < H * W; ++i) {
		c = field + i;
		if (*c == '.') {
			*c = 0;
		} else if (*c >= 'A' && *c <= 'Z') {
			*c -= 'A' - 1;
		} else if (*c >= '0' && *c <= '9') {
			*c -= '0' - TC_SPECIAL_FIRST_SPECIAL;
		} else {
			printf("Unexpected character in field: 0x%x '%c'\n", *c, *c);
		}
	}
	blocklist[0] = readblock(fh);
	if (! singlelookahead) {
		blocklist[1] = readblock(fh);
	}
	printfield(field, NULL, blocklist, NEWX, NEWY);
	i = findmove(field, blocklist, NEWX, NEWY, movelist, &score);
	printf("findmove() returned %i, score %f\n", i, score);
	for(i = 0; movelist[i]; ++i)
		; /* find end of move list */
	dprintf(3, "Printing %i moves...\n", i);
	for(--i ; i >= 0; --i) {
		printmove(movelist[i]);
		printf("\n");
	}
	dprintf(3, "Executing moves...\n");
	x = NEWX;
	y = NEWY;
	block = blocklist[0];
	for(movep = movelist; *movep; ++movep) {
		switch (*movep) {
			case TC_MOVE_LEFT:
				--x;
				break;
			case TC_MOVE_RIGHT:
				++x;
				break;
			case TC_MOVE_DOWN:
				++y;
				break;
			case TC_MOVE_DROP:
				dprintf(1, "Drop not implemented here.\n");
				break;
			case TC_ROTATE_ANTICLOCKWISE:
				block = block->rotlist[1];
				break;
			default:
				dprintf(1, "Unknown block move: %i\n", *movep);
				break;
		}
	}
	playblock(field, block, x, y);
	printfield(field, NULL, blocklist, x, y);
	return 0;
}

int main(int argc, char *argv[]) {
	char *user = "kimo", *hostname = "localhost", *fieldfilename = NULL;
	int port = 9467, c;

	while(1) {
		c = getopt(argc, argv, "Hvd::fsh:p:u:l:");
		if (c == -1)
			break;
		switch(c) {
		case 'H':
			puts(helptext);
			return 1;
		case 'v':
			puts(version);
			return 1;
		case 'd':
			if (optarg) {
				debug = atoi(optarg);
			} else {
				++debug;
			}
			break;
		case 'f':
			++flush;
			break;
		case 'h':
			hostname = optarg;
			break;
		case 'u':
			user = optarg;
			break;
		case 'p':
			port = atoi(optarg);
			break;
		case 's':
			singlelookahead = !singlelookahead;
			break;
		case 'l':
			fieldfilename = optarg;
			break;
		default:
			printf("Use -H for help.\n");
		}
	}
#if 0
	if (flush) {
		setbuf(stdout, NULL);
	}
#endif
	dprintf(1, "Debug level set to %i.\n", debug);
	blockinit();
	if (debug >= 3)
		printallblocks();
	if (fieldfilename) {
		loadfield(fieldfilename);
		return 0;
	}
	dprintf(2, "Connecting to server as %s\n", user);
	if (connectToServer(hostname, port, user, "", "") < 0) {
		dprintf(1, "error: %s\n", tc_error_string);
		exit(1);
	} else {
		dprintf(2, "Connected to server.\n");
	}
	while(1) {
		dprintf(1, "\nWaiting for game start\n");
		if (flush)
			fflush(stdout);
		if (Start(&gdata) != 0) {
			dprintf(1, "Game start failed\n");
			return 1;
		} else {
			dprintf(1, "Game starting\n");
		}
		numplayers = GetNumPlayers(gdata);
		if (debug >= 3) {
			printgamedetails();
		}
		playgame();
		dprintf(1, "\nGame has finished.\n");
	}
	return 0;
}
