/*
 * block, a tetribot AI
 * Copyright (C) 2004 Sean Burford <sean.burford@adelaide.edu.au>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <termios.h>
#include <string.h>

#include <tclient.h>
#include "block.h"
#include "ai-support.h"

#include "dropper.h"

struct fieldstate {
	unsigned int valid;
	unsigned int friend;
	unsigned int height;
	unsigned int specials;
};

int FillFieldStates(struct fieldstate *others, int pcount, struct TC_GameData *gdata)
{
	int me = GetOwnPlayerNumber(gdata);
	int i, overallhighest=0;

	for(i=0;i<pcount;i++)
	{
		char *f;
		others[i].valid=0;
		if((f = GetField(i)) != NULL)
		{
			int x, y, height=-1;

			others[i].valid=1;
			others[i].specials=0;
			others[i].friend = (i == me)?1:0;

			for(y=0;y<TC_FIELD_HEIGHT;y++)
			{
				for(x=0;x<TC_FIELD_WIDTH;x++)
				{
					if(f[(y*TC_FIELD_WIDTH)+x])
					{
						int offset = (y * TC_FIELD_WIDTH)+x;
						if(height == -1)
						{
							height = TC_FIELD_HEIGHT-y;
							if(height > overallhighest)
								overallhighest = height;
						}
						if((f[offset] >= TC_SPECIAL_FIRST_SPECIAL) &&
						   (f[offset] <= TC_SPECIAL_LAST_SPECIAL))
						{
							others[i].specials++;
							// Take special notice of special specials
							if((f[offset] == TC_SPECIAL_NUKE) ||
							   (f[offset] == TC_SPECIAL_SWITCH_FIELDS))
								others[i].specials += 10;
						}
					}
				}
			}
			others[i].height = height;
			free(f);
		}
	}
	return(overallhighest);
}
void ApplySpecial(struct TC_GameData *gdata, unsigned char sp, int id)
{
	char specials[50];

	GetSpecials((char *)&specials);
	while(specials[0] && (specials[0] != sp))
	{
		DiscardSpecial();
		GetSpecials((char *)&specials);
	}
	if(specials[0] == sp)
		UseSpecial(id);
}

int EvaluateSpecials(struct TC_GameData *gdata, struct state *game, int mode)
{
	char specials[50];
	struct fieldstate others[10];
	int i;
	int pcount = GetNumPlayers(gdata);
	int me = GetOwnPlayerNumber(gdata);

	GetSpecials((char *)&specials);

	if(specials[0])
	{
		int highest;
		int done=0;

		highest = FillFieldStates(others, pcount, gdata);

		i=0;
		while(specials[i] && !done)
		{
			switch(specials[i])
			{
			// Specials to apply to my best specials competitor
			case TC_SPECIAL_CLEAR_SPECIALS:
				{
					int j;
					int mostspecials=0, buddy=-1;
					for(j=0;j<pcount;j++)
						if((others[j].valid) &&
						   (!others[j].friend) &&
						   (others[j].specials > mostspecials))
						{
							buddy=j;
							mostspecials=others[j].specials;
						}
					if((buddy != me) && (buddy != -1))
						ApplySpecial(gdata, specials[i], buddy);
				}
				break;
			// Specials to apply to my worst height competitor
			case TC_SPECIAL_BLOCK_QUAKE:
			case TC_SPECIAL_ADD_LINE:
				{
					int j;
					int highest=0, buddy=-1;
					for(j=0;j<pcount;j++)
						if((others[j].valid) &&
						   (!others[j].friend) &&
						   (others[j].height > highest))
						{
							buddy=j;
							highest = others[j].height;
						}
					if((buddy != me) && (buddy != -1) && (highest > TC_FIELD_HEIGHT / 3))
						ApplySpecial(gdata, specials[i], buddy);
				}
				break;
			// Specials to apply to my best height competitor
			case TC_SPECIAL_BLOCK_BOMB:
			case TC_SPECIAL_CLEAR_RANDOM:
				{
					if(mode == MODE_CLEAR)
					{
						int j;
						int lowest=TC_FIELD_HEIGHT, buddy=-1;
						for(j=0;j<pcount;j++)
							if((others[j].valid) &&
							   (!others[j].friend) &&
							   (others[j].height < lowest))
							{
								buddy=j;
								lowest = others[j].height;
							}
						if((buddy != me) && (buddy != -1))
							ApplySpecial(gdata, specials[i], buddy);
					}
				}
				break;
			// Specials to apply to me if my field gets too high
			case TC_SPECIAL_CLEAR_LINE:
				{
					if((mode == MODE_CLEAR) &&
					   (others[me].height > (TC_FIELD_HEIGHT/2)))
						ApplySpecial(gdata, specials[i], me);
				}
				break;
			// Specials to apply to me if my field gets critical
			case TC_SPECIAL_BLOCK_GRAVITY:
			case TC_SPECIAL_NUKE:
				{
					if((mode == MODE_CLEAR) &&
					   (others[me].height >= (TC_FIELD_HEIGHT - 5)))
						ApplySpecial(gdata, specials[i], me);
					done=1;
				}
				break;
			// Nasty
			case TC_SPECIAL_SWITCH_FIELDS:
				{
					if(others[me].height == highest)
						mode = MODE_FILL;
					done=1;
				}
				break;
			}
			i++;
		}

		//for(i=0;i<pcount;i++)
			//if(others[i].valid)
				//printf("%d is a %s of height %d with %d specials\n",
					//i,
					//(others[i].friend)?"friend":"foe",
					//others[i].height,
					//others[i].specials);
			//else
				//printf("%d is dead\n", i);
	}
	if((mode == MODE_FILL) && (others[me].height > (TC_FIELD_HEIGHT-3)))
	{
		int lowest, lowestheight=TC_FIELD_HEIGHT;
		for(i=0;i<pcount;i++)
		{
			if((others[i].valid) && (!others[i].friend) && (others[i].height < lowestheight))
			{
				lowest=i;
				lowestheight=others[i].height;
			}
		}
		// Check, just in case...
		if(lowest != me)
		{
			ApplySpecial(gdata, TC_SPECIAL_SWITCH_FIELDS, lowest);
			mode = MODE_CLEAR;
		}
	}
	return(mode);
}

void DisplayState(struct state *game, struct block *pos)
{
	int y;

	printf("Field:\n");
	for(y=0;y<TC_FIELD_HEIGHT+1;y++)
	{
		unsigned int bit=0x80000000;
		while(bit)
		{
			char c;

			c = (bit & game->field[y])?'X':' ';

			if((game->curr.type != 0) && 
			   (y >= game->curr.y) &&
			   (y <= game->curr.y + 3))
			{
				int py = y - game->curr.y;

				if(bit & game->curr.mask[py])
					c = (c == ' ')?'O':'%';
			}
			if(pos && (pos->type != 0) && 
			   (y >= pos->y) && (y <= pos->y + 3))
			{
				int py = y - pos->y;

				if(bit & pos->mask[py])
					c = '=';
			}
			printf("%c", c);

			bit >>= 1;
		}
		if(y == 0)
			printf("\tNext: %d", game->pending.type);
		if((y == game->curr.y) && (game->curr.type != 0))
			printf("\tCurr: %d (%d,%d)", game->curr.type, game->curr.x, game->curr.y);
		printf("\n");
	}
}

void AddFieldToState(struct state *game, char *field)
{
	int x, y;

	for(y=0;y<TC_FIELD_HEIGHT;y++)
	{
		unsigned int bit = 0x08000000;
		game->field[y]   = 0xFFFFFFFF;

		for(x=0;x<TC_FIELD_WIDTH;x++)
		{
			game->field[y] ^= bit*(((field[(y*TC_FIELD_WIDTH)+x])?0:1));
			bit >>= 1;
		}
	}
	game->field[y]   = 0xFFFFFFFF;
}

int UpdateState(struct state *game)
{
	char *field;
	int type, x, y;

	// Add the play field to the game state
	if((field = GetField(game->playerno)) == NULL)
		return(1);
	AddFieldToState(game, field);
	free(field);

	// Add the current block to the game state
	GetCurrentBlock(&type, &x, &y);
	UpdateBlock(&(game->curr), type, x, y, 0);

	// Add the pending block to the game state
	GetNextBlock(&type);
	UpdateBlock(&(game->pending), type, 0, 4, 0);

	return(0);
}

int main(int argc, char *argv[])
{
	char *username = "block";
	char *host = "127.0.0.1";
	struct state game;
	struct TC_GameData *gdata;
	int mode=MODE_CLEAR;

	if (argc>1)
	{
		if(!strcmp(argv[1], "-h"))
			host=strdup(argv[2]);
		else
			username = argv[1];
	}

	if (connectToServer(host, 9467, username, username, "")<0)
	{
		printf("error: %s\n", tc_error_string);
		exit(1);
	}
	else
	{
		printf("Successfully connected\n");
	}

	while(1)
	{
		printf("Waiting for game start\n");
		if ((Start(&gdata) != 0) || (gdata == NULL))
		{
			printf("Game start failed\n");
			printf("Check we have a unique player name\n");
			exit(0);
		}

		game.playerno = GetOwnPlayerNumber(gdata);

		while(UpdateState(&game) == 0)
		{
			struct block t;
			if(game.curr.type != 0)
			{
				if(DropStrategy(game.field, &(game.curr), &(game.pending), &t, 0, mode) != -20000)
					DropStrategyMove(&(game.curr), &t);
			}
			//DisplayState(&game, &t);

			mode = EvaluateSpecials(gdata, &game, mode);
		}
	}

	exit(0);
}
