/*
 * Allocate and dirty memory.
 *
 * Usage: usemem Nmegs
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/mman.h>

#define ALIGN(x,a) (((x)+(a)-1)&~((a)-1))

char *ourname;
int pagesize;
unsigned long megs = 16;
unsigned long i;
unsigned long *p;
int sleep_secs = 0;
int reps = 1;
int rep;
int do_mlock = 0;
int do_getchar = 0;
int opt_randomise = 0;
int opt_readonly;
int opt_repeat;
int opt_malloc;
int nthreads = 0;
int quiet = 0;
int msync_mode;
int c;
char *filename = "/dev/zero";
int map_shared = MAP_PRIVATE;
int fd;
int parent;


void usage(void)
{
	fprintf(stderr,
	"Usage: %s [-aAgMoqRS] [-m megs] [-n threads] [-s sleep_secs] "
				"[-r reps] [-f file]\n"
	"    -a:           obtain memory from malloc()\n"
	"    -g:           wait for <enter> before quitting\n"
	"    -M:           mlock() the memory\n"
	"    -N:           repeat indefinitely\n"
	"    -o:           readonly access\n"
	"    -q:           operate quietly\n"
	"    -R:           random access pattern\n"
	"    -A:           msync(MS_ASYNC) the memory\n"
	"    -S:           msync(MS_SYNC) the memory\n"
	,		ourname);
	exit(1);
}

unsigned long randomise(unsigned long max)
{
	return (unsigned long)((double)max * rand() / (RAND_MAX + 1.0));
}

void do_child(void)
{
	volatile long d;

	if (opt_randomise)
		srand(time(0) ^ getpid());

	if (opt_malloc) {
		p = malloc(megs);
		if (p == NULL) {
			perror("malloc");
			exit(1);
		}
	} else {
		p = mmap(NULL, megs, PROT_READ|PROT_WRITE, map_shared, fd, 0);
		if (p == MAP_FAILED) {
			fprintf(stderr, "%s: mmap failed: %s\n",
				ourname, strerror(errno));
			exit(1);
		}
		p = (unsigned long *)ALIGN((unsigned long )p, pagesize - 1);

		lseek(fd, megs - 1, SEEK_SET);
		write(fd, "", 1);
	}

	if (do_mlock) {
		if (mlock(p, megs)) {
			perror("mlock");
			exit(1);
		}
	}
repeat:
	for (rep = 0; rep < reps; rep++) {
		unsigned long m = megs / sizeof(*p);
		unsigned long idx;

		if (rep > 0 && !quiet) {
			printf(".");
			fflush(stdout);
		}

		if (opt_randomise) {
			for (i = 0; i < m; i += pagesize / sizeof(*p)) {
				idx = randomise(m - 1);
				if (opt_readonly)
					d = p[idx];
				else
					p[idx] = idx;
			}
			continue;
		}

		for (i = 0; i < m; i += pagesize / sizeof(*p)) {
			if (opt_readonly)
				d = p[i];
			else
				p[i] = i;
		}
		if (reps > 1 && !opt_readonly) {
			for (i = 0; i < m; i += pagesize / sizeof(*p)) {
				if (p[i] != i) {
					fprintf(stderr, "data wrong at offset "
						"0x%lx. Expected 0x%08lx, got "
						"0x%08lx\n",
						i * sizeof(*p), i, p[i]);
					exit(1);
				}
			}
		}
		if (msync_mode)
			msync(p, megs, msync_mode);
	}
	if (sleep_secs) {
		fflush(stdout);
		sleep(sleep_secs);
		printf(" exit");
	}
	if (do_getchar) {
		for ( ; ; ) {
			switch (getchar()) {
			case 'u':
				printf("munlocking\n");
				munlock(p, megs);
				break;
			case 'l':
				printf("munlocking\n");
				mlock(p, megs);
				break;
			}
		}
	}
	if (opt_repeat)
		goto repeat;
}

int main(int argc, char *argv[])
{
	ourname = argv[0];
	pagesize = getpagesize();
	if (argc < 2)
		usage();

	while ((c = getopt(argc, argv, "aAf:gqoNRMm:n:s:Sr:")) != -1) {
		switch (c) {
		case 'a':
			opt_malloc++;
			break;
		case 'A':
			msync_mode = MS_ASYNC;
			break;
		case 'f':
			filename = optarg;
			map_shared = MAP_SHARED;
			break;
		case 'g':
			do_getchar = 1;
			break;
		case 'm':
			megs = strtol(optarg, NULL, 10);
			break;
		case 'n':
			nthreads = strtol(optarg, NULL, 10);
			break;
		case 'M':
			do_mlock = 1;
			break;
		case 'q':
			quiet = 1;
			break;
		case 's':
			sleep_secs = strtol(optarg, NULL, 10);
			break;
		case 'r':
			reps = strtol(optarg, NULL, 10);
			break;
		case 'R':
			opt_randomise++;
			break;
		case 'S':
			msync_mode = MS_SYNC;
			break;
		case 'N':
			opt_repeat++;
			break;
		case 'o':
			opt_readonly++;
			break;
		default:
			usage();
		}
	}

	if (optind != argc)
		usage();

	fd = open(filename, O_RDWR|O_CREAT, 0666);
	if (fd < 0) {
		fprintf(stderr, "%s: failed to open `%s': %s\n",
			ourname, filename, strerror(errno));
		exit(1);
	}

	megs *= 1024 * 1024;

	parent = 0;
	if (nthreads) {
		for (i = 0; i < nthreads; i++) {
			if (fork() == 0) {
				do_child();
				exit(0);
			}
		}

		for (i = 0; i < nthreads; i++) {
			int status;

			if (wait3(&status, 0, 0) < 0) {
				if( errno != EINTR ) {
					perror("wait3");
					exit(1);
				}
			}
		}
		printf("\n");
	} else {
		do_child();
	}
	exit(0);
}
