#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "slacker.h"

char *ourname;
int o_direct;
unsigned long align_offset;

static void usage(void)
{
	fprintf(stderr,
		"Usage: %s [-koO] [-a align] [-b blocksize] [-h file ] offset size filename\n"
		"          -a align      : O_DIRECT memory alignment (bytes)\n"
		"          -b blocksize  : set device blocksize\n"
		"          -h file       : perform IO out of hugepage file\n"
		"          -k            : offset and size are in kbytes\n"
		"          -o            : use O_DIRECT\n"
		"          -O            : file overwrite\n",
		ourname);
	exit(1);
}

static void *round_up(void *ptr, unsigned long align, unsigned long offset)
{
	unsigned long ret = (unsigned long)ptr;

	ret = ((ret + align - 1) & ~(align - 1));
	ret += offset;
	return (void *)ret;
}

static void set_blocksize(int fd, int size)
{
	struct stat stat;

	if (fstat(fd, &stat)) {
		perror("fstat");
		exit(1);
	}
	if (S_ISBLK(stat.st_mode)) {
		if (ioctl(fd, BLKBSZSET, &size) < 0) {
			perror("ioctl");
			exit(1);
		} else {
			printf("blocksize is %d\n", size);
		}
	}
}

int main(int argc, char *argv[])
{
	char *filename;
	size_t size;
	loff_t offset;
	int fd;
	int overwrite = O_CREAT|O_TRUNC;
	char *_meg;
	char *meg;
	int c;
	int i;
	int k = 0;
	int blocksize = 0;
	char *hugefile = NULL;
	size_t write_ret;

	ourname = argv[0];

	while ((c = getopt(argc, argv, "koOa:b:h:")) != -1) {
		switch (c) {
		case 'a':
			align_offset = strtol(optarg, NULL, 10);
			break;
		case 'b':
			blocksize = strtol(optarg, NULL, 10);
			break;
		case 'h':
			hugefile = optarg;
			break;
		case 'k':
			k++;
			break;
		case 'o':
			o_direct = O_DIRECT;
			break;
		case 'O':
			overwrite = 0;
			break;
		default:
			usage();
		}
	}

	if (optind == argc)
		usage();
	offset = strtol(argv[optind++], NULL, 10);

	if (optind == argc)
		usage();
	size = strtol(argv[optind++], NULL, 10);

	if (optind == argc)
		usage();
	filename = argv[optind++];
	if (optind != argc)
		usage();

	if (k) {
		offset *= 1024;
		size *= 1024;
	}

	if (hugefile) {
		int hfd = open(hugefile, O_RDWR|O_CREAT, 0644);

		if (hfd < 0) {
			perror(hugefile);
			exit(1);
		}
		_meg = mmap(NULL, 4*1024*1024, PROT_READ|PROT_WRITE,
				MAP_PRIVATE, hfd, 0);
		if (_meg == MAP_FAILED) {
			perror("mmap");
			exit(1);
		}
	} else {
		_meg = malloc(size + 10000);
		if (!_meg) {
				fprintf(stderr, "oom\n");
			exit(1);
		}
	}
	meg = round_up(_meg, 4096, align_offset);
	if (align_offset)
		printf("IO buffer is at %p\n", meg);

	fd = open(filename,
		o_direct|O_RDWR|overwrite, 0666);
	if (fd < 0) {
		fprintf(stderr, "%s: cannot create %s: %s\n",
			argv[0], filename, strerror(errno));
		exit(1);
	}
	if (blocksize)
		set_blocksize(fd, blocksize);

	for (i = 0; i < size; i += sizeof(long)) {
		long *p = (long *)(meg + i);

		*p = i + offset;
	}

	write_ret = pwrite(fd, meg, size, offset);

	if (write_ret != size) {
		if ((ssize_t)write_ret < 0)
			perror("pwrite");
		else
			fprintf(stderr, "write(%zd) returned %zd\n",
					size, write_ret);
		exit(1);
	}

	close(fd);
	exit(0);
}
