/* rmmod.c: remove a module from the kernel.
    Copyright (C) 2001  Rusty Russell.
    Copyright (C) 2002  Rusty Russell, IBM Corporation.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <asm/unistd.h>
#include <stdarg.h>

#include "backwards_compat.c"

static void fatal(const char *fmt, ...)
__attribute__ ((noreturn, format (printf, 1, 2)));

static void fatal(const char *fmt, ...)
{
	va_list arglist;

	fprintf(stderr, "FATAL: ");

	va_start(arglist, fmt);
	vfprintf(stderr, fmt, arglist);
	va_end(arglist);

	exit(1);
}

static void warn(const char *fmt, ...)
__attribute__ ((format (printf, 1, 2)));

static void warn(const char *fmt, ...)
{
	va_list arglist;

	fprintf(stderr, "WARNING: ");

	va_start(arglist, fmt);
	vfprintf(stderr, fmt, arglist);
	va_end(arglist);
}

/* If we can check usage without entering kernel, do so. */
static void check_usage(const char *modname)
{
	FILE *module_list;
	char line[10240], name[64], dep[64];
	unsigned long size, refs;

	module_list = fopen("/proc/modules", "r");
	if (!module_list) {
		if (errno == ENOENT) /* /proc may not be mounted. */
			return;
		perror("Opening /proc/modules");
		exit(1);
	}
	while (fgets(line, sizeof(line)-1, module_list) != NULL) {
		if (strchr(line, '\n') == NULL) {
			fprintf(stderr, "V. v. long line broke rmmod.\n");
			exit(1);
		}
		switch (sscanf(line, "%s %lu %lu %s", name,&size,&refs,dep)) {
		case 2:
			fatal("Kernel does not have unload support\n");
			exit(1);
		case 3:
			if (strcmp(name, modname) == 0) {
				if (refs != 0)
					fatal("Module is in use\n");
				fclose(module_list);
				return;
			}
			break;
		case 4:
			if (strcmp(name, modname) == 0)
				fatal("Module is in use by%s",
				     strchr(strchr(strchr(line,' ')+1,
						   ' ')+1,' '));
			break;
		default:
			fatal("Unknown format in /proc/modules: %s", line);
		}
	}
	warn("Module %s does not exist in /proc/modules\n", modname);
	fclose(module_list);
}

static void print_usage(const char *progname)
{
	fprintf(stderr,
		"Usage: %s [--wait|-f] <modulename>\n"

		"  The --wait option begins a module removal even if it is used\n"
		"  and will stop new users from accessing the module (so it\n"
		"  should eventually fall to zero)."

		"\n"
		"  The -f option forces a module unload, and may crash your\n"
		"  machine.  This requires the Forced Module Removal option\n"
		"  when the kernel was compiled.\n", progname);
	exit(1);
}

int main(int argc, char *argv[])
{
	long int ret;
	unsigned int flags = O_NONBLOCK;
	char *progname = argv[0], *name, *ext;

	try_old_version("rmmod", argv);

	if (argv[1] && strcmp(argv[1], "--wait") == 0) {
		flags &= ~O_NONBLOCK;
		argv++;
		argc--;
	}
	if (argv[1] && (strcmp(argv[1], "-f") == 0
			|| strcmp(argv[1], "--force") == 0)) {
		flags |= O_TRUNC;
		argv++;
		argc--;
	}
	if (!argv[1])
		print_usage(progname);

	/* Strip path and .o off filename to give name */
	name = strrchr(argv[1], '/');
	if (!name) name = argv[1];
	else name++;
	ext = strrchr(name, '.');
	if (ext) *ext = '\0';

	if ((flags & O_NONBLOCK) && !(flags & O_TRUNC))
		check_usage(name);

	ret = syscall(__NR_delete_module, name, flags);
	if (ret != 0)
		fatal("Error removing `%s': %s\n", name, strerror(errno));

	exit(0);
}
