#define _GNU_SOURCE
#include <limits.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fnmatch.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>
#include "stdrusty.h"
#include "ccontrol.h"

static void insert_arg(char *argv[], unsigned argc, unsigned pos, char *arg)
{
	memmove(argv+pos+1, argv+pos, (argc+1) * sizeof(char *));
	argv[pos] = arg;
}

void __verbose(const char *msg, ...)
{
	va_list arglist;
	char *str, pidstr[CHAR_SIZE(int)];

	int_to_string(pidstr, getpid());
	write(STDERR_FILENO, "ccontrol: ", strlen("ccontrol: "));
	write(STDERR_FILENO, pidstr, strlen(pidstr));
	write(STDERR_FILENO, ": ", strlen(": "));
	write(STDERR_FILENO, msg, strlen(msg));

	va_start(arglist, msg);
	while ((str = va_arg(arglist, char *)) != NULL)
		write(STDERR_FILENO, str, strlen(str));
	va_end(arglist);

	write(STDERR_FILENO, "\n", 1);
}

static bool make_target_match(char *argv[], char *targets)
{
	char *p, *target = "";
	unsigned int i;

	if (!targets)
		return false;

	if (!targets[0])
		return true;

	/* Heuristic: choose last arg which is neither option nor variable */
	for (i = 0; argv[i]; i++) {
		if (argv[i][0] == '-')
			continue;
		if (strchr(argv[i], '='))
			continue;
		target = argv[i];
	}

	for (p = strtok(targets, " \t"); p; p = strtok(NULL, " \t"))
		if (fnmatch(p, target, 0) == 0)
			return true;
	return false;
}

/* Earlier arg parsing might have revealed distcc is useless. */
static void adjust_args_and_environment(struct section sec,
					enum type type,
					char *argv[], unsigned argc)
{
	argv[0] = sec.names[type];

	switch (type) {
	case TYPE_MAKE:
		/* Ensure make calls us for children, not /usr/bin/make. */
		setenv("MAKE", "make", 1);

		/* We insert -j to make in parallel. */
		if (!make_target_match(argv, sec.no_parallel)) {
			verbose(sec, "Inserting arg -j in make!", NULL);
			insert_arg(argv, argc++, 1, "-j");
		}
		while (sec.make_add) {
			insert_arg(argv, argc++, 1, sec.make_add->arg);
			sec.make_add = sec.make_add->next;
		}
		break;
	case TYPE_LD:
		break;
	case TYPE_CC:
	case TYPE_CPLUSPLUS:
		/* ccache + distcc needs special care. */
		if (sec.distcc) {
			char *hosts;

			if (type == TYPE_CC)
				hosts = sec.distcc_hosts;
			else
				hosts = sec.distcplusplus_hosts;
			verbose(sec, "Setting DISTCC_HOSTS=", hosts, NULL);
			setenv("DISTCC_HOSTS", hosts, 1);
			if (sec.ccache) {
				verbose(sec, "Setting CCACHE_PREFIX=",
					sec.distcc, NULL);
				setenv("CCACHE_PREFIX", sec.distcc, 1);
			} else {
				verbose(sec, "Prefixing arg ",
					sec.distcc, NULL);
				insert_arg(argv, argc++, 0, sec.distcc);
			}
		}

		/* Don't set CCACHE_UNIFY: misses not so bad with distcc. */
		if (sec.ccache) {
			verbose(sec, "Prefixing arg ", sec.ccache, NULL);
			insert_arg(argv, argc++, 0, sec.ccache);
		}
	}
}

#ifndef TESTING
static void __attribute__((noreturn)) run_command(struct section sec,
						  char *argv[])
{
	verbose(sec, "Execing '", argv[0], "'", NULL);
	execv(argv[0], argv);
	fatal("failed to exec '", errno, argv[0], "'", NULL);
}
#else
static void __attribute__((noreturn)) run_command(struct section sec,
						  char *argv[])
{
	if (strcmp(getenv("FAILDISTCC") ?: "", "1") == 0)
		exit(103);

	if (getenv("DISTCC_HOSTS")) {
		printf("DISTCC_HOSTS='%s' ", getenv("DISTCC_HOSTS"));
	}

	while (argv[0]) {
		printf("%s ", argv[0]);
		argv++;
	}
	printf("\n");

	exit(atoi(getenv("EXITCODE") ?: "0"));
}
#endif

/* Fork command, wait for it, return exit status. */
static int fork_command(struct section sec, char *argv[])
{
	int status, pid = fork();
	char num[20];

	if (pid < 0)
		fatal("failed to fork", errno, NULL);
	if (pid == 0)
		run_command(sec, argv);
	int_to_string(num, pid);
	verbose(sec, "Forked ", num, NULL);
	waitpid(pid, &status, 0);
	int_to_string(num, WIFEXITED(status) ? WEXITSTATUS(status) : 255);
	verbose(sec, "Waitpid returned ", num, NULL);
	return WIFEXITED(status) ? WEXITSTATUS(status) : 255;
}

static bool file_altered(int fd, const struct stat *st)
{
	struct stat st2;

	fstat(fd, &st2);
	/* If they deleted file, ctime change not guaranteed by POSIX,
	 * but Linux, FreeBSD and Solaris all change. */
	return st2.st_ctime != st->st_ctime || st2.st_mtime != st->st_mtime;
}

int main(int orig_argc, char *orig_argv[])
{
	enum type type;
	struct section sec;
	int fd, ret, argc, nodistcc = 0;
	struct stat st;
	char configname[PATH_MAX];
	char dirname[PATH_MAX];
	char *new_argv[orig_argc + 23], **argv;
	undofn_t undo;

	/* Run low priority; people like to use apps while compiling. */
	nice(10);

	strcpy(configname, getenv("HOME"));
	strcat(configname, "/.ccontrol/default");

again_restore_args:
	/* Make room to add args. */
	argc = orig_argc;
	memcpy(new_argv, orig_argv, (argc + 1)*sizeof(argv[0]));
	argv = new_argv;

#ifdef TESTING
	if (getenv("ARGV0"))
		argv[0] = getenv("ARGV0");
#endif
	getcwd(dirname, sizeof(dirname));

	if (strends(argv[0], "ccontrol")) {
		if (argv[1] && strstarts(argv[1], "--section=")) {
			strcpy(dirname, argv[1] + strlen("--section="));
			argv++;
			argc--;
		}
		if (!argv[1]) {
			__verbose("version " VERSION, NULL);
			exit(0);
		}
		argv++;
		argc--;
	}
	type = what_am_i(argv);

again:
	/* Since we later grab an exclusive lock on this, must be writable */
	fd = open(configname, O_RDWR);

	/* This handles open failure if fd < 0. */
	sec = read_config(configname, dirname, fd);
	fstat(fd, &st);

	/* Check we really distcc if asked to. */
	if (sec.distcc) {
		if (nodistcc || type == TYPE_LD || type == TYPE_MAKE)
			sec.distcc = NULL;
		else {
			if (type == TYPE_CC && !sec.distcc_hosts)
				sec.distcc = NULL;
			else if (type == TYPE_CPLUSPLUS
				 && !sec.distcplusplus_hosts)
				sec.distcc = NULL;
			else if (!can_distcc(argv)) {
				verbose(sec, "Cannot distcc this", NULL);
				sec.distcc = NULL;
			}
		}
	}

	/* Grab lock returns undofn if it slept: restat config file.  This
	 * decreases latency when config file changed. */
	undo = grab_lock(fd, sec, type);
	if (undo && file_altered(fd, &st)) {
		close(fd);
		undo();
		goto again;
	}

	adjust_args_and_environment(sec, type, argv, argc);

	if (!sec.distcc)
		run_command(sec, argv);

	/* We have to wait, to make sure distcc actually distributes,
	 * otherwise we end up running 20 compiles locally. */
	setenv("DISTCC_FALLBACK", "0", 1);

	ret = fork_command(sec, argv);
	if (ret == 103 || ret == 116) {
		/* distcc failed to distribute: run single-thread. */
		close(fd);
		undo();
		nodistcc = 1;
#ifdef TESTING
		unsetenv("FAILDISTCC");
#endif
		goto again_restore_args;
	}
	return ret;
}
