#include "ccontrol.h"
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <fnmatch.h>
#include <fcntl.h>
#include <errno.h>
#include "stdrusty.h"

enum token_type {
	CC = TYPE_CC,
	CPLUSPLUS = TYPE_CPLUSPLUS,
	LD = TYPE_LD,
	MAKE = TYPE_MAKE,
	NONE,
	INDENT,
	NO_PARALLEL,
	PARALLEL,
	CCACHE,
	DISTCC,
	DISTCC_HOSTS,
	DISTCPLUSPLUS_HOSTS,
	LEFT_BRACKET,
	RIGHT_BRACKET,
	EQUALS,
	NUMBER,
	STRING,
	VERBOSE,
	INCLUDE,
	CPUS,
	OFF,
	ADD,
	END,
};

#include "keywords.c"

struct token
{
	enum token_type type;
	struct string string;
};

static unsigned int sremain(struct string string)
{
	return string.end - string.pos;
}

/* Length up to and including char. 0 == not found.  */
static unsigned int upto(struct string data, char c)
{
	char *end;

	end = memchr(data.pos, c, sremain(data));
	if (end)
		return end - data.pos + 1;
	return 0;
}

static void __attribute__((noreturn))
parse_error(struct token tok, const char *str)
{
	unsigned int line = 1;
	char linestr[CHAR_SIZE(line)];
	char * input_head = tok.string.start;

	/* Count what line we're on. */
	while (input_head < tok.string.pos) {
		if (*input_head == '\n')
			line++;
		input_head++;
	}
	int_to_string(linestr, line);
	fatal("parse error on line ", 0, linestr, ": expected ", str, NULL);
}

static bool swallow_whitspace(struct string *data)
{
	bool ret = false;

	while (data->pos[0] == ' ' || data->pos[0] == '\t') {
		ret = true;
		data->pos++;
		if (data->pos == data->end)
			break;
	}
	return ret;
}

static struct string get_string(struct string *data)
{
	struct string str;
	str.start = data->start;
	str.pos = data->pos;
	while (data->pos != data->end
	       && *data->pos != '='
	       && *data->pos != '['
	       && *data->pos != ']'
	       && *data->pos != ' '
	       && *data->pos != '\n'
	       && *data->pos != '\t')
		data->pos++;
	str.end = data->pos;
	return str;
}

/* Token takes one character. */
static struct token single_digit_token(struct string data,enum token_type type)
{
	return ((struct token) { type, { data.pos, data.pos+1 } });
}

static struct token peek_token(struct string data)
{
	struct token tok;
	unsigned int num;
	bool new_line = (data.pos == data.start);

	/* We might need to skip over comments. */
	while (data.pos != data.end) {
		const struct ccontrol_command *cmd;
		char *start;

		if (data.pos[0] == '\n') {
			new_line = true;
			data.pos++;
			continue;
		}

		/* Whitespace: ignored unless after newline */
		start = data.pos;
		if (swallow_whitspace(&data) && new_line) {
			if (!sremain(data) || data.pos[0] != '#') {
				tok.type = INDENT;
				tok.string.start = data.start;
				tok.string.pos = start;
				tok.string.end = data.pos;
				return tok;
			}
		}

		/* Comment?  Ignore to end of line. */
		if (data.pos[0] == '#') {
			num = upto(data, '\n');
			if (!num)
				break;
			data.pos += num;
			new_line = true;
			continue;
		}

		if (data.pos[0] == '[')
			return single_digit_token(data, LEFT_BRACKET);

		if (data.pos[0] == ']')
			return single_digit_token(data, RIGHT_BRACKET);

		if (data.pos[0] == '=')
			return single_digit_token(data, EQUALS);

		tok.string = get_string(&data);
		cmd = find_keyword(tok.string.pos, sremain(tok.string));
		if (cmd)
			tok.type = cmd->type;
		else {
			/* Number or string? */
			unsigned int i;

			for (i = 0; i < sremain(tok.string); i++)
				if (!isdigit(tok.string.pos[i]))
					break;
			if (i > 0 && i == sremain(tok.string))
				tok.type = NUMBER;
			else
				tok.type = STRING;
		}
		return tok;
	}

	tok.type = END;
	tok.string = data;
	return tok;
}

static void swallow_token(struct string *data, struct token tok)
{
	data->pos = tok.string.end;
}

static struct token get_token(struct string *data)
{
	struct token tok = peek_token(*data);
	swallow_token(data, tok);
	return tok;
}

/* Optional = <token>. */
static struct token get_value(struct string *data)
{
	struct token tok;

	tok = peek_token(*data);
	if (tok.type == EQUALS) {
		swallow_token(data, tok);
		return get_token(data);
	}
	tok.type = NONE;
	tok.string.pos = data->pos;
	tok.string.end = data->pos;
	return tok;
}

static unsigned int to_uint(struct token tok)
{
	if (tok.type != NUMBER)
		parse_error(tok, "'= some-number'");

	return atol(tok.string.pos);
}

static char *to_path(struct token tok)
{
	char *p;

	if (tok.type != STRING)
		parse_error(tok, "'= some-path'");

	p = new_array(char, sremain(tok.string) + 1);
	memcpy(p, tok.string.pos, sremain(tok.string));
	p[sremain(tok.string)] = '\0';
	return p;
}

static char *to_optional_path(struct token tok)
{
	if (tok.type == OFF)
		return NULL;

	return to_path(tok);
}

static char *to_eol(struct string *data)
{
	char *p;
	unsigned int len;

	swallow_whitspace(data);
	len = upto(*data, '\n');
	if (!len) {
		struct token tok;

		tok.type = END;
		tok.string = *data;
		parse_error(tok, "something");
	}

	/* Turn \n into \0. */
	p = malloc(len);
	memcpy(p, data->pos, len);
	p[len-1] = '\0';

	/* Leave \n so we can detect indent. */
	data->pos += len-1;

	return p;
}

static struct add *new_add(char *arg, struct add *next)
{
	struct add *add = malloc(sizeof(struct add));
	add->arg = arg;
	add->next = next;
	return add;
}

static void read_section_file(const char *configname, struct section *sec);

/* With thanks to Jospeh Heller. */
static void read_section_section(struct string *data, struct section *sec)
{
	struct token tok;

	while ((tok = peek_token(*data)).type == INDENT) {
		swallow_token(data, tok);
		tok = get_token(data);
		/* Lines are of form "x" or "x = value". */
		switch (tok.type) {
		case NO_PARALLEL:
			tok = peek_token(*data);
			if (tok.type != EQUALS) {
				/* Not "*": we strtok it later. */
				sec->no_parallel = "";
				break;
			}
			swallow_token(data, tok);
			sec->no_parallel = to_eol(data);
			break;
		case PARALLEL:
			sec->no_parallel = NULL;
			unsetenv("CCONTROL_NO_PARALLEL");
			break;
		case CPUS:
			sec->cpus = to_uint(get_value(data));
			if (!sec->cpus)
				parse_error(tok, "greater than zero");
			break;
		case CCACHE:
			sec->ccache = to_optional_path(get_value(data));
			break;
		case DISTCC:
			sec->distcc = to_optional_path(get_value(data));
			break;
		case DISTCC_HOSTS:
			tok = get_token(data);
			if (tok.type != EQUALS)
				parse_error(tok, "= some-hosts");
			if (peek_token(*data).type == OFF) {
				get_token(data);
				sec->distcc_hosts = NULL;
			} else
				sec->distcc_hosts = to_eol(data);
			if (!sec->distcplusplus_hosts_set)
				sec->distcplusplus_hosts = sec->distcc_hosts;
			break;
		case DISTCPLUSPLUS_HOSTS:
			tok = get_token(data);
			if (tok.type != EQUALS)
				parse_error(tok, "= some-hosts");
			if (peek_token(*data).type == OFF) {
				get_token(data);
				sec->distcplusplus_hosts = NULL;
			} else
				sec->distcplusplus_hosts = to_eol(data);
			sec->distcplusplus_hosts_set = true;
			break;
		case VERBOSE:
			sec->verbose = true;
			break;
		case CC:
		case CPLUSPLUS:
		case LD:
		case MAKE:
			sec->names[tok.type] = to_path(get_value(data));
			break;
		case INCLUDE:
			read_section_file(to_path(get_value(data)), sec);
			break;
		case ADD:
			tok = get_token(data);
			if (tok.type != MAKE)
				parse_error(tok, "make");
			tok = get_token(data);
			if (tok.type != EQUALS)
				parse_error(tok, "= argument");
			sec->make_add = new_add(to_eol(data), sec->make_add);
			break;
		default:
			parse_error(tok, "some instruction");
		}
	}
}

static void read_section_file(const char *configname, struct section *sec)
{
	unsigned long len;
	struct string data;

	data.start = suck_file(open(configname, O_RDONLY), &len);
	if (!data.start)
		fatal("reading included file ", errno, configname, NULL);
	data.pos = data.start;
	data.end = data.start + len;

	read_section_section(&data, sec);
}

static bool read_section(struct string *data, struct section *sec)
{
	struct token tok = get_token(data);

	if (tok.type == END)
		return false;

	if (tok.type != LEFT_BRACKET)
		parse_error(tok, "'[' to start new section");

	tok = get_token(data);
	if (tok.type != STRING)
		parse_error(tok, "path after '[' in section start");
	sec->name = tok.string;
	
	tok = get_token(data);
	if (tok.type != RIGHT_BRACKET)
		parse_error(tok, "']' after name in section start");

	read_section_section(data, sec);

	return true;
}

#ifndef PATH_MAX
#define PATH_MAX 4096
#endif

static bool section_matches(const char *dirname, struct section sec)
{
	unsigned int len = sremain(sec.name);
	char pattern[len + 2];

	/* Default always matches. */
	if (len == 0)
		return true;

	memcpy(pattern, sec.name.pos, len);
	/* Append a * if not there already. */
	if (pattern[len-1] == '*')
		pattern[len] = '\0';
	else {
		pattern[len] = '*';
		pattern[len+1] = '\0';
	}
	return (fnmatch(pattern, dirname, 0) == 0);
}

static struct section get_default(void)
{
	struct section def = { .cpus = 1, .distcplusplus_hosts_set = false };
	return def;
}

static void read_config_file(const char *configname, int fd,
			     const char *dirname,
			     struct section *result)
{
	unsigned long len;
	struct string data;

	data.start = suck_file(fd, &len);
	if (!data.start)
		fatal("reading ", errno, configname, NULL);
	data.pos = data.start;
	data.end = data.start + len;

	/* Trivial parser. */
	for (;;) {
		struct section sec = *result;
		struct token tok;

		tok = peek_token(data);
		if (tok.type == INCLUDE) {
			char *included;
			int incfd;
			swallow_token(&data, tok);

			included = to_path(get_value(&data));
			incfd = open(included, O_RDONLY);
			read_config_file(included, incfd, dirname, result);
			close(incfd);
			continue;
		}

		if (!read_section(&data, &sec))
			break;

		if (section_matches(dirname, sec)) {
			if (sec.verbose) {
				unsigned int len = sremain(sec.name);
				char str[len+1];
				memcpy(str, sec.name.pos, len);
				str[len] = '\0';
				verbose(sec, "Using section ", str, NULL);
			}
			*result = sec;
		}
	}

	free(data.start);
}

struct section read_config(const char *configname, const char *dir, int fd)
{
	struct section result;

	result = get_default();
	read_config_file(configname, fd, dir, &result);
	return result;
}
