/* Simple code to grab a lock to restrict number of parallel processes. */
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include "ccontrol.h"
#include "stdrusty.h"

/* We place much looser limits on "fast" operations: distcc-ables and make */
#define DISTCC_LIMIT 20
/* Make is limited separately for each depth, since it can recurse. */
#define MAKE_LIMIT 3

#define IPC_KEY 0xCCD1ED

static int fcntl_lock(const char *configname, unsigned int offset)
{
	struct flock fl;
	int fd;

	fl.l_type = F_WRLCK;
	fl.l_whence = SEEK_SET;
	fl.l_start = offset;
	fl.l_len = 1;

	fd = open(configname, O_RDWR);
	if (fd < 0)
		fatal("cannot open ", errno, configname, NULL);
	if (fcntl(fd, F_SETLKW, &fl) < 0)
		fatal("cannot lock ", errno, configname, NULL);
	return fd;
}

/* To create an initialized semaphore, we need a lock.  Use fcntl lock. */
static int make_semaphore(const char *configname, struct section sec)
{
	int fd, id;

	fd = fcntl_lock(configname, 0);
	id = semget(IPC_KEY, 1, IPC_CREAT|IPC_EXCL);
	if (id < 0) {
		int saved_errno = errno;
		/* We might have raced, try again. */
		id = semget(IPC_KEY, 1, 0);
		errno = saved_errno;
	} else {
		struct semid_ds ds;

		verbose(sec, "Created slow lock", NULL);
		memset(&ds, 0, sizeof(ds));
		ds.sem_perm.uid = geteuid();
		ds.sem_perm.gid = getegid();
		ds.sem_perm.mode = 0600;
		
		if (semctl(id, 0, IPC_SET, &ds) < 0)
			fatal("cannot set semaphore permissions",
			      errno, NULL);
		if (semctl(id, 0, SETVAL, sec.cpus) < 0)
			fatal("cannot set semaphore value", errno, NULL);
	}
	/* Close unlocks, too. */
	close(fd);
	return id;
}

/* Semaphores give us exact control over rate, but SEM_UNDO space is
 * often limited (not on Linux tho AFAICT). */
static void grab_sem(const char *configname, struct section sec)
{
	struct sembuf sop;
	int id;

	id = semget(IPC_KEY, 1, 0);
	if (id < 0 && errno == ENOENT)
		id = make_semaphore(configname, sec);
	if (id < 0)
		fatal("cannot get semaphore", errno, NULL);

again:
	sop.sem_num = 0;
	sop.sem_op = -1;
	sop.sem_flg = SEM_UNDO;

	if (semop(id, &sop, 1) != 0) {
		if (errno == EINTR)
			goto again;
		fatal("cannot decrement semaphore", errno, NULL);
	}
}

/* Since we have lots of these, we use fcntl locks as an approximate
 * means to limit them. */
static void grab_fcntl_lock(const char *configname,
			    unsigned int base,
			    unsigned int max)
{
	srand(getpid());

	fcntl_lock(configname, base + rand()%max);
}

void grab_lock(const char *configname,
	       struct section sec,
	       bool slow,
	       enum type type)
{
	char *lock = getenv("CCONTROL_LOCK");

	verbose(sec, "Grabbing lock for ",
		type == TYPE_CC ? "CC"
		: type == TYPE_CPLUSPLUS ? "C++"
		: type == TYPE_MAKE ? "MAKE"
		: type == TYPE_LD ? "LD" : "UNKNOWN", NULL);

	/* If we already have slow lock, don't grab again (gcc calls ld). */
	if (lock && lock[0] == '0') {
		verbose(sec, "Already got it", NULL);
		return;
	}

	if (slow) {
		setenv("CCONTROL_LOCK", "0", 1);
		verbose(sec, "Getting slow lock", NULL);
		grab_sem(configname, sec);
	} else {
		/* We're fast: make, gcc or g++.  gcc & g++ are
		 * limited together, as are each level of make (make
		 * can be recursive). */
		char locktype[2];
		unsigned int distcc_lim, make_off, make_lim;

		/* Position 0 is used to initialize slow semaphore.
		 * Next range is used by distcc-able builds.
		 * Then a series of ranges for each makefile depth. */
		make_off = distcc_lim = sec.cpus*DISTCC_LIMIT;
		make_lim = sec.cpus*MAKE_LIMIT;

		/* If using ccache, don't run as many in parallel,
		 * since ccache catches ~50% of them. */
		if (sec.ccache)
			distcc_lim /= 5;

		if (type == TYPE_CC || type == TYPE_CPLUSPLUS) {
			if (lock && lock[0] == '1')
				fatal("called myself?", 0, NULL);
			verbose(sec, "Getting fast lock for compile", NULL);
			grab_fcntl_lock(configname, 1, distcc_lim);
			locktype[0] = '1';
		} else {
			unsigned int depth = 0;
			if (lock)
				depth = lock[0] - 'A' + 1;
			verbose(sec, "Getting fast lock for make", NULL);
			grab_fcntl_lock(configname,
					1 + make_off + depth * make_lim,
					make_lim);
			locktype[0] = 'A' + depth;
		}
		locktype[1] = '\0';
		setenv("CCONTROL_LOCK", locktype, 1);
	}
}
