/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   Parts written by Sean Burford <unix.gurus@gmail.com> 2005
                                                                                
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
                                                                                
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
                                                                                
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include "RingBuffer.H"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

RingBuffer::RingBuffer(void)
    : ringBufferStart(0), ringBufferEnd(0)
{
}

int
RingBuffer::ContainsData(void) const
{
    return (ringBufferStart != ringBufferEnd);
}

int
RingBuffer::BufferUsed(void) const
{
    int bufferUsed;

    if (ringBufferStart < ringBufferEnd) {
        bufferUsed = ringBufferEnd - ringBufferStart;
    } else if (ringBufferStart == ringBufferEnd) {
        bufferUsed = 0;
    } else {
        bufferUsed = RING_BUFFER_SIZE - ringBufferStart + ringBufferEnd;
    }

    return(bufferUsed);
}

int
RingBuffer::BytesFreeToEnd(void) const
{
    int bytesToEnd;

    if (ringBufferStart <= ringBufferEnd) {
        bytesToEnd = RING_BUFFER_SIZE - ringBufferEnd;
    } else {
        bytesToEnd = ringBufferStart - ringBufferEnd;
    }

    return(bytesToEnd);
}

int
RingBuffer::BytesUsedToEnd(void) const
{
    int bytesToEnd;

    if (ringBufferStart <= ringBufferEnd) {
        bytesToEnd = ringBufferEnd - ringBufferStart;
    } else {
        bytesToEnd = RING_BUFFER_SIZE - ringBufferStart;
    }

    return(bytesToEnd);
}


char *
RingBuffer::BufferEnd(void)
{
    return(&ringBuffer[ringBufferEnd]);
}

const char *
RingBuffer::BufferEnd(int length)
{
    int bufferFree = RING_BUFFER_SIZE - BufferUsed();

    assert(bufferFree > length);

    ringBufferEnd = ( ringBufferEnd + length ) % RING_BUFFER_SIZE;

    return(&ringBuffer[ringBufferEnd]);
}

const char *
RingBuffer::BufferStart(void) const
{
    return(&ringBuffer[ringBufferStart]);
}

const char *
RingBuffer::BufferStart(int length)
{
    int bufferUsed = BufferUsed();

    assert(bufferUsed >= length);

    ringBufferStart = ( ringBufferStart + length ) % RING_BUFFER_SIZE;

    return(&ringBuffer[ringBufferStart]);
}

char *
RingBuffer::Peek(char *dest, int length, int offset) const
{
    assert (length > 0 && length < RING_BUFFER_SIZE);
    assert (offset >= 0 && offset < RING_BUFFER_SIZE);

    int start = (ringBufferStart + offset) % RING_BUFFER_SIZE;
    int bufferUsed = BufferUsed();

    // Check we are not asking for more bytes than we have on the buffer
    assert(offset + length <= bufferUsed);

    if (dest == NULL)
    {
        dest = (char *)malloc(length);
        if (dest == NULL) {
            printf("failed to malloc %i bytes\n", length);
            assert(0);
        }
    }

    if (start + length < RING_BUFFER_SIZE) {
        memcpy(dest, &ringBuffer[start], length);
    } else {
        // read across end of ring buffer
        const int bytesToEnd = RING_BUFFER_SIZE - start;
        memcpy(dest, &ringBuffer[start], bytesToEnd);
        memcpy(&dest[bytesToEnd], ringBuffer, length - bytesToEnd);
    }

    return(dest);
}

char *
RingBuffer::ReadFromStart(char *dest, int length)
{
    dest = Peek(dest, length, 0);

    ringBufferStart = ( ringBufferStart + length ) % RING_BUFFER_SIZE;

    return(dest);
}

void
RingBuffer::AppendToEnd(const char *src, int length)
{
    assert (src != NULL && length > 0 && length < RING_BUFFER_SIZE);
    int bufferLeft = RING_BUFFER_SIZE - BufferUsed();

    // Check we have enough space left in the ring buffer
    // Should really handle it better than this - disconnect dead client?
    assert(bufferLeft > length);

    if (ringBufferEnd + length < RING_BUFFER_SIZE) {
        memcpy(&ringBuffer[ringBufferEnd], src, length);
    } else {
        // write across end of ring buffer
        const int bytesToEnd = RING_BUFFER_SIZE - ringBufferEnd;
        memcpy(&ringBuffer[ringBufferEnd], src, bytesToEnd);
        memcpy(ringBuffer, &src[bytesToEnd], length - bytesToEnd);
    }

    ringBufferEnd = ( ringBufferEnd + length ) % RING_BUFFER_SIZE;
}

