/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/

#include <assert.h>
#include "Wizard.H"
#include "LogFile.H"

#define WIZARD_MAX_NUM_HIT_POINTS 14

Wizard::Wizard()
    : BaseCreature(1, WIZARD_MAX_NUM_HIT_POINTS),
      counterBlindness(0), counterInvisibility(0),
      counterFear(0), counterCharmPerson(0),
      charmer(NULL)
{
}

void
Wizard::addGestures(Gesture LeftHand, Gesture RightHand)
{
    /* Need to add check for spells like confusion, amnesia, paralysis */
    gestures.addLeftGesture(LeftHand);
    gestures.addRightGesture(RightHand);
}

void
Wizard::setLeftGesture(Gesture NewGesture)
{
    gestures.setLeftGesture(NewGesture);
}

void
Wizard::setRightGesture(Gesture NewGesture)
{
    gestures.setRightGesture(NewGesture);
}

const GestureHistory &
Wizard::getGestureHistory() const
{
    return gestures;
}

bool
Wizard::applyRemoveEnchantmentSpell(string *Message)
{
    *Message = "%S removes all enchantments on %T";
    clearStateBits(ENCHANTMENT_SPELLS);
    return true;
}

bool 
Wizard::applyDispelMagicSpell(string *Message)
{
    /* This spell has other effects which are handle elsewhere,
       but it also doubles as a shield spell, the state of which
       we set here */
    applyShieldSpell(Message);
    *Message = "%S casts Dispel Magic on %T";
    return true;
}

bool
Wizard::applyCharmPersonSpell(string *Message, BaseCreature *Source)
{
    *Message = "%S casts Charm Person on %T";
    setStateBits(CS_CHARM_PERSON);
    counterCharmPerson = 2;
    charmer = dynamic_cast<Wizard *>(Source);
    assert(charmer!=NULL);
    return true;
}

bool
Wizard::applyCharmMonsterSpell(string *Message, BaseCreature *)
{
    *Message = "%S casts Charm Monster on %T but has no effect";
    return false;
}

bool
Wizard::applyFearSpell(string *Message)
{
    *Message = "%S casts Fear on %T";
    setStateBits(CS_FEAR);
    counterFear = 2;
    return true;
}

bool
Wizard::applyAntiSpell(string *Message)
{
    *Message = "%S casts Anti Spell on %T";
    gestures.addLeftGesture(GST_ANTISPELL);
    gestures.addRightGesture(GST_ANTISPELL);
    return true;
}

bool
Wizard::applyBlindnessSpell(string *Message)
{
    *Message = "%S casts Blindness on %T";
    counterBlindness = 3;
    setStateBits(CS_BLINDED);
    return true;
}

bool
Wizard::applyInvisibilitySpell(string *Message)
{
    *Message = "%S casts Invisiblity on %T";
    counterInvisibility = 3;
    setStateBits(CS_INVISIBILITY);
    return true;
}

bool 
Wizard::applyDelayedEffectSpell(string *Message)
{
    *Message = "%S casts Delayed Effect on %T";
    setStateBits(CS_DELAYED_EFFECT);
    return true;
}

bool
Wizard::applyPermanencySpell(string *Message)
{
    *Message = "%S casts Permanency on %T";
    setStateBits(CS_PERMANENCY);
    return true;
}

void
Wizard::addMonster(NonWizard *Monster)
{
    monstersControlled.push_back(Monster);
}

const vector<NonWizard *> &
Wizard::getMonsters() const
{
    return monstersControlled;
}

void
Wizard::removeMonster(NonWizard *Monster)
{
    vector<NonWizard *>::iterator i;
    i = find(monstersControlled.begin(), monstersControlled.end(), Monster);
    assert(i!=monstersControlled.end());
    monstersControlled.erase(i);
}

Wizard *
Wizard::getCharmer() const
{
    return charmer;
}

void
Wizard::decrementCounters() 
{
    if (counterBlindness) {
	counterBlindness--;
	if (counterBlindness==0) {
	    clearStateBits(CS_BLINDED);
	}
    }

    if (counterInvisibility) {
	counterInvisibility--;
	if (counterInvisibility==0) {
	    clearStateBits(CS_INVISIBILITY);
	}
    }

    if (counterFear) {
	counterFear--;
	if (counterFear==0) {
	    clearStateBits(CS_FEAR);
	}
    }

    if (counterCharmPerson) {
	counterCharmPerson--;
	if (counterCharmPerson==0) {
	    clearStateBits(CS_CHARM_PERSON);
	}
    }

    BaseCreature::decrementCounters();
}

void
Wizard::Reset()
{
    counterBlindness = 0;
    counterInvisibility = 0;
    counterFear = 0;
    counterCharmPerson = 0;
    charmer = NULL;
    gestures.clear();
    monstersControlled.clear();

    // Call base class
    BaseCreature::Reset();
}
