/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include "RemoteConnection.H"
#include "RingBuffer.H"

#include <assert.h>
#include <errno.h>
#include <netinet/in.h>

RemoteConnection::RemoteConnection(int ConnectionFD)
    : connectionFD(ConnectionFD)
{
}

int
RemoteConnection::getFD() const
{
    return connectionFD;
}

bool
RemoteConnection::pendingDataToWrite() const
{
    return writeRingBuffer.ContainsData();
}

bool
RemoteConnection::haveMessageToRead() const
{
    return msgsToRead.size()!=0;
}

Message *
RemoteConnection::getNextMessage()
{
    assert(haveMessageToRead());

    
    int length = msgsToRead.front();
    msgsToRead.pop();

    assert(length != 0);

    char *messageBuffer = NULL;
    
    // Copy data to message buffer
    messageBuffer = readRingBuffer.ReadFromStart(messageBuffer, length);

    Message::ConvertToHostByteOrder((Message *)messageBuffer);
    return (Message *)messageBuffer;
}

void
RemoteConnection::writeMessage(Message *MsgToSend)
{
    const char *messageBuffer = (const char *)MsgToSend;
    int messageBufferSize = MsgToSend->length;

    // Temporarily convert network byte order
    Message::ConvertToNetworkByteOrder(MsgToSend);
    
    writeRingBuffer.AppendToEnd(messageBuffer, messageBufferSize);

    // Convert back to host byte order in case its MsgToSend is reused
    Message::ConvertToHostByteOrder(MsgToSend);

    // Attempt to write as much data as possible
    writeData();
}

void
RemoteConnection::readData() 
{
    int bytesRead;
    int bytesToEnd;

    // Append read bytes onto the read ring buffer
    bytesToEnd = readRingBuffer.BytesFreeToEnd();
    bytesRead = read(connectionFD, readRingBuffer.BufferEnd(), bytesToEnd);
                                                                                
    assert(bytesRead != 0);

    if (bytesRead<0) {
	perror("RemoteConnection::readData read failed");
	// Need to handle disconnects from client
	// raise an exception?
	assert(0);
    }

    // Increment the amount of storage used on the buffer
    readRingBuffer.BufferEnd(bytesRead);

    // Check for new message

    // Work out how many bytes we have left to read
    unsigned int bytesAvailableToRead = readRingBuffer.BufferUsed();

    // Assume all messages are of struct Message format
    // so we have header followed by length of entire message
    // could get multiple messages in single read
    int msgStartPoint = 0;
    while (bytesAvailableToRead>=sizeof(Message)) {
        Message msg;

        readRingBuffer.Peek((char *)&msg, sizeof(msg), msgStartPoint);
                                                                              
        unsigned int length = msg.length;
        length = ntohl(length);
        assert(length>=sizeof(Message) && length <= 8192);

        msgStartPoint += length;

        if (bytesAvailableToRead >= length) {
            // We have the entire message
            msgsToRead.push(length);
            bytesAvailableToRead -= length;
        } else {
            // Need more data to complete next message
            bytesAvailableToRead = 0;
        }
    }
}

void
RemoteConnection::writeData()
{
    while(writeRingBuffer.ContainsData())
    {
        int bytesToEnd;
        int bytesWritten;

        bytesToEnd = writeRingBuffer.BytesUsedToEnd();
        assert(bytesToEnd > 0 && bytesToEnd < 8192 );

        bytesWritten = write(connectionFD, writeRingBuffer.BufferStart(),
                           bytesToEnd);

        assert(bytesWritten!=0);

        if (bytesWritten<0) {
            perror("Write failed to client. Need to handle better");
            exit(1);
        } else {
            // Decrement the amount of storage used on the buffer
            writeRingBuffer.BufferStart(bytesWritten);
        }
    }
}
