/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>
#include <string.h>
#include <netinet/in.h>

#include "Player.H"
#include "Message.H"
#include "protocol.h"
#include "NonWizard.H"

Player::Player(int FD)
    : remoteConnection(FD), waitingForResponse(true)
{
    // Send welcome message
    printf("Sending welcome message\n");
    const char *welcomeMessage = "Welcome to SC Server";

    Message *msg;
    msg = Message::createMessage(MSG_ASK_WELCOME, strlen(welcomeMessage)+1);
    
    strcpy(msg->data, welcomeMessage);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
    waitingForResponse = true;
}

void
Player::sendNewPlayerInfo(Player *NewPlayer)
{
    // Send ID, Name
    int playerID = htonl(NewPlayer->getID());
    const char *playerName = NewPlayer->getName();
    int playerNameLength = strlen(playerName)+1;

    Message *msg = Message::createMessage(MSG_SEND_NEWPLAYER_INFO,
					  sizeof(int)*2+playerNameLength);

    int index = 0;
    memcpy(msg->data, &playerID, sizeof(int));
    index += sizeof(int);
    playerNameLength = htonl(playerNameLength);
    memcpy(msg->data+index, &playerNameLength, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, playerName, strlen(playerName)+1);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void 
Player::sendNewMonsterInfo(NonWizard *Monster)
{
    // Send ID, Type, OwnerID, Name
    int monsterID = htonl(Monster->getID());
    int monsterType = htonl(Monster->getMonsterType());
    int ownerID = htonl(Monster->getOwner()->getID());
    const char *monsterName = Monster->getName();
    int monsterNameLength = strlen(monsterName)+1;

    Message *msg = Message::createMessage(MSG_SEND_NEW_MONSTER_INFO,
					  sizeof(int)*4+monsterNameLength);
    int index = 0;
    memcpy(msg->data+index, &monsterID, sizeof(int));
    index += sizeof(int);

    memcpy(msg->data+index, &monsterType, sizeof(int));
    index += sizeof(int);

    memcpy(msg->data+index, &ownerID, sizeof(int));
    index += sizeof(int);

    monsterNameLength = htonl(monsterNameLength);
    memcpy(msg->data+index, &monsterNameLength, sizeof(int));
    index += sizeof(int);

    memcpy(msg->data+index, monsterName, strlen(monsterName)+1);

    assert(msg->Ok());
    
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::askForGestures()
{
    Message *msg;
    msg = Message::createMessage(MSG_ASK_FOR_GESTURES, 0);
    remoteConnection.writeMessage(msg);
    free(msg);
    waitingForResponse = true;
}

void 
Player::sendGesturesSeen(Player *OfPlayer)
{
    Gesture leftHand = GST_FOG, rightHand = GST_FOG;
    int playerID = htonl(OfPlayer->getID());

    if (!OfPlayer->isStateSet(CS_BLINDED||CS_PERM_BLINDED)) {
	OfPlayer->getGestureHistory()
	    .getCurrentGestures(&leftHand, &rightHand);
    }

    Message *msg;
    msg = Message::createMessage(MSG_SEND_GESTURES_SEEN, sizeof(int)*3);
    assert(sizeof(leftHand)==sizeof(int));
    leftHand = Gesture(htonl(leftHand));
    rightHand = Gesture(htonl(rightHand));
    memcpy(msg->data, &playerID, sizeof(int));
    memcpy(msg->data+sizeof(int), &leftHand, sizeof(int));
    memcpy(msg->data + sizeof(int)*2, &rightHand, sizeof(int));

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendGesturesSeenEnd()
{
    Message *msg;
    msg = Message::createMessage(MSG_SEND_GESTURES_SEEN, sizeof(int)*3);
    int IDZero = 0;

    // Send all zeros
    memcpy(msg->data, &IDZero, sizeof(int));
    memcpy(msg->data+sizeof(int), &IDZero, sizeof(int));
    memcpy(msg->data+sizeof(int)*2, &IDZero, sizeof(int));

    assert(msg->Ok());
       
    remoteConnection.writeMessage(msg);
    free(msg);

}

void
Player::askForSpellSelection(const vector<Spell> &LeftHandSpells,
			     const vector<Spell> &RightHandSpells)
{
    int numLeftSpells = LeftHandSpells.size();
    int numRightSpells = RightHandSpells.size();

    if (numLeftSpells>1 || numRightSpells>1) {
	// Keep list of possible spells for later validation
	validLHSpells = LeftHandSpells;
	validRHSpells = RightHandSpells;

	Message *msg;
	msg = Message::createMessage(MSG_ASK_FOR_SPELLS_CAST,
				     sizeof(int) * (2 + numLeftSpells
						    + numRightSpells));
	numLeftSpells = htonl(numLeftSpells);
	numRightSpells = htonl(numRightSpells);

	int index = 0;
	memcpy(msg->data, &numLeftSpells, sizeof(int));
	index += sizeof(int);
    
	assert(sizeof(Spell)==sizeof(int));

	vector<Spell>::const_iterator spell;
	int send_spell;
	for (spell=LeftHandSpells.begin(); 
	     spell!=LeftHandSpells.end(); 
	     spell++) {
	    send_spell = htonl(*spell);
	    memcpy(msg->data + index, &send_spell, sizeof(Spell)); 
	    index += sizeof(Spell);
	}

	memcpy(msg->data+index, &numRightSpells, sizeof(int));
	index += sizeof(int);
    
	for (spell=RightHandSpells.begin(); 
	     spell!=RightHandSpells.end(); spell++) {
	    send_spell = htonl(*spell);
	    memcpy(msg->data + index, &send_spell, sizeof(Spell)); 
	    index += sizeof(Spell);
	}
    
	assert(msg->Ok());

	remoteConnection.writeMessage(msg);
	free(msg);

	leftHandSpell = SPL_NONE;
	rightHandSpell = SPL_NONE;

	waitingForResponse = true;
    } else {
	// No need to ask as they have no choice
	if (numLeftSpells==1) {
	    leftHandSpell = LeftHandSpells[0];
	} else {
	    leftHandSpell = SPL_NONE;
	}
		
	if (numRightSpells==1) {
	    rightHandSpell = RightHandSpells[0];
	} else {
	    rightHandSpell = SPL_NONE;
	}
    }
}

void
Player::getSpellsCast(Spell *LeftHandSpell, int *LeftTargetID,
		      Spell *RightHandSpell, int *RightTargetID)
{
    *LeftHandSpell = leftHandSpell;
    *LeftTargetID = leftHandSpellTargetID;
    *RightHandSpell = rightHandSpell;
    *RightTargetID = rightHandSpellTargetID;
}

const map<NonWizard *, int> &
Player::getMonsterDirections() const 
{
    return monsterTargets;
}

void
Player::sendSpellCast(const SpellCast &SC) 
{
    int sourceID, targetID;
    int spell, spellWorked;
    int msgLength;

    sourceID = htonl(SC.source->getID());
    targetID = htonl(SC.target->getID());
    spell = htonl((int)SC.spellCast);
    spellWorked = htonl(SC.spellWorked ? 1 : 0);
    msgLength = SC.resultMessage.length()+1;
    
    Message *msg = Message::createMessage(MSG_SEND_SPELL_CAST, 
					  5*sizeof(int) + msgLength);
    int index = 0;
    memcpy(msg->data, &sourceID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &targetID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &spell, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &spellWorked, sizeof(int));
    index += sizeof(int);
    msgLength = htonl(msgLength);
    memcpy(msg->data+index, &msgLength, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, SC.resultMessage.c_str(), 
	   SC.resultMessage.length()+1);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void Player::sendSpellCastEnd()
{
    int IDZero = 0;
    Message *msg = Message::createMessage(MSG_SEND_SPELL_CAST,
					  sizeof(int));
    memcpy(msg->data, &IDZero, sizeof(int));
    
    assert(msg->Ok());
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendMonsterAttack(int SourceID, int TargetID, int Damage,
			  const string &ResultMessage) 
{
    printf("Player::sendMonsterAttack %i %i %i\n", SourceID, TargetID, Damage);
    SourceID = htonl(SourceID);
    TargetID = htonl(TargetID);
    Damage = htonl(Damage);
    int msgLength = ResultMessage.length()+1;

    Message *msg = Message::createMessage(MSG_SEND_MONSTER_ATTACK_INFO,
					  sizeof(int)*4 + msgLength);
    msgLength = htonl(msgLength);

    int index = 0;
    memcpy(msg->data+index, &SourceID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &TargetID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &Damage, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &msgLength, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, ResultMessage.c_str(), 
	   strlen(ResultMessage.c_str())+1);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendMonsterAttackEnd()
{
    int idZero = 0;

    Message *msg = Message::createMessage(MSG_SEND_MONSTER_ATTACK_INFO,
					  sizeof(int));
    memcpy(msg->data, &idZero, sizeof(int));
    
    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}



void
Player::sendCreatureState(BaseCreature *Creature) 
{
    int creatureID = htonl(Creature->getID());
    int hp = htonl(Creature->getNumHitPoints());
    unsigned int state = htonl(Creature->getState());
    
    Message *msg = Message::createMessage(MSG_SEND_CREATURE_STATE, 
					  sizeof(int)*3);

    int index = 0;
    memcpy(msg->data, &creatureID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &hp, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &state, sizeof(int));

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendCreatureStateEnd()
{
    int IDZero = 0;
    Message *msg = Message::createMessage(MSG_SEND_CREATURE_STATE,
					  sizeof(int));
    memcpy(msg->data, &IDZero, sizeof(int));
    
    assert(msg->Ok());
    
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendEndGame(Player *Winner) 
{
    int winnerID = htonl(Winner->getID());
    
    Message *msg = Message::createMessage(MSG_SEND_END_GAME, sizeof(int));
    memcpy(msg->data, &winnerID, sizeof(int));

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::askForSpellDirections(const vector<int> &ValidTargetIDs)
{
    // Target self by default
    leftHandSpellTargetID = getID();
    rightHandSpellTargetID = getID();

    // Need to change this to send SPL_NONE for non spell hand to
    // tell different between left and right. Also add use of ValidTargetIDs

    bool rightHandSpellTarget = false, leftHandSpellTarget = false;

    switch (leftHandSpell) {
    case SPL_NONE:
    case SPL_PERMANENCY:
    case SPL_DELAYED_EFFECT:
    case SPL_SURRENDER:
	leftHandSpell = SPL_NONE;
	break;

    default:
	leftHandSpellTarget = true;
    }

    switch (rightHandSpell) {
    case SPL_NONE:
    case SPL_PERMANENCY:
    case SPL_DELAYED_EFFECT:
    case SPL_SURRENDER:
	rightHandSpell = SPL_NONE;
	break;

    default:
	rightHandSpellTarget = true;
    }

    int LHSpell = htonl(leftHandSpell);
    int RHSpell = htonl(rightHandSpell);
    
    if (leftHandSpellTarget || rightHandSpellTarget) {
	printf("Have to ask for spell direction\n");
	Message *msg = Message::createMessage(MSG_ASK_SPELL_DIRECTIONS,
					      sizeof(int) * 
					      (3 + ValidTargetIDs.size()));
	int index = 0;
	assert(sizeof(leftHandSpell)==sizeof(int));
	memcpy(msg->data, &LHSpell, sizeof(int));
	index += sizeof(int);
	memcpy(msg->data+index, &RHSpell, sizeof(int));
	index += sizeof(int);

	int numTargets = htonl(ValidTargetIDs.size());
	
	memcpy(msg->data+index, &numTargets, sizeof(int));
	index += sizeof(int);

	vector<int>::const_iterator targetID;
	int target;
	
	for (targetID=ValidTargetIDs.begin(); 
	     targetID!=ValidTargetIDs.end();
	     targetID++) {
	    target = htonl(*targetID);
	    memcpy(msg->data+index, &target, sizeof(int));
	    index += sizeof(int);
	}

	assert(msg->Ok());
	remoteConnection.writeMessage(msg);
	free(msg);
	validTargetIDs = ValidTargetIDs;
	waitingForResponse = true;
    }
}

void
Player::askForMonsterDirections(const vector<int> &ValidTargetIDs)
{
    vector<NonWizard *> monsters = getMonsters();
    vector<NonWizard *>::iterator monster;

    monsterTargets.clear();
    validTargetIDs = ValidTargetIDs;

    for (monster = monsters.begin(); monster!=monsters.end(); monster++) {
	// Only look at live monsters
	if (!(*monster)->isStateSet(CS_DEAD)) {
	    // Target owner by default
	    monsterTargets[*monster] = getID();
	}
    }
    
    // FIXME -- need to filter out elemental monsters as they
    // do not need to be targetted and attack everyone/everything

    if (monsterTargets.size()>0) {
	// If we have at least one monster alive
	Message *msg;
	msg = Message::createMessage(MSG_ASK_MONSTER_DIRECTIONS,
				     sizeof(int)*(monsterTargets.size()+2
						  +ValidTargetIDs.size()));
	
	int numMonsters = htonl(monsterTargets.size());

	int index = 0;
	memcpy(msg->data+index, &numMonsters, sizeof(numMonsters));
	index += sizeof(numMonsters);

	map<NonWizard *, int>::iterator monSend;
	int monsterID;
	for (monSend = monsterTargets.begin();
	     monSend != monsterTargets.end();
	     monSend++) {
	    monsterID = htonl(monSend->first->getID());
	    memcpy(msg->data+index, &monsterID, sizeof(monsterID));
	    index += sizeof(monsterID);
	}

	int numTargets = htonl(ValidTargetIDs.size());
	memcpy(msg->data+index, &numTargets, sizeof(int));
	index += sizeof(int);
	
	vector<int>::const_iterator target;
	int targetID;
	for (target = ValidTargetIDs.begin();
	     target != ValidTargetIDs.end();
	     target++) {
	    targetID = htonl(*target);
	    memcpy(msg->data+index, &targetID, sizeof(int));
	    index += sizeof(int);
	}
	       
	assert(msg->Ok());
	remoteConnection.writeMessage(msg);
	free(msg);
	waitingForResponse = true;
	printf("Asking %i for monster directions\n", getID());
    } else {
	printf("Not asking %i for monster directions\n", getID());
    }
}

bool
Player::amWaitingForResponse() const
{
    return waitingForResponse;
}

const RemoteConnection &
Player::connection() const
{
    return remoteConnection;
}

RemoteConnection &
Player::connection()
{
    return remoteConnection;
}

void
Player::ProcessMessage() 
{
    assert(remoteConnection.haveMessageToRead());
    
    // Pull next message off the queue
    Message *msg = remoteConnection.getNextMessage();
    assert(msg->Ok());

    switch (msg->command) {
    case MSG_RCV_CLIENT_DETAILS:
	// Expect name of player
	// Should in future do checking that someone else doesn't already 
	// use that name
	if (msg->dataLength()>0) {
	    msg->data[msg->dataLength()-1] = 0;
	    printf("Client with name %s connected\n", msg->data);
	    setName(msg->data);
	    waitingForResponse = false;
	} else {
	    printf("Received dodgy MSG_RCV_CLIENT_DETAILS data\n");
	    // DISCONNECT CLIENT?
	}
	break;

    case MSG_RCV_GESTURES_USED:
	// Expect left hand and right hand gestures
	if (msg->dataLength()==sizeof(int)*2) {
	    int leftGesture, rightGesture;
	    memcpy(&leftGesture, msg->data, sizeof(int));
	    memcpy(&rightGesture, msg->data+sizeof(int), sizeof(int));
	    addGestures((Gesture)ntohl(leftGesture),
			(Gesture)ntohl(rightGesture));
	    waitingForResponse = false;
	} else {
	    printf("Received dodgy MSG_RCV_GESTURES_USED data\n");
	    // DISCONNECT CLIENT?
	}
	break;


    case MSG_RCV_SPELLS_CAST:
	// Expect left hand spell, right hand spell
	if (msg->dataLength()==sizeof(int)*2) {
	    int ls, rs;
	    Spell leftSpell, rightSpell;
	    memcpy(&ls, msg->data, sizeof(int));
	    memcpy(&rs, msg->data + sizeof(int), sizeof(int));
	    leftSpell = (Spell)ntohl(ls);
	    rightSpell = (Spell)ntohl(rs);

	    printf("RSC: %s Received %i %i\n", getName(), leftSpell, 
		   rightSpell);

	    // Check spell is from list originally given to client
	    if (find(validLHSpells.begin(), validLHSpells.end(), leftSpell)
		== validLHSpells.end()) {
		printf("Client tried to send invalid spell\n");
		leftSpell = SPL_NONE;
		// DISCONNECT CLIENT?
	    }
	    if (find(validRHSpells.begin(), validRHSpells.end(), rightSpell)
		== validRHSpells.end()) {
		printf("Client tried to send invalid spell\n");
		rightSpell = SPL_NONE;
		// DISCONNECT CLIENT?
	    }

	    leftHandSpell = leftSpell;
	    rightHandSpell = rightSpell;
	    waitingForResponse = false;
	}
	break;

    case MSG_RCV_SPELL_DIRECTIONS:
	// Expect left hand target, right hand target
	if (msg->dataLength()==sizeof(int)*2) {
	    memcpy(&leftHandSpellTargetID, msg->data, sizeof(int));
	    leftHandSpellTargetID = ntohl(leftHandSpellTargetID);
	    memcpy(&rightHandSpellTargetID, msg->data+sizeof(int),
		   sizeof(int));
	    rightHandSpellTargetID = ntohl(rightHandSpellTargetID);
	    
	    waitingForResponse = false;

	    // FIXME: NEED TO CHECK targets passed back!

	} else {
	    waitingForResponse = false;
	    ; // DISCONNECT CLIENT?
	}
	break;

    case MSG_RCV_MONSTER_DIRECTIONS:
	// Receives one monster direction at a time, rather than in a group
	if (msg->dataLength()>=sizeof(int)*2) {
	    int numMonsters;
	    int monsterID;
	    int targetID;
	    memcpy(&numMonsters, msg->data, sizeof(int));
	    numMonsters = ntohl(numMonsters);

	    // Check incoming data is of correct size
	    assert(numMonsters*sizeof(int)*2+sizeof(int) == msg->dataLength());

	    int index = sizeof(int);
	    for (int i=0; i<numMonsters; i++) {
		memcpy(&monsterID, msg->data+index, sizeof(int));
		index += sizeof(int);
		monsterID = ntohl(monsterID);
		
		memcpy(&targetID, msg->data+index, sizeof(int));
		index += sizeof(int);
		targetID = ntohl(targetID);

		printf("Received monster directions: %i %i\n",
		       monsterID, targetID);
		NonWizard *monster = 
		    dynamic_cast<NonWizard *>(BaseCreature::mapIDToPointer(monsterID));
		// Check we have a monster
		assert(monster!=NULL);

		// FIXME: Need to check target passed back is valid
		// FIXME: Need to check monster is controlled by this person
		monsterTargets[monster] = targetID;
	    }
	    waitingForResponse = false;

	} else {
	    waitingForResponse = false;
	    // DISCONNECT CLIENT?
	}
	break;


    default:
	printf("Received unknown message %i. Ignoring\n", msg->command);
    }

    // free message
}

