 /*
 * Copyright (C) Michael Fagan <Michael.Fagan@protocom.com.au> 2005
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ClientDispatch.h"
#include "ThothException.h"
#include "BrainInit.h"
#include "common.h"
#include <stdio.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>
#include <vector>

namespace thoth
{
using std::vector;


#define SERVER_PORT 12701
   

    ClientDispatch::~ClientDispatch()
{
    int fd = m_client->getConnectionFD();
    delete m_client;
    close(fd);
} // ~ClientDispatch()


void ClientDispatch::connect()
{
    struct sockaddr_in sa;
    struct hostent *server;

    DEBUG("Awakening the wizard '%s'", m_name.c_str());

    server = ::gethostbyname(m_hostname.c_str());
    assert(server);

    int fd = ::socket(AF_INET, SOCK_STREAM, 0);
    if ( fd < 0 ) {
        throw ThothException("Could not create socket");
    }

    memset(&sa, 0, sizeof(struct sockaddr_in)); 
    sa.sin_family = AF_INET;
    sa.sin_port = htons(SERVER_PORT);
    memcpy(&sa.sin_addr.s_addr, server->h_addr_list[0], server->h_length);
    
    if ( ::connect(fd, (struct sockaddr *)&sa, sizeof(sa)) < 0 ) {
        throw ThothException("Connect failed");
    }

    m_client = new SCClient(fd, m_name.c_str());

    srandom(time(NULL));
    
    DEBUG("ServerMessage: %s", m_client->getServerWelcomeMessage().c_str() );
} // connect()


void ClientDispatch::dispatchLoop()
{
    int msg = 0;
    int wins = 0;
    int loses = 0;
    int draws = 0;
    int timeout = 120;
    int maxRounds = 15;
    int round = 0;
    int rounds2kill = 0;
    int points = 0;
    int fd = m_client->getConnectionFD();
    fd_set readFS, writeFS;


    DEBUG("Initing the Brain!");    
    InitBrain(m_brain);
    
    while(true)
    {
        FD_ZERO(&readFS);
        FD_ZERO(&writeFS);

        FD_SET(fd, &readFS);
        if ( m_client->haveDataToWrite() )
        {
            FD_SET(fd, &writeFS);
        }
        if ( select(fd+1, &readFS, &writeFS, NULL, NULL) < 0 )
        {
            throw ThothException("Error occurred with comms to server");
        }

        if ( FD_ISSET(fd, &writeFS) )
        {
            m_client->writeData();
        }

        if ( !FD_ISSET(fd, &readFS) )
        {
            continue;
        }
        
        m_client->readData();
        while( m_client->messageAvailable() )
        {
            msg = m_client->processMessage();

            switch(msg)
            {
                case MSG_ASK_FOR_GESTURES:
                {
                    DEBUG("--------------------");
                    ++round;
                    DEBUG("Round %i", round);

                    signal("round", round);
                    Signal* tmp = SIGNALS.getSignal("send-gestures");
                    int left, right;

                    tmp->set(GST_NOTHING, GST_NOTHING);

                    m_brain.cycle();
                    
                    tmp->getValues(left, right);

                    m_client->sendGestures( (Gesture)left, (Gesture)right );
                }
                break;
                                
                case MSG_ASK_FOR_SPELLS_CAST:
                {
                    vector<Spell> left_opts;
                    Spell leftSP = SPL_NONE;
                    vector<Spell> right_opts;
                    Spell rightSP = SPL_NONE;

                    m_client->getPossibleSpells(&left_opts, &right_opts);

                    switch( left_opts.size() )
                    {
                        case 0:
                        break;

                        case 1:
                        {
                            leftSP = left_opts[0];
						}    
                        break;

                        default:
                        {
                            leftSP = SPL_NONE;
                            int len = 0;
                            vector<Spell>::iterator itr = left_opts.begin();
                            for( ; itr != left_opts.end(); ++itr)
                            {
                                if ( getSpellLength(*itr) > len )
                                {
                                    leftSP = *itr;
                                    len = getSpellLength(*itr);
                                }
                            }
                        }
                        break;
                    }

                    switch( right_opts.size() )
                    {
                        case 0:
                        break;

                        case 1:
                        {
                            rightSP = right_opts[0];
						}    
                        break;

                        default:
                        {
                            rightSP = SPL_NONE;
                            int len = 0;
                            vector<Spell>::iterator itr = right_opts.begin();
                            for( ; itr != right_opts.end(); ++itr)
                            {
                                if ( getSpellLength(*itr) > len )
                                {
                                    rightSP = *itr;
                                    len = getSpellLength(*itr);
                                }
                            }
                        }
                        break;
                    }
                    correctSpell(leftSP, rightSP);

                    DEBUG("** CAST %s **", Spells::getSpellName(leftSP));
                    DEBUG("** CAST %s **", Spells::getSpellName(rightSP));
                    m_client->sendSpellSelection(leftSP, rightSP);
                }
                break;
                
                case MSG_SEND_GESTURES_SEEN:
                {
                    if ( !m_client->haveAllGestures() )
                        break;

                    updateHistory();
                }
                break;
                
                case MSG_ASK_SPELL_DIRECTIONS:
                {
                    Spell leftSP, rightSP;
                    int left, right;
                    vector<int> v;
                    m_client->getSpellsCast(&leftSP, &rightSP, &v);

                    findTarget(leftSP, left);
                    findTarget(rightSP, right);

                    DEBUG("Targets %i, %i", left, right);
                    m_client->sendSpellTargets(left, right);
                }
                break;

                case MSG_SEND_SPELL_CAST:
                {
                    // TODO get spell cast info
                }
                break;
                
                case MSG_SEND_CREATURE_STATE:
                {
                    if ( !m_client->haveAllCreatureInfo() )
                        break;

                    updateCreatures();
                }
                break;
                
                case MSG_SEND_END_GAME:
                {
                    vector<int> v = m_client->getGameWinner();
                    if ( v.size() != 1 ) {
                        ++draws;
                        ++points;
                    } else if ( v[0] == m_client->getOwnID() ) {
                        ++wins;
                        rounds2kill += round;
                        points += 3;
                    } else {
                        ++loses;
                    }
                    DEBUG("Wins: %i, Draws: %i, Loses: %i, Rounds2Kill: %f, points: %i, avg pts: %f",
                          wins, draws, loses,
                          (float)rounds2kill / (float)wins,
                          points, (float)points/(float)(wins+draws+loses) );
                }
                break;
                
                case MSG_SEND_EVENT_INFO:
                {
                    SCClient::EventMessage evt_msg = m_client->getEventMessage();
                    DEBUG("EVent: %s", m_client->decodeEventMessage(evt_msg).c_str() );
                }
                break;
                
                case MSG_ASK_MONSTER_DIRECTIONS:
                {
                        // Monsters always attack the enemy wizard
                    vector<int> t;
                    vector<int> v = m_client->getMonstersToDirect(&t);
                    map<int,int> m;
                    vector<int>::iterator itr = v.begin();
                    for( ; itr != v.end(); ++itr)
                    {
                        m[(*itr)] = m_red_ID;
                    }
                    m_client->sendMonsterTargets(m);
                }
                break;
                                
                case MSG_SEND_START_GAME:
                {
                    timeout = m_client->getTurnTimeout();
                    maxRounds = m_client->getMaxRounds();
                    round = 0;
                    m_white_ID = m_client->getOwnID();
                    m_red_ID = -1;
                    m_resistCold = m_resistHeat = false;
                    m_red_l_hand = m_red_r_hand = 0;
                    m_disenchant = false;

                    m_brain.reset();

                    DEBUG("max_rounds: %i, timeout: %i", maxRounds, timeout);

                    signal("round", round);
                    signal("max-round", maxRounds);
                    signal("white-gesture", GST_NOTHING, GST_NOTHING);
                }
                break;
                
                case MSG_ASK_CHARM_PERSON_CTRL_HAND:
                {
                    int aID = 0;
                    m_client->GetCharmedHandWizard(&aID);
                    m_client->SendCharmedHandWizard(aID, findHand(true) );
                }
                break;
                
                case MSG_ASK_PARALYSIS_CTRL_HAND:
                {
                    int aID = 0;
                    m_client->GetParalysedHandWizard(&aID);
                    m_client->SendParalysedHandWizard(aID, findHand(false) );
                }
                break;
                
                case MSG_ASK_CHARM_PERSON_CTRL_GESTURE:
                {
                    int aID = 0;
                    int aHand = 0;
                    Gesture g = GST_NOTHING;

                    m_client->GetCharmedWizard(&aID, &aHand);
                    if ( aHand == SC_LEFT_HAND )
                    {
                        if ( m_red_r_hand == 1 )
                        {
                            g = GST_PALM;
                        }
                    } else {
                        if ( m_red_l_hand == 1 )
                        {
                            g = GST_PALM;
                        }
                    }
                    m_client->SendCharmedWizard(aID, g);
                }
                break;
                
                default:
                break;
            }
        }
    }
} // dispatchLoop()


void ClientDispatch::updateCreatures()
{
    vector<int> v;
    SCClient::CreatureState state;

    m_client->getAllCreatureIDs(&v);

    vector<int>::iterator itr = v.begin();
    for( ; itr != v.end(); ++itr)
    {
        int ID = *itr;
        m_client->getCreatureInfo(*itr, &state);
        if ( ( ID != m_white_ID ) && ( m_red_ID == -1 ) && ( state.type == 0 ) )
        {
            m_red_ID = ID;
        }
        if ( ( ID != m_white_ID ) && ( ID != m_red_ID ) )
        {
            signal(state.owner == m_white_ID ? "white-monster" : "red-monster",
                   state.type, ID);
        }
        DEBUG("Name: %s[%i], type: %s, hps: %i, owner: %i %s",
                state.name.c_str(), ID, monster2str(state.type),
                state.hitPoints, state.owner,
                (ID == m_white_ID ? "you" : (ID == m_red_ID ? "enemy" : "") ) );
    }
    if ( m_red_ID == -1 )
        throw ThothException("Unable to determine opponent ID");
} // updateCreatures()


void ClientDispatch::signal(const string& name, int param1, int param2)
{
    Signal* tmp = SIGNALS.getSignal(name);
    tmp->signal(param1, param2);
} // signal()


void ClientDispatch::updateHistory()
{
    Gesture left, right;
    bool antispell = false;

        // update white
    m_client->getGesturesForPlayer(m_white_ID, &left, &right, &antispell);

    if ( antispell )
    {
        left = right = GST_ANTISPELL;
    }
    signal("white-gesture", left, right);


        // update red
    m_client->getGesturesForPlayer(m_red_ID, &left, &right, &antispell);

    if ( antispell )
    {
        left = right = GST_ANTISPELL;
    }
    signal("red-gesture", left, right);

} // updateHistory()


void ClientDispatch::findTarget(Spell sp, int& target)
{
    switch(sp)
    {
        case SPL_CHARM_MONSTER:
            target = 0;
            //findMonster()
        break;

        case SPL_REMOVE_ENCHANTMENT:
        {
            if (m_disenchant)
                target = m_white_ID;
            else
                target = m_red_ID;
        }
        break;

        case SPL_RAISE_DEAD:
        case SPL_DISPEL_MAGIC:
        case SPL_COUNTER_SPELL:
        case SPL_COUNTER_SPELL1:
        case SPL_MAGIC_MIRROR:
        case SPL_PROTECTION_FROM_EVIL:
        case SPL_RESIST_HEAT:
        case SPL_RESIST_COLD:
        case SPL_INVISIBILITY:
        case SPL_CURE_LIGHT_WOUNDS:
        case SPL_CURE_HEAVY_WOUNDS:
        case SPL_SUMMON_GOBLIN:
        case SPL_SUMMON_OGRE:
        case SPL_SUMMON_TROLL:
        case SPL_SUMMON_GIANT:
        case SPL_SUMMON_FIRE_ELEMENTAL:
        case SPL_SUMMON_ICE_ELEMENTAL:
            target = m_white_ID;
        break;

        case SPL_SHIELD:
        case SPL_MISSILE:
        case SPL_FINGER_OF_DEATH:
        case SPL_LIGHTNING_BOLT:
        case SPL_LIGHTNING_BOLT1:
        case SPL_CAUSE_LIGHT_WOUNDS:
        case SPL_CAUSE_HEAVY_WOUNDS:
        case SPL_FIREBALL:
        case SPL_FIRESTORM:
        case SPL_ICESTORM:
        case SPL_AMNESIA:
        case SPL_CONFUSION:
        case SPL_CHARM_PERSON:
        case SPL_PARALYSIS:
        case SPL_FEAR:
        case SPL_ANTI_SPELL:
        case SPL_DISEASE:
        case SPL_POISON:
        case SPL_BLINDNESS:
        case SPL_STAB:
            target = m_red_ID;
        break;

        case SPL_HASTE:
        case SPL_TIME_STOP:
        case SPL_DELAYED_EFFECT:
        case SPL_PERMANENCY:
        case SPL_SURRENDER:
        case SPL_NONE:
        default:
            target = 0;
        break;
    }
} // findTarget()


int ClientDispatch::findHand(bool charm)
{
    int val = (charm ? 2 : 1);
    switch(m_red_l_hand)
    {
        case 0:
        {
            m_red_l_hand = val;
            return SC_LEFT_HAND;
        }
        break;
        
        case 1:
        {
            switch(m_red_r_hand)
            {
                case 0:
                {
                    m_red_r_hand = val;
                    return SC_RIGHT_HAND;
                }
                break;
                
                case 1:
                case 2:
                {
                    m_red_l_hand = val;
                    return SC_LEFT_HAND;
                }
                break;
            }
        }
        break;
        
        case 2:
        {
            m_red_r_hand = val;
            return SC_RIGHT_HAND;
        }
        break;
    }
    return SC_LEFT_HAND;
} // findHand()


int ClientDispatch::getSpellLength(Spell sp)
{
    switch(sp)
    {
        case SPL_STAB:
        case SPL_SHIELD:
            return 1;
        break;

        case SPL_MAGIC_MIRROR:
        case SPL_MISSILE:
            return 2;
        break;

        case SPL_COUNTER_SPELL:
        case SPL_COUNTER_SPELL1:
        case SPL_PROTECTION_FROM_EVIL:
        case SPL_CURE_LIGHT_WOUNDS:
        case SPL_SUMMON_GOBLIN:
        case SPL_CAUSE_LIGHT_WOUNDS:
        case SPL_AMNESIA:
        case SPL_CONFUSION:
        case SPL_PARALYSIS:
        case SPL_FEAR:
        case SPL_ANTI_SPELL:
            return 3;
        break;

        case SPL_CHARM_MONSTER:
        case SPL_REMOVE_ENCHANTMENT:
        case SPL_DISPEL_MAGIC:
        case SPL_RESIST_HEAT:
        case SPL_RESIST_COLD:
        case SPL_INVISIBILITY:
        case SPL_CURE_HEAVY_WOUNDS:
        case SPL_SUMMON_OGRE:
        case SPL_LIGHTNING_BOLT1:
        case SPL_CAUSE_HEAVY_WOUNDS:
        case SPL_FIRESTORM:
        case SPL_ICESTORM:
        case SPL_CHARM_PERSON:
            return 4;
        break;        

        case SPL_SUMMON_TROLL:
        case SPL_SUMMON_FIRE_ELEMENTAL:
        case SPL_SUMMON_ICE_ELEMENTAL:
        case SPL_LIGHTNING_BOLT:
        case SPL_FIREBALL:
            return 5;
        break;        

        case SPL_RAISE_DEAD:
        case SPL_SUMMON_GIANT:
        case SPL_DISEASE:
        case SPL_POISON:
        case SPL_BLINDNESS:
            return 6;
        break;       

        case SPL_FINGER_OF_DEATH:
            return 8;
        break;        

        case SPL_HASTE:
        case SPL_TIME_STOP:
        case SPL_DELAYED_EFFECT:
        case SPL_PERMANENCY:
        case SPL_SURRENDER:
        case SPL_NONE:
        default:
            return 0;
        break;        
    }
} // getSpellLength()


void ClientDispatch::correctSpell(Spell& left, Spell& right)
{
} // correctSpell()

    
}; // namespace thoth    
