#!/usr/bin/env python2.3

# vim: set shiftwidth=4 tabstop=4 expandtab smartindent textwidth=80:

# Part of ecar, an entry for the LCA 2005 hackfest:
# http://lca2005.linux.org.au/hackfest/
# (C) 2005, Russell Steicke, <russells@adelie.cx>
# Licensed under GPL, see the file COPYING.


# Make us faster, if available.
try:
    import psyco
    psyco.full()
except ImportError, reason:
    pass
import sys
import gtk
import gobject
import socket
from random import randint
import traceback
from types import StringType
from time import time

# Local stuff
from hflib.protocol import *
from hflib.spells import *
from Player import *
from ScMonster import ScMonster

# Debugging flag
df = False

def d(s):
    global df
    if df:
        print "%05.1f: %s" % (time()%1000, s)

# Prepare for l10n, although it's not implemented yet.
def _(s):
    return s

# Widgets to put in players images.
creaturesWidgetsTable = None

# Tables in which to place players' monsters, indexed by playerID.
monsterContainers = {}

#
statusBar = None
statusBar2 = None

# These are defaults, and can be changed by the user in the connection dialog
# box.  The player name _should_ be changed, or it becomes difficult to tell
# which player is ours when the server lists them.
serverHost = "localhost"
serverPort = DEFAULT_SPELLCAST_SERVER_PORT
myPlayerName = "Ecar"
myPlayerNumber = 0

# Our own player needs to be handled specially.
myPlayer = None

# Server timeout
turnTimeout = 0

# This dictionary is needed so we can recognise our own player when it comes
# back in a MSG_SEND_NEWPLAYER_INFO message from the server.  We want to leave
# our own player on the left of the list.
creaturesNamesDictionary = {}

# Messages from the server (except for MSG_SEND_NEWPLAYER_INFO) indicate players
# by number, so we need this dictionary to change the attributes of the player.
creaturesNumbersDictionary = {}

licenceText = '''\
Copyright (C) Russell Steicke <russells@adelie.cx> 2005
   
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.
   
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
   
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
'''



class HelpAboutDialog(gtk.Dialog):

    """A dialog to use if the version of pygtk we are running on does not have
    AboutDialog (< 2.6)."""

    def __init__(self, mainWindow):
        gtk.Dialog.__init__(self, _("About Ecar"), mainWindow,
                                   gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                   (_("OK"),gtk.RESPONSE_OK))
        self.connect("response", lambda *args: self.hide())
        # If string[i][1] is true, the string is selectable.
        strings = (
            ('<span weight="bold" size="xx-large">Ecar 0.1</span>'                    , 0),
            (),
            ('A spellcast client for the LCA 2005 Hackfest'                            , 0),
            ('<span size="small"><tt>http://lca2005.linux.org.au/hackfest/</tt></span>', 1),
            (),
            ('Copyright 2005, Russell Steicke'                                         , 0),
            ('<span size="small">&lt;<tt>russells@adelie.cx</tt>&gt;</span>'           , 1),
            ('<span size="small"><tt>http://adelie.cx/hackfest/</tt></span>'           , 1),
            ('Copy under the terms of the GNU General Public Licence'                  , 0))
        svbox = gtk.VBox()
        for string in strings:
            if len(string) == 0:
                svbox.pack_start(gtk.HSeparator(), padding=2)
            else:
                label = gtk.Label()
                label.set_use_markup(True)
                if string[1]:
                    label.set_selectable(True)
                label.set_markup(string[0])
                svbox.pack_start(label, padding=2)
        hbox = gtk.HBox()
        hbox.pack_start(svbox, padding=50)
        icon = gtk.Image()
        icon.set_from_stock(gtk.STOCK_DIALOG_INFO, gtk.ICON_SIZE_DIALOG)
        hbox.pack_start(icon, padding=10)
        vbox = self.vbox
        vbox.pack_start(hbox, padding=10)
        vbox.show_all()


sbContext = 0
def showTimedStatusBarMessage(sec, msg):
    global sbContext
    sbContext += 1
    sbc = sbContext
    statusBar.push(sbc, msg)
    gobject.timeout_add(sec*1000, lambda : statusBar.pop(sbc))


class AskQuitDialog(gtk.Dialog):

    askNumber = 0

    askStrings = ( _("But we're having so much fun..."),
                   _("Go on, bugger off then."),
                   _("NOOOO!  Don't go."),
                   _("What are you going to do\nwith your time now?"),
                   _("Quick, hide me from the boss!"),
                   _("Don't you love me?"),
                   _("It's been a pleasure."),
                   _("Come back real soon, now."),
                   _("Really?"),
                   _("Cast finger-of-death\nat the game?"),
                   _("Yeah, I don't like\nthis game, either."),
                   _("Loser!"),
                   _("Goodbye?  <sob>"),
                   _("Stay, please.  I'll be your friend..."),
                   _("Are you really really really reeealllly\nsure you want to exit?")
                   )
    numAskStrings = len(askStrings)

    def getAskString(self):
        return self.askStrings[randint(0, self.numAskStrings-1)]

    def __init__(self, mainWindow):
        gtk.Dialog.__init__(self, "Quit?", mainWindow,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT,
                             gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT))
        vbox = self.vbox
        hbox = gtk.HBox()
        if AskQuitDialog.askNumber == 38:
            askString = _("I pity you.")
        elif AskQuitDialog.askNumber == 16:
            askString = _("You really are persistent.")
        elif AskQuitDialog.askNumber >= 12:
            askString = _("Quit?")
        elif AskQuitDialog.askNumber >= 10:
            askString = _("You're killing me.")
        elif AskQuitDialog.askNumber >= 8:
            askString = _("NO MORE!  PLEASE STOP!")
        elif AskQuitDialog.askNumber >= 6:
            askString = _("I'm not showing you\nany more silly things.")
        elif AskQuitDialog.askNumber >= 4:
            askString = _("Quit?")
        else:
            askString = self.getAskString()
        AskQuitDialog.askNumber += 1
        hbox.pack_start(gtk.Label(askString), padding=10)
        icon = gtk.Image()
        icon.set_from_stock(gtk.STOCK_DIALOG_QUESTION, gtk.ICON_SIZE_DIALOG)
        hbox.pack_start(icon, padding=10)
        vbox.pack_start(hbox, padding=10)
        vbox.show_all()
        self.connect("key-press-event", self.keyPressed)

    def keyPressed(self, widget, event):
        if event.keyval == 0xff1b:
            self.emit("response", gtk.RESPONSE_REJECT)


class AskConnectionDialog(gtk.Dialog):

    def __init__(self, mainWindow, defaultServerHost, defaultServerPort, defaultPlayerName):
        gtk.Dialog.__init__(self, "Connect", mainWindow,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT,
                              gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT))
        self.serverHost = defaultServerHost
        self.serverPort = defaultServerPort
        self.playerName = defaultPlayerName
        table = gtk.Table()
        table.set_row_spacings(5)
        table.set_col_spacings(5)
        table.set_border_width(5)

        table.attach(gtk.Label(_("Player name:")), 0,1, 0,1)
        self.nameEntry = gtk.Entry(30)
        self.nameEntry.set_width_chars(30)
        self.nameEntry.set_text(defaultPlayerName)
        table.attach(self.nameEntry, 1,3, 0,1)

        table.attach(gtk.Label(_("Host:")), 0,1, 1,2)
        self.hostEntry = gtk.Entry(60)
        self.hostEntry.set_width_chars(30)
        self.hostEntry.set_text(defaultServerHost)
        table.attach(self.hostEntry, 1,3, 1,2, xoptions=0, yoptions=0)

        table.attach(gtk.Label(_("Port:")), 0,1, 2,3)
        self.portEntry = gtk.Entry(6)
        self.portEntry.set_width_chars(6)
        self.portEntry.set_text(str(defaultServerPort))
        table.attach(self.portEntry, 1,2, 2,3, xoptions=gtk.FILL, yoptions=0)
        defaultPortButton = gtk.Button(_("Use default port"))
        defaultPortButton.connect("clicked", \
                        lambda w: self.portEntry.set_text(str(DEFAULT_SPELLCAST_SERVER_PORT)))
        table.attach(defaultPortButton, 2,3, 2,3, xoptions=0, yoptions=0,
                     xpadding=5, ypadding=5)

        self.vbox.add(table)
        self.vbox.show_all()

        self.connect("key-press-event", self.keyPressed)

        self.done = False


    def keyPressed(self, widget, event):
        # There must be a better way to tell an enter or escape key.
        if event.keyval == 0xff1b:
            self.emit("response", gtk.RESPONSE_REJECT)
        elif event.keyval == 0xff0d:
            self.emit("response", gtk.RESPONSE_ACCEPT)


    def getHostEntryText(self):
        return self.hostEntry.get_text()


    def getPortEntryText(self):
        return self.portEntry.get_text()


    def getNameEntryText(self):
        return self.nameEntry.get_text()


class MyBoldableLabel(gtk.Label):

    '''A label that we can easily set to bold.'''
    def __init__(self, text):
        gtk.Label.__init__(self, text)
        self.normalText = text
        self.boldText = "<b>"+text+"</b>"
        self.set_use_markup(True)
        self.boldFlag = False

    def setBold(self, boldFlag):
        # Handle the pathological case where boldFlag or self.boldFlag is
        # not a boolean...
        if (boldFlag and (not self.boldFlag)) or ((not boldFlag) and self.boldFlag):
            if boldFlag:
                self.set_markup(self.boldText)
            else:
                self.set_markup(self.normalText)
            self.boldFlag = boldFlag


class MyInputTab:
    '''A widget and a label that can be (de-)activated together.'''

    def __init__(self, widget, boldableLabel):
        self.widget = widget
        self.boldableLabel = boldableLabel

    def setActive(self, flag):
        self.widget.set_sensitive(flag)
        self.boldableLabel.setBold(flag)


class MyInputTabCollection(gtk.Notebook):

    def __init__(self):
        gtk.Notebook.__init__(self)
        self.tabNames = {}
        self.tabIndexes = {}

    def addInputTab(self, widget, name):
        boldableLabel = MyBoldableLabel(name)
        index = self.append_page(widget, boldableLabel)
        tab = MyInputTab(widget, boldableLabel)
        tab.index = index
        tab.setActive(False)
        self.tabNames[name] = tab
        self.tabIndexes[index] = tab

    def setActive(self, active):
        if active is None:
            for key in self.tabIndexes.keys():
                self.tabIndexes[key].setActive(False)
        else:
            if type(active) is str:
                tabs = self.tabNames
            elif type(active) is int:
                tabs = self.tabIndexes
            # Exception here if not int or str.
            for key in tabs.keys():
                tab = tabs[key]
                if active == key:
                    tab.setActive(True)
                    self.set_current_page(tab.index)
                else:
                    tab.setActive(False)


### ---------- MainWindow

class MainWindow(gtk.Window):

    class MyImageMenuItem(gtk.ImageMenuItem):

        """A MenuItem with a specified label and a stock image.  I can't figure
        out how to do this with the stock gtk widgets.

        'gtk.ImageMenuItem(gtk.STOCK_CONNECT)' gives me a menu item with the
        correct image but a label 'gtk-connect', which is not quite right."""

        def __init__(self, label, stockimage):
            gtk.MenuItem.__init__(self, label)
            image = gtk.Image()
            image.set_from_stock(stockimage, gtk.ICON_SIZE_MENU)
            self.set_image(image)


    ## --- MainWindow constructor

    def __init__(self):
        gtk.Window.__init__(self)
        self.set_property("resizable", False)
        self.set_title(_("Ecar - Spellcast"))
        self.connect("delete-event", self.on_delete_event)
        self.connect("destroy", lambda w: gtk.main_quit())

        menuBar = gtk.MenuBar()

        gameMenu = gtk.Menu()

        # STOCK_CONNECT (and probably STOCK_DISCONNECT) were added in pygtk 2.6.
        # If we're running on something earlier, we can't use them.  But rather
        # than test for the pygtk version, test for the functionality.
        try:
            xxx = gtk.STOCK_CONNECT
        except:
            gameConnectMenuItem = gtk.MenuItem(_("_Connect"))
            gameDisconnectMenuItem = gtk.MenuItem(_("_Disconnect"))
        else:
            gameConnectMenuItem = MainWindow.MyImageMenuItem(_("_Connect"), gtk.STOCK_CONNECT)
            gameDisconnectMenuItem = MainWindow.MyImageMenuItem(_("_Disconnect"),
                                                                gtk.STOCK_DISCONNECT)

        gameConnectMenuItem.connect("activate", self.on_gameConnect_activate)
        gameMenu.append(gameConnectMenuItem)
        gameDisconnectMenuItem.connect("activate", self.on_gameDisconnect_activate)
        gameMenu.append(gameDisconnectMenuItem)
        gameMenu.append(gtk.SeparatorMenuItem())
        gameQuitMenuItem = gtk.ImageMenuItem(stock_id=gtk.STOCK_QUIT)
        gameQuitMenuItem.connect("activate",
                                 lambda item: self.emit("delete-event",
                                                        gtk.gdk.Event(gtk.gdk.DELETE)))
        gameMenu.append(gameQuitMenuItem)
        gameMenuItem = gtk.MenuItem(_("_Game"))
        gameMenuItem.set_submenu(gameMenu)
        menuBar.append(gameMenuItem)

        helpMenu = gtk.Menu()
        helpAboutMenuItem = gtk.ImageMenuItem(stock_id=gtk.STOCK_ABOUT)
        helpAboutMenuItem.connect("activate", self.on_helpAbout_activate)
        helpMenu.append(helpAboutMenuItem)
        helpMenuItem = gtk.MenuItem(_("_Help"))
        helpMenuItem.set_submenu(helpMenu)
        menuBar.append(helpMenuItem)

        windowBox = gtk.VBox(False)
        windowBox.pack_start(menuBar, expand=0)

        mainTable = gtk.Table()
        mainTable.set_col_spacings(5)
        mainTable.set_row_spacings(5)
        mainTable.set_border_width(5)

        global creaturesWidgetsTable
        creaturesWidgetsTable = gtk.Table()
        #creaturesWidgetsTable.set_border_width(5)
        #creaturesWidgetsTable.set_row_spacings(5)
        #creaturesWidgetsTable.set_col_spacings(5)
        mainTable.attach(creaturesWidgetsTable, 10, 20, 0, 10)

        leftBox = gtk.VBox()
        leftBox.set_border_width(5)
        leftBox.set_spacing(5)

        ## --- Text messages (not implemented in the server yet)
        self.textMessagesView = gtk.TextView()
        self.textMessagesView.set_editable(False)
        self.textMessagesView.set_cursor_visible(False)
        self.textMessagesScrolledWindow = gtk.ScrolledWindow()
        self.textMessagesScrolledWindow.add_with_viewport(self.textMessagesView)
        leftBox.pack_start(self.textMessagesScrolledWindow, expand=1, fill=1)

        textMessageBox = gtk.HBox()
        textMessageBox.set_spacing(5)
        self.textMessageEntry = gtk.Entry()
        self.textMessageEntry.set_width_chars(50)
        textMessageBox.pack_start(gtk.Label(_("Message:")))
        textMessageBox.pack_start(self.textMessageEntry, expand=1, fill=1)
        leftBox.pack_start(textMessageBox, expand=0, fill=0)

        ## --- Tabbed box (gtk.Notebook) for the controls
        self.myInputTabs = MyInputTabCollection()
        self.addSpellSelectTab()
        self.addSpellTargetSelectTab()
        self.addMonsterDirectionsTab()
        self.addHandsAndGesturesTab()

        ## --- Pack the left box and other stuff in.

        leftBox.pack_start(self.myInputTabs, expand=0, fill=0)

        mainTable.attach(leftBox, 8,9, 0,10)
        mainTable.show_all()

        windowBox.pack_start(mainTable, expand=1)

        global statusBar, statusBar2
        statusBar = gtk.Statusbar()
        statusBar.set_has_resize_grip(False)
        statusBar2 = gtk.Statusbar()
        statusBar2.set_has_resize_grip(False)
        statusBarHBox = gtk.HBox()
        statusBarHBox.pack_start(statusBar)
        statusBarHBox.pack_start(statusBar2)
        windowBox.pack_start(statusBarHBox, expand=0)

        self.add(windowBox)
        windowBox.show_all()

        iconImage = gtk.Image()
        iconImage.set_from_file('spellcast_gui.png')
        self.set_icon(iconImage.get_pixbuf())

        self.connection = None

        # Add a timeout function to listen for messages.
        gobject.timeout_add(100, self.messageReadingTimeout)
        # Activate Game->Connect after the window shows.  This is only called
        # once as the lambda returns None.  This would be better called once
        # when the window is shown, but that seems a bit trickier.  (I can't
        # find the correct event or signal for that.)
        gobject.timeout_add(500, lambda: gameConnectMenuItem.emit("activate"))


    ## --- Enter selected spells
    def addSpellSelectTab(self):
        spellSelectTable = gtk.Table()
        spellSelectTable.set_row_spacings(5)
        spellSelectTable.set_col_spacings(5)
        spellSelectTable.set_border_width(5)
        # Setting the table to be homogeneous solves a layout problem where the
        # comboboxes decide to be very narrow when they're empty.  This way
        # they're forced to be as wide as the widest label, which seems to be
        # good enough.  There may be a better gtk way to do this.
        spellSelectTable.set_homogeneous(True)

        spellSelectTable.attach(gtk.Label(_("Left spell:")), 0,1, 0,1, yoptions=0)
        leftSpellSelectComboBox = gtk.combo_box_new_text()
        spellSelectTable.attach(leftSpellSelectComboBox, 1,2, 0,1, yoptions=0)

        spellSelectTable.attach(gtk.Label(_("Right spell")), 0,1, 1,2, yoptions=0)
        rightSpellSelectComboBox = gtk.combo_box_new_text()
        spellSelectTable.attach(rightSpellSelectComboBox, 1,2, 1,2, yoptions=0)

        spellSelectTable.attach(gtk.HSeparator(), 0,2, 2,3)

        spellSelectButton = gtk.Button(_("Send spells"))
        spellSelectTable.attach(spellSelectButton, 1,2, 3,4, yoptions=0)
        spellSelectButton.connect("clicked", self.on_spellSelectButton_clicked)

        spellSelectTable.set_sensitive(False)

        self.leftSpellSelectComboBox = leftSpellSelectComboBox
        self.rightSpellSelectComboBox = rightSpellSelectComboBox
        self.myInputTabs.addInputTab(spellSelectTable, _("Spells"))


    ## --- Enter spell targets
    def addSpellTargetSelectTab(self):
        spellTargetSelectTable = gtk.Table()
        spellTargetSelectTable.set_row_spacings(5)
        spellTargetSelectTable.set_col_spacings(5)
        spellTargetSelectTable.set_border_width(5)
        spellTargetSelectTable.set_homogeneous(True)

        leftSpellTargetSelectLabel = gtk.Label(_("Left spell target:"))
        spellTargetSelectTable.attach(leftSpellTargetSelectLabel, 0,1, 0,1, yoptions=0)
        leftSpellTargetSelectComboBox = gtk.combo_box_new_text()
        spellTargetSelectTable.attach(leftSpellTargetSelectComboBox, 1,2, 0,1, yoptions=0)

        rightSpellTargetSelectLabel = gtk.Label(_("Right spell target:"))
        spellTargetSelectTable.attach(rightSpellTargetSelectLabel, 0,1, 1,2, yoptions=0)
        rightSpellTargetSelectComboBox = gtk.combo_box_new_text()
        spellTargetSelectTable.attach(rightSpellTargetSelectComboBox, 1,2, 1,2, yoptions=0)

        spellTargetSelectTable.attach(gtk.HSeparator(), 0,2, 2,3)

        spellTargetSelectButton = gtk.Button(_("Send targets"))
        spellTargetSelectTable.attach(spellTargetSelectButton, 1,2, 3,4, yoptions=0)
        spellTargetSelectButton.connect("clicked", self.on_spellTargetSelectButton_clicked)

        spellTargetSelectTable.set_sensitive(False)

        self.leftSpellTargetSelectLabel = leftSpellTargetSelectLabel
        self.leftSpellTargetSelectComboBox = leftSpellTargetSelectComboBox
        self.rightSpellTargetSelectLabel = rightSpellTargetSelectLabel
        self.rightSpellTargetSelectComboBox = rightSpellTargetSelectComboBox

        self.myInputTabs.addInputTab(spellTargetSelectTable, _("Targets"))


    ## --- Enter monster directions
    def addMonsterDirectionsTab(self):
        monsterDirectionsTable = gtk.Table()
        monsterDirectionsTable.set_border_width(5)
        monsterDirectionsTable.set_row_spacings(5)
        monsterDirectionsTable.set_col_spacings(5)

        monsterDirectionsButton = gtk.Button(_("Send directions"))
        monsterDirectionsButton.connect("clicked", self.on_monsterDirectionsButton_clicked)

        self.monsterDirectionsTable = monsterDirectionsTable
        self.monsterDirectionsButton = monsterDirectionsButton

        self.myInputTabs.addInputTab(self.monsterDirectionsTable, _("Monster directions"))


    def addHandsAndGesturesTab(self):
        ## --- Hands and gestures.
        handsAndGesturesTable = gtk.Table()
        handsAndGesturesTable.set_border_width(5)
        handsAndGesturesTable.set_row_spacings(5)
        handsAndGesturesTable.set_col_spacings(5)

        ## --- MSG_ASK_CHARM_PERSON_CTRL_HAND
        charmPersonControlHandFrame = gtk.Frame(_("Charmed hand"))
        charmPersonControlHandTable = gtk.Table()
        charmPersonControlHandTable.set_border_width(5)
        charmPersonControlHandTable.set_row_spacings(5)
        charmPersonControlHandTable.set_col_spacings(5)

        t = gtk.Table()
        charmPersonControlHandNameLabel = gtk.Label(_("Target:"))
        t.attach(charmPersonControlHandNameLabel, 0,1, 0,1, xoptions=0, yoptions=0)
        t.attach(gtk.HBox(), 1,2, 0,1)
        charmPersonControlHandTable.attach(t, 0,1, 0,1)

        hbox = gtk.HBox()
        charmPersonControlHandLeftButton = gtk.RadioButton(None, _("Left"))
        hbox.pack_start(charmPersonControlHandLeftButton)
        charmPersonControlHandRightButton = gtk.RadioButton(charmPersonControlHandLeftButton, _("Right"))
        hbox.pack_start(charmPersonControlHandRightButton)
        charmPersonControlHandTable.attach(hbox, 1,2, 0,1)

        charmPersonControlHandButton = gtk.Button(_("Send"))
        charmPersonControlHandButton.connect("clicked", self.on_charmPersonControlHandButton_clicked)
        charmPersonControlHandTable.attach(charmPersonControlHandButton, 2,3, 0,1, xoptions=0, yoptions=0)

        charmPersonControlHandFrame.add(charmPersonControlHandTable)
        handsAndGesturesTable.attach(charmPersonControlHandFrame, 0,1, 0,1)

        self.charmPersonControlHandNameLabel = charmPersonControlHandNameLabel
        self.charmPersonControlHandLeftButton = charmPersonControlHandLeftButton
        self.charmPersonControlHandRightButton = charmPersonControlHandRightButton
        self.charmPersonControlHandWidget = charmPersonControlHandFrame
        # Be ready.  If the server times out waiting for us to reply to which
        # hand to control, this would never get set, and then we get an
        # exception.  Bad.
        self.charmPersonControlHandHand = 1

        ## --- MSG_ASK_CHARM_PERSON_CTRL_GESTURE
        charmPersonControlGestureFrame = gtk.Frame(_("Charmed gesture"))
        charmPersonControlGestureTable = gtk.Table()
        charmPersonControlGestureTable.set_border_width(5)
        charmPersonControlGestureTable.set_row_spacings(5)
        charmPersonControlGestureTable.set_col_spacings(5)

        t = gtk.Table()
        charmPersonControlGestureNameLabel = gtk.Label(_("Target:"))
        t.attach(charmPersonControlGestureNameLabel, 0,1, 0,1, xoptions=0, yoptions=0)
        t.attach(gtk.HBox(), 1,2, 0,1)
        charmPersonControlGestureTable.attach(t, 0,1, 0,1)

        t = gtk.Table()
        t.attach(gtk.HBox(), 0,1, 0,1)
        charmPersonControlGestureImage = GestureImage(selectable=True)
        charmPersonControlGestureImage.setActive(True)
        t.attach(charmPersonControlGestureImage, 1,2, 0,1, xoptions=0)
        t.attach(gtk.HBox(), 2,3, 0,1)
        charmPersonControlGestureTable.attach(t, 1,2, 0,1)

        charmPersonControlGestureButton = gtk.Button(_("Send"))
        charmPersonControlGestureButton.connect("clicked", self.on_charmPersonControlGestureButton_clicked)
        charmPersonControlGestureTable.attach(charmPersonControlGestureButton, 2,3, 0,1, xoptions=0, yoptions=0)

        charmPersonControlGestureFrame.add(charmPersonControlGestureTable)
        handsAndGesturesTable.attach(charmPersonControlGestureFrame, 0,1, 1,2)

        self.charmPersonControlGestureNameLabel = charmPersonControlGestureNameLabel
        self.charmPersonControlGestureImage = charmPersonControlGestureImage
        self.charmPersonControlGestureWidget = charmPersonControlGestureFrame
        self.charmPersonControlGestureImage.setActive = charmPersonControlGestureImage

        ## --- MSG_ASK_PARALYSIS_CTRL_GESTURE
        paralysisControlHandFrame = gtk.Frame(_("Paralysed hand"))
        paralysisControlHandTable = gtk.Table()
        paralysisControlHandTable.set_border_width(5)
        paralysisControlHandTable.set_row_spacings(5)
        paralysisControlHandTable.set_col_spacings(5)

        t = gtk.Table()
        paralysisControlHandNameLabel = gtk.Label(_("Target:"))
        t.attach(paralysisControlHandNameLabel, 0,1, 0,1, xoptions=0, yoptions=0)
        t.attach(gtk.HBox(), 1,2, 0,1)
        paralysisControlHandTable.attach(t, 0,1, 0,1)

        hbox = gtk.HBox()
        paralysisControlHandLeftButton = gtk.RadioButton(None, _("Left"))
        hbox.pack_start(paralysisControlHandLeftButton)
        paralysisControlHandRightButton = gtk.RadioButton(paralysisControlHandLeftButton, _("Right"))
        hbox.pack_start(paralysisControlHandRightButton)
        paralysisControlHandTable.attach(hbox, 1,2, 0,1)

        paralysisControlHandButton = gtk.Button(_("Send"))
        paralysisControlHandButton.connect("clicked", self.on_paralysisControlHandButton_clicked)
        paralysisControlHandTable.attach(paralysisControlHandButton, 2,3, 0,1, xoptions=0, yoptions=0)

        paralysisControlHandFrame.add(paralysisControlHandTable)
        handsAndGesturesTable.attach(paralysisControlHandFrame, 0,1, 2,3)
        self.paralysisControlHandWidget = paralysisControlHandFrame

        ## ---
        handsAndGesturesTable.set_sensitive(False)
        self.handsAndGesturesTable = handsAndGesturesTable
        self.myInputTabs.addInputTab(self.handsAndGesturesTable, _("Hands and gestures"))


    ### ---------- MainWindow: player manipulation.

    def showNewPlayer(self, newPlayer, position):
        global creaturesWidgetsTable, monsterContainers
        # This allows us to put the player's monsters just after the player.
        realPosition = position * 2
        creaturesWidgetsTable.attach(newPlayer.getWidget(), realPosition, realPosition+1, 0,1, xpadding=5, ypadding=5)
        monsterContainer = gtk.HBox()
        creaturesWidgetsTable.attach(monsterContainer, realPosition+1, realPosition+2, 0,1, xpadding=5, ypadding=5)
        monsterContainers[newPlayer.getNumber()] = monsterContainer
        creaturesWidgetsTable.show_all()
        newPlayer.setGettingGestures(0)
        self.show_all()


    def showNewMonster(self, newMonster, owner, ownerPosition):
        d("Adding a new monster: %s" % str(newMonster))
        monsterContainer = monsterContainers[owner.getNumber()]
        monsterContainer.pack_start(newMonster.getWidget(), padding=5)
        monsterContainer.show_all()


    def removePlayer(self, player):
        d("Removing widget for %s" % player.getName())
        global monsterContainers
        monsterContainer = monsterContainers[player.getNumber()]
        for child in monsterContainer.get_children():
            monsterContainer.remove(child)
        del monsterContainers[player.getNumber()]
        creaturesWidgetsTable.remove(player.getWidget())


    def setPlayerNumber(self, player, playerNumber):
        global creaturesWidgetsTable
        creaturesWidgetsTable.attach(player, 10+playerNumber*10, 10+playerNumber*10, 10,20)
        global creaturesNumbersDictionary
        creaturesNumbersDictionary[playerNumber] = player


    reconnect = True

    def messageReadingTimeout(self):
        try:
            # Handle up to 5 messages.  This is an arbitrary number to make us
            # respond a bit faster, but to not lock up the gui too long.
            for zz in (0,1,2,3,4):
                if self.connection is None:
                    return True
                msg = self.connection.readMessage()

                if msg is None:
                    return True
                d("\n<<< msg:%s" % str(msg))

                for ii in range(0,5): statusBar2.pop(0)
                statusBar2.push(0, msg.getMessageName())

                self.handleMessage(msg)
        except Exception, reason:
            # We need to handle all exceptions here.  If we don't, an exception
            # results in this function returning None, and it gets removed from
            # the timeout list, so we don't see any more game messages.
            ei = sys.exc_info()
            traceback.print_tb(ei[2])
            print >>sys.stderr, str(ei[0])
            print >>sys.stderr, str(ei[1])
            sys.exc_clear()
            self.connection = None
            statusBar.pop(0)
            statusBar.push(0, "Not connected")
            showTimedStatusBarMessage(5, str(reason.args))
            self.myInputTabs.setActive(None)
            if myPlayer is not None:
                myPlayer.setGettingGestures(False)
            self.on_gameConnect_activate(None)

        # Must return true to stay on the timeout callback list.
        return True


    def set_property(self, name, value):
        if name == 'non-existent-property':
            d("Setting non-existent-property to %s" % str(value))
        else:
            gtk.Window.set_property(self, name, value)


    def gestureHandler(self, gestures):
        '''This is a callback (from Player.py) that is called when the user has
        finished inputting the gestures for the next move.

        This part is fairly ugly architecturally, and needs to be redone.'''

        myPlayer.setGettingGestures(False)
        reply = SpellcastMessage(msgtype=MSG_RCV_GESTURES_USED)
        gestures = myPlayer.getGestures()
        reply['leftGesture'] = gestures[0]
        reply['rightGesture'] = gestures[1]
        d("Sending message %s" % str(reply))
        self.sendMessage(reply)


    def showTextMessage(self, text):
        buf = self.textMessagesView.get_buffer()
        it = buf.get_end_iter()
        buf.insert(it, text)
        buf.insert(it, "\n")
        # I tried getting the TextView to scroll to a mark at its end, but that
        # doesn't work, probably because it's inside a Scrolled Window, and
        # doesn't really know what part of itself is being exposed by the
        # ScrolledWindow.  So, manipulate the ScrolledWindow's vertical
        # scrollbar instead.
        va = self.textMessagesScrolledWindow.get_vadjustment()
        va.set_value(va.upper)


    ### ---------- MainWindow: GUI event handling.


    def on_charmPersonControlHandButton_clicked(self, button):
        if self.charmPersonControlHandLeftButton.get_active():
            hand = 1
        else:
            hand = 2
        reply = SpellcastMessage(msgtype=MSG_RCV_CHARM_PERSON_CTRL_HAND)
        reply['target'] = self.charmPersonControlHandTargetNumber
        reply['hand'] = hand
        self.charmPersonControlHandHand = hand
        self.myInputTabs.setActive(None)
        self.sendMessage(reply)


    def on_charmPersonControlGestureButton_clicked(self, button):
        reply = SpellcastMessage(msgtype=MSG_RCV_CHARM_PERSON_CTRL_GESTURE)
        reply['target'] = self.charmPersonControlGestureTargetNumber
        reply['gesture'] = self.charmPersonControlGestureImage.getGestureNumber()
        self.myInputTabs.setActive(None)
        self.sendMessage(reply)


    def on_paralysisControlHandButton_clicked(self, button):
        if self.paralysisControlHandLeftButton.get_active():
            hand = 1
        else:
            hand = 2
        reply = SpellcastMessage(msgtype=MSG_RCV_PARALYSIS_CTRL_HAND)
        reply['target'] = self.paralysisControlHandTargetNumber
        reply['hand'] = hand
        self.myInputTabs.setActive(None)
        self.sendMessage(reply)


    def on_helpAbout_activate(self, w):
        global statusBar
        # AboutDialog is in pygtk 2.6 (and above).
        try:
            xxx = gtk.AboutDialog
        except:
            # This is an earlier pygtk version, so use our own dialog.
            helpAboutDialog = HelpAboutDialog(self)
        else:
            helpAboutDialog = gtk.AboutDialog()
            helpAboutDialog.set_name(_("Ecar"))
            helpAboutDialog.set_version("0.1")
            helpAboutDialog.set_comments(_("A spellcast client for the LCA 2005 hackfest"))
            helpAboutDialog.set_website("http://lca2005.linux.org.au/hackfest/\nhttp://adelie.cx/hackfest/")
            helpAboutDialog.set_authors( ("Russell Steicke <russells@adelie.cx>",) )
            helpAboutDialog.set_artists( ("Russell", "Christopher", "Erin") )
            helpAboutDialog.set_license(licenceText)

        helpAboutDialog.run()
        helpAboutDialog.destroy()


    def checkGameConnectResponses(self, askConnectionDialog):
        """Check the validity of responses from the connection dialog box."""
        responses = ['', -1, '']
        responses[0] = askConnectionDialog.getHostEntryText()
        if len(responses[0]) == 0:
            return _("Need a host name")
        try:
            responses[1] = int(askConnectionDialog.getPortEntryText())
        except ValueError:
            return _("Port number must be an integer")
        if responses[1] <= 0  or  responses[1] > 65535:
            return _("Port number must be 1 to 65535")
        responses[2] = askConnectionDialog.getNameEntryText()
        if len(responses[2]) == 0:
            return _("Need a player name")
        return responses


    def on_gameConnect_activate(self, item):

        global serverHost, serverPort, myPlayerName
        askConnectionDialog = AskConnectionDialog(self, serverHost, serverPort, myPlayerName)
        while 1:
            response = askConnectionDialog.run()
            if response != gtk.RESPONSE_ACCEPT:
                askConnectionDialog.destroy()
                return True
            responses = self.checkGameConnectResponses(askConnectionDialog)
            if type(responses) is StringType:
                d("Bad data")
                d = gtk.Dialog(_("Bad data"), self,
                               gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                               (gtk.STOCK_OK, gtk.RESPONSE_OK))
                hb = gtk.HBox()
                hb.pack_start(gtk.Label(responses), padding=10)
                icon = gtk.Image()
                icon.set_from_stock(gtk.STOCK_DIALOG_ERROR, gtk.ICON_SIZE_DIALOG)
                hb.pack_start(icon, padding=10)
                d.vbox.add(hb)
                d.vbox.show_all()
                d.connect("response", lambda d: d.destroy())
                dr = d.run()
                d.destroy()
            else:
                serverHost, serverPort, myPlayerName = responses
                break
        askConnectionDialog.destroy()
        statusBar.push(0, _("Connecting to %s:%d..." % (serverHost,serverPort)))
        self.set_title("%s - Spellcast" % myPlayerName)
        self.connection = SpellcastConnection(serverHost, serverPort)
        try:
            self.connection.connect()
        except socket.error, reason:
            statusBar.pop(0)
            statusBar.push(0, reason[1])
            return True
        else:
            statusBar.pop(0)
            statusBar.push(0, _("Connected to %s:%d") % (serverHost,serverPort))


    def on_gameDisconnect_activate(self, item):
        statusBar.push(0, _("Disconnecting..."))
        self.connection.disconnect()
        statusBar.pop(0)
        statusBar.push(0, _("Disconnected"))


    def on_delete_event(self, w, e):
        askQuitDialog = AskQuitDialog(self)
        response = askQuitDialog.run()
        askQuitDialog.destroy()
        if response == gtk.RESPONSE_ACCEPT:
            gtk.main_quit()
        else:
            return True


    def on_spellSelectButton_clicked(self, button):
        '''Reply to the request for spells.'''
        d("spell select button clicked")
        reply = SpellcastMessage(msgtype=MSG_RCV_SPELLS_CAST)

        if len(self.leftSpells) == 1:
            reply['leftSpell'] = 0
        else:
            leftActive = self.leftSpellSelectComboBox.get_active()
            self.leftSpell = self.leftSpells[leftActive]
            reply['leftSpell'] = self.leftSpell

        if len(self.rightSpells) == 1:
            reply['rightSpell'] = 0
        else:
            rightActive = self.rightSpellSelectComboBox.get_active()
            self.rightSpell = self.rightSpells[rightActive]
            reply['rightSpell'] = self.rightSpell

        d("Sending left=%d right=%d" % (reply['leftSpell'], reply['rightSpell']))

        self.myInputTabs.setActive(None)
        self.sendMessage(reply)


    def on_spellTargetSelectButton_clicked(self, button):
        '''Reply to the request for targets.'''
        d("spell target select button clicked")
        reply = SpellcastMessage(msgtype=MSG_RCV_SPELL_DIRECTIONS)

        if len(self.leftSpellTargets) == 0:
            reply['leftTarget'] = 0
        else:
            leftActive = self.leftSpellTargetSelectComboBox.get_active()
            if leftActive == 0  or  leftActive == -1:
                reply['leftTarget'] = 0
            else:
                target = self.leftSpellTargets[leftActive]
                if target is not None:
                    reply['leftTarget'] = target
                else:
                    reply['rightTarget'] = 0

        if len(self.rightSpellTargets) == 0:
            reply['rightTarget'] = 0
        else:
            rightActive = self.rightSpellTargetSelectComboBox.get_active()
            if rightActive == 0  or  rightActive == -1:
                reply['rightTarget'] = 0
            else:
                target = self.rightSpellTargets[rightActive]
                if target is not None:
                    reply['rightTarget'] = target
                else:
                    reply['rightTarget'] = 0

        self.sendMessage(reply)
        self.myInputTabs.setActive(None)


    def on_monsterDirectionsButton_clicked(self, button):
        '''Send the monster directions.'''
        monsterDirections = []
        for i in range(len(self.monsterIndexes)):
            monsterNumber = self.monsterIndexes[i]
            comboboxindex = self.targetComboBoxes[i].get_active()
            targetNumber = self.targetIndexes[comboboxindex]
            monsterDirections.append( (monsterNumber, targetNumber) )
        reply = SpellcastMessage(msgtype=MSG_RCV_MONSTER_DIRECTIONS)
        reply['monsterDirections'] = monsterDirections
        self.myInputTabs.setActive(None)
        self.sendMessage(reply)


    ### --------- MainWindow: game message handling.

    def handleMessage(self, msg):
        m = msg.msgtype
        if   m == MSG_ASK_WELCOME:                   self.handle_MSG_ASK_WELCOME(msg)
        elif m == MSG_ASK_FOR_GESTURES:              self.handle_MSG_ASK_FOR_GESTURES(msg)
        elif m == MSG_SEND_NEWPLAYER_INFO:           self.handle_MSG_SEND_NEWPLAYER_INFO(msg)
        elif m == MSG_SEND_CREATURE_STATE:           self.handle_MSG_SEND_CREATURE_STATE(msg)
        elif m == MSG_ASK_FOR_SPELLS_CAST:           self.handle_MSG_ASK_FOR_SPELLS_CAST(msg)
        elif m == MSG_ASK_SPELL_DIRECTIONS:          self.handle_MSG_ASK_SPELL_DIRECTIONS(msg)
        elif m == MSG_SEND_GESTURES_SEEN:            self.handle_MSG_SEND_GESTURES_SEEN(msg)
        elif m == MSG_SEND_SPELL_CAST:               self.handle_MSG_SEND_SPELL_CAST(msg)
        elif m == MSG_SEND_END_GAME:                 self.handle_MSG_SEND_END_GAME(msg)
        elif m == MSG_SEND_EVENT_INFO:               self.handle_MSG_SEND_EVENT_INFO(msg)
        elif m == MSG_SEND_NEW_MONSTER_INFO:         self.handle_MSG_SEND_NEW_MONSTER_INFO(msg)
        elif m == MSG_SEND_MONSTER_ATTACK_INFO:      self.handle_MSG_SEND_MONSTER_ATTACK_INFO(msg)
        elif m == MSG_ASK_MONSTER_DIRECTIONS:        self.handle_MSG_ASK_MONSTER_DIRECTIONS(msg)
        elif m == MSG_SEND_START_GAME:               self.handle_MSG_SEND_START_GAME(msg)
        elif m == MSG_SEND_ROUND_BEGIN:              self.handle_MSG_SEND_ROUND_BEGIN(msg)
        elif m == MSG_USERNAME_IN_USE_ALREADY:       self.handle_MSG_USERNAME_IN_USE_ALREADY(msg)
        elif m == MSG_ASK_CHARM_PERSON_CTRL_HAND:    self.handle_MSG_ASK_CHARM_PERSON_CTRL_HAND(msg)
        elif m == MSG_ASK_PARALYSIS_CTRL_HAND:       self.handle_MSG_ASK_PARALYSIS_CTRL_HAND(msg)
        elif m == MSG_ASK_CHARM_PERSON_CTRL_GESTURE: self.handle_MSG_ASK_CHARM_PERSON_CTRL_GESTURE(msg)
        else:
            statusBar2.pop(0)
            statusBar2.push(0, "U: %s" % msg.getMessageName())


    def handle_MSG_ASK_WELCOME(self, msg):
        self.showTextMessage('"'+msg['welcomeMessage']+'"')
        reply = SpellcastMessage(msgtype=MSG_RCV_CLIENT_DETAILS)
        reply['clientDetails'] = myPlayerName
        d("Sending message %s" % str(reply))
        self.sendMessage(reply)
        # Just to be sure.
        self.myInputTabs.setActive(None)


    def handle_MSG_ASK_FOR_GESTURES(self, msg):
        self.myInputTabs.setActive(None)
        myPlayer.setGettingGestures(1)
            #reply = SpellcastMessage(msgtype=MSG_RCV_GESTURES_USED)
            #reply['leftGesture'] = GST_KNIFE
            #reply['rightGesture'] = GST_NOTHING
            #self.sendMessage(reply)


    def handle_MSG_SEND_START_GAME(self, msg):
        global myPlayerNumber, turnTimeout
        myPlayerNumber = msg['myPlayerID']
        turnTimeout = msg['turnTimeout']
        self.showTextMessage("----------")
        self.showTextMessage(_("The game begins!"))
        # Clean up the window and clear the player dictionaries (get rid of the
        # players).  This is mainly because more than one game can be played in
        # a session.
        global creaturesNamesDictionary, creaturesNumbersDictionary
        for creatureNumber in creaturesNumbersDictionary.keys():
            creature = creaturesNumbersDictionary[creatureNumber]
            if isinstance(creature, Player):
                # The player's monsters get removed with it.
                self.removePlayer(creature)
        creaturesNamesDictionary.clear()
        creaturesNumbersDictionary.clear()
        self.myInputTabs.setActive(None)


    def handle_MSG_SEND_NEWPLAYER_INFO(self, msg):
        global myPlayerNumber, myPlayer, myPlayerName
        global creaturesNumbersDictionary, creaturesNamesDictionary
        playerNumber = msg['playerNumber']
        playerName = msg['playerName']
        d(_("Got player %d: '%s': I am '%s'") % (playerNumber, playerName, myPlayerName))
        newPlayer = Player(playerName)
        creaturesNumbersDictionary[playerNumber] = newPlayer
        creaturesNamesDictionary[playerName] = newPlayer
        newPlayer.setNumber(playerNumber)
        # We can add our own player at postion 0 safely, because all the players
        # and other creatures have IDs starting at 1.  When the list comes back,
        # it's guaranteed that there won't be a player 0, therefore our player
        # will be at the left hand side of the group of players which is exactly
        # where we want him.  If our player does not have ID=1, there will be a
        # logical gap in the sequence, but the HBox containing the player
        # widgets will squash it down to zero width, so it's not a problem.
        if playerNumber == myPlayerNumber:
            myPlayer = newPlayer
            myPlayerName = playerName
            self.showNewPlayer(newPlayer, 0)
            self.showTextMessage(_("You advance confidently into the arena."))
            # When the user has input all the required gestures, we need to call
            # back here to send a message back to the server.
            myPlayer.setGestureHandler(self.gestureHandler)
        else:
            self.showNewPlayer(newPlayer, playerNumber)
            self.showTextMessage(_("%s advances confidently into the arena.") % playerName)
            newPlayer.gesturesSeen(new=GST_FOG)


    def handle_MSG_SEND_CREATURE_STATE(self, msg):
        creatureNumber = msg['creatureNumber']
        if creatureNumber != 0:
            creatureHealth = msg['health']
            creatureState = msg['state']
            creatureStateDescription = msg['stateDescription']
            d("Setting creature %d: health %d: state %d" % (creatureNumber, creatureHealth, creatureState))
            creaturesNumbersDictionary[creatureNumber].setHealth(creatureHealth)
            creaturesNumbersDictionary[creatureNumber].setState(creatureState)


    def handle_MSG_ASK_FOR_SPELLS_CAST(self, msg):
        '''We have been asked what spells are to be cast by one or both hands.'''
        nLeftSpells = msg['nLeftSpells']
        leftSpells = msg['leftSpells']
        nRightSpells = msg['nRightSpells']
        rightSpells = msg['rightSpells']
        d("Asked about %d possible left hand spells" % nLeftSpells)
        if nLeftSpells > 0:
            d(str(leftSpells))
        d("Asked about %d possible right hand spells" % nRightSpells)
        if nRightSpells > 0:
            d(str(rightSpells))

        # Clear the combobox
        for _z in range(10):
            self.leftSpellSelectComboBox.remove_text(0)
        # Add the current selections
        self.leftSpells = [0]
        if nLeftSpells > 0:
            self.leftSpellSelectComboBox.append_text(_("None"))
            for spell in leftSpells:
                self.leftSpells.append(spell)
                self.leftSpellSelectComboBox.append_text(spellTypeToName(spell))
            self.leftSpellSelectComboBox.set_sensitive(True)
            self.leftSpellSelectComboBox.set_active(1)
            self.leftSpell = self.leftSpells[1]
        else:
            self.leftSpellSelectComboBox.set_sensitive(False)

        for _z in range(10):
            self.rightSpellSelectComboBox.remove_text(0)
        self.rightSpells = [0]
        if nRightSpells > 0:
            self.rightSpellSelectComboBox.append_text(_("None"))
            for spell in rightSpells:
                self.rightSpells.append(spell)
                self.rightSpellSelectComboBox.append_text(spellTypeToName(spell))
            self.rightSpellSelectComboBox.set_sensitive(True)
            self.rightSpellSelectComboBox.set_active(1)
            self.rightSpell = self.rightSpells[1]
        else:
            self.rightSpellSelectComboBox.set_sensitive(False)

        self.myInputTabs.setActive("Spells")


    def handle_MSG_ASK_SPELL_DIRECTIONS(self, msg):
        '''We have been asked where we will be casting our spells.'''
        leftSpell = msg['leftSpell']
        rightSpell = msg['rightSpell']
        targets = msg['targets']
        d("left=%s right=%s targets=%s" % (str(leftSpell), str(rightSpell), str(targets)))

        # Remove the maximum number of text items that can be in the list.  This
        # doesn't seem to generate an error if the combobox is empty and we keep
        # removing things, which is nice in this case.
        for _z in range(len(creaturesNumbersDictionary)+1):
            self.leftSpellTargetSelectComboBox.remove_text(0)
        self.leftSpellTargets = []
        if leftSpell:
            self.leftSpellTargets.append(None)
            self.leftSpellTargetSelectComboBox.append_text(_("None"))
            for target in targets:
                self.leftSpellTargetSelectComboBox.append_text(creaturesNumbersDictionary[target].getName())
                self.leftSpellTargets.append(target)
            self.leftSpellTargetSelectComboBox.set_active(self.suggestTarget(self.leftSpellTargets, leftSpell))
            self.leftSpellTargetSelectComboBox.set_sensitive(True)
            self.leftSpellTargetSelectLabel.set_text(_("Left target (%s):") % spellTypeToName(self.leftSpell))
        else:
            self.leftSpellTargetSelectComboBox.set_sensitive(False)
            self.leftSpellTargetSelectLabel.set_text(_("Left target:"))

        for _z in range(len(creaturesNumbersDictionary)+1):
            self.rightSpellTargetSelectComboBox.remove_text(0)
        self.rightSpellTargets = []
        if rightSpell:
            self.rightSpellTargets.append(None)
            self.rightSpellTargetSelectComboBox.append_text(_("None"))
            for target in targets:
                self.rightSpellTargetSelectComboBox.append_text(creaturesNumbersDictionary[target].getName())
                self.rightSpellTargets.append(target)
            self.rightSpellTargetSelectComboBox.set_active(self.suggestTarget(self.rightSpellTargets, rightSpell))
            self.rightSpellTargetSelectComboBox.set_sensitive(True)
            self.rightSpellTargetSelectLabel.set_text(_("Right target (%s):") % spellTypeToName(self.rightSpell))
        else:
            self.rightSpellTargetSelectComboBox.set_sensitive(False)
            self.rightSpellTargetSelectLabel.set_text(_("Right target:"))

        # Make the whole of the spell selection frame sensitive at the one time.
        self.myInputTabs.setActive("Targets")


    def suggestTarget(self, targets, spell):
        '''Suggest a target for a spell.  Tries to make offensive spells point
        to other players by default, and defensive spells point to our own
        player by default.'''
        if isOffensiveSpell(spell):
            d("Spell %s is offensive" % spellTypeToName(spell))
            for index in range(len(targets)):
                d("index==%d playerNumber==%s" % (index, str(targets[index])))
                if targets[index] is not None and targets[index] != myPlayerNumber:
                    d("Returning %s" % str(index))
                    return index
        elif isDefensiveSpell(spell):
            d("Spell %s is defensive" % spellTypeToName(spell))
            for index in range(len(targets)):
                d("index==%d playerNumber==%s" % (index, str(targets[index])))
                if targets[index] is not None and targets[index] == myPlayerNumber:
                    d("Returning %s" % str(index))
                    return index
        else:
            d("Returning 0")
            return 0


    def handle_MSG_SEND_GESTURES_SEEN(self, msg):
        playerNumber = msg['playerNumber']
        if playerNumber == 0:
            return
        global myPlayerNumber
        if playerNumber == myPlayerNumber:
            # when the server gives up waiting for us to send our gesture list,
            # it can start sending these messages without a response from us.
            # (This is probably full of race conditions.)
            myPlayer.setGettingGestures(0)
            newGesture = GST_NOTHING
        else:
            newGesture = GST_FOG
        leftGesture = msg['leftGesture']
        rightGesture = msg['rightGesture']
        player = creaturesNumbersDictionary[playerNumber]
        d("Player %s did %d %d" % (str(player), leftGesture, rightGesture))
        d("Player %s did %s %s" % (str(player),
                                   gestureTypeToName(leftGesture),
                                   gestureTypeToName(rightGesture)))
        player.gesturesSeen( next=(leftGesture,rightGesture), new=newGesture )
        self.myInputTabs.setActive(None)


    def handle_MSG_SEND_SPELL_CAST(self, msg):
        source = msg['source']
        if source == 0:
            return
        sourceName = creaturesNumbersDictionary[source].getName()
        target = msg['target']
        targetName = creaturesNumbersDictionary[target].getName()
        message = msg['message']
        spell = msg['spell']
        spellName = spellTypeToName(spell)
        spellWorked = msg['spellWorked']
        d('''source="%s" target="%s" spell="%s" spellWorked=%d message="%s"''' % \
          (sourceName, targetName, spellName, spellWorked, message))
        message = message.replace("%S", sourceName)
        message = message.replace("%T", targetName)
        summoned = isSummoningSpell(spell)
        if summoned is not None:
            message = message.replace("%M", summoned)
        self.showTextMessage(message)


    def handle_MSG_SEND_END_GAME(self, msg):
        self.myInputTabs.setActive(None)
        nWinners = msg['nWinners']
        winnerNumbers = msg['winners']
        d("nWinners=%s  winners=%s" % (nWinners, winnerNumbers))
        winnerNames = []
        for winnerNumber in winnerNumbers:
            if creaturesNumbersDictionary.has_key(winnerNumber):
                winnerNames.append(creaturesNumbersDictionary[winnerNumber].getName())
            else:
                winnerNames.append("?? %s" % winnerNumber)
        if nWinners == 0:
            messageText = _("Nobody won!")
            dialogText  = _("Nobody won!")
        elif nWinners == 1:
            messageText = _("The winner is %s.") % winnerNames[0]
            dialogText  = _("The winner is %s.") % winnerNames[0]
        elif nWinners == 2:
            messageText = _("The winners are\n%s and %s.") % (winnerNames[0], winnerNames[1])
            dialogText  = _("The winners are\n%s and %s.") % (winnerNames[0], winnerNames[1])
        else:
            messageText = _("The winners are") + "\n"
            dialogText  = _("The winners are") + "\n"
            for winnerNumber in range(nWinners-2):
                messageText = messageText + "%s, "  % winnerNames[winnerNumber]
                dialogText  = dialogText  + "%s,\n" % winnerNames[winnerNumber]
            messageText = messageText + "%s and %s."  % (winnerNames[nWinners-2], winnerNames[nWinners-1])
            dialogText  = dialogText  + "%s\nand %s." % (winnerNames[nWinners-2], winnerNames[nWinners-1])

        self.showTextMessage(messageText)

        if nWinners==1 or nWinners==0:
            title = _("Game Over")
        else:
            title = _("Game Over (draw)")

        dialog = gtk.Dialog(title, self, gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        hbox = gtk.HBox()
        hbox.set_border_width(5)
        hbox.set_spacing(5)
        hbox.pack_start(gtk.Label(dialogText))
        icon = gtk.Image()
        icon.set_from_file('spellcast_gui-reversed.png')
        hbox.pack_start(icon)
        dialog.vbox.pack_start(hbox)
        dialog.vbox.show_all()
        dialog.connect("response", lambda d,r: dialog.destroy())
        dialog.show()


    def handle_MSG_SEND_EVENT_INFO(self, msg):

        message = msg['message']
        messageSource = msg['messageSource']
        source = msg['source']
        if creaturesNumbersDictionary.has_key(source):
            sourceCreatureName = creaturesNumbersDictionary[source].getName()
        else:
            sourceCreatureName = "Creature %d" % source
        target = msg['target']
        if creaturesNumbersDictionary.has_key(target):
            targetCreatureName = creaturesNumbersDictionary[target].getName()
        else:
            targetCreatureName = "Creature %d" % target
        message = message.replace("%S", sourceCreatureName)
        message = message.replace("%T", targetCreatureName)
        message = message.replace("%M", str(messageSource))
        self.showTextMessage(message)


    def handle_MSG_SEND_NEW_MONSTER_INFO(self, msg):
        monsterID = msg['monsterID']
        monsterType = msg['monsterType']
        monsterTypeName = monsterTypeToName(monsterType)
        ownerID = msg['ownerID']
        monsterName = msg['monsterName']
        d("New monster: Id=%d Type=%d:%s Owner=%d Name=%s" % \
          (monsterID, monsterType, monsterTypeName, ownerID, monsterName))
        newMonster = ScMonster(monsterType, monsterName)
        newMonster.setOwner(ownerID)
        creaturesNumbersDictionary[monsterID] = newMonster
        creaturesNamesDictionary[monsterName] = newMonster
        if ownerID == myPlayerNumber:
            ownerPosition = 0
        else:
            ownerPosition = ownerID
        self.showNewMonster(newMonster, creaturesNumbersDictionary[ownerID], ownerPosition)


    def handle_MSG_SEND_MONSTER_ATTACK_INFO(self, msg):
        source = msg['source']
        if source != 0:
            sourceName = creaturesNumbersDictionary[source].getName()
            target = msg['target']
            targetName = creaturesNumbersDictionary[target].getName()
            damage = msg['damage']
            message = msg['message']
            message = message.replace("%S", sourceName)
            message = message.replace("%T", targetName)
            message = message.replace("%D", str(damage))
            self.showTextMessage(message)


    def handle_MSG_ASK_MONSTER_DIRECTIONS(self, msg):

        # The widgets for the monster directions are constructed on the fly
        # because the list of monsters can vary each time this questino is
        # asked, and it's just easier to do it like this.

        # First, clear the old children from the table.
        tableChildren = self.monsterDirectionsTable.get_children()
        for child in tableChildren:
            self.monsterDirectionsTable.remove(child)

        global creaturesNumbersDictionary

        # Now add new labels and such.
        monsterNumbers = msg['monsters']
        targetNumbers = msg['targets']
        # We need to know the names of the monsters to display to the user.
        monsterNames = []
        # And we need to keep a record of the monster indexes to send back to
        # the server.
        self.monsterIndexes = []
        for monsterNumber in monsterNumbers:
            mname = creaturesNumbersDictionary[monsterNumber].getName()
            mtypename = creaturesNumbersDictionary[monsterNumber].getTypeName()
            if mtypename[0] in 'aeiouAEIOU':
                wholename = "%s (an %s)" % (mname, mtypename)
            else:
                wholename = "%s (a %s)" % (mname, mtypename)
            monsterNames.append(wholename)
            self.monsterIndexes.append(monsterNumber)
        # A list of the target names.
        targetNames = []
        # A list of target numbers.  The 0 is here because the first name on
        # each target dropdown list will be "None", and 0 as a target number
        # means no target.
        self.targetIndexes = [0]
        for targetNumber in targetNumbers:
            targetNames.append(creaturesNumbersDictionary[targetNumber].getName())
            self.targetIndexes.append(targetNumber)

        # We have enough information now to construct the monster labels and
        # target comboboxes.
        self.targetComboBoxes = []
        for monsterIndex in range(len(monsterNames)):
            label = gtk.Label(monsterNames[monsterIndex])
            self.monsterDirectionsTable.attach(label, 0,1, monsterIndex,monsterIndex+1, yoptions=0)
            combobox = gtk.combo_box_new_text()
            combobox.append_text("None")
            for targetIndex in range(len(targetNames)):
                combobox.append_text(targetNames[targetIndex])
            # Set the default entry to the first target who is not me!
            global myPlayerNumber
            for targetIndex in range(1,len(targetNames)):
                if self.targetIndexes[targetIndex] != myPlayerNumber:
                    combobox.set_active(targetIndex)
                    break
            self.monsterDirectionsTable.attach(combobox, 1,2, monsterIndex,monsterIndex+1, yoptions=0)
            self.targetComboBoxes.append(combobox)
        comboBottom = len(monsterNames)
        self.monsterDirectionsTable.attach(gtk.HSeparator(), 0,2, comboBottom,1+comboBottom)
        self.monsterDirectionsTable.attach(self.monsterDirectionsButton, 1,2, 1+comboBottom,2+comboBottom, yoptions=0)
        self.monsterDirectionsTable.show_all()

        self.myInputTabs.setActive("Monster directions")


    def handle_MSG_USERNAME_IN_USE_ALREADY(self, msg):
        # Perhaps this should call a method that is also called by
        # on_gameConnect_activate, but this works just as well, mainly because
        # on_gameConnect_activate does not check its widget argument.
        self.on_gameConnect_activate(None)


    def handle_MSG_SEND_ROUND_BEGIN(self, msg):
        self.showTextMessage("-- Round %d..." % msg['roundNumber'])
        # Make all the controls inactive.  We then wait for the appropriate
        # messages to activate certain parts of the GUI that gather replies to
        # send back to the server.
        self.myInputTabs.setActive(None)


    def handle_MSG_ASK_CHARM_PERSON_CTRL_HAND(self, msg):
        targetNumber = msg['target']
        self.charmPersonControlHandTargetNumber = targetNumber
        targetName = creaturesNumbersDictionary[targetNumber].getName()
        self.charmPersonControlHandNameLabel.set_text(_("Target:") + targetName)
        self.myInputTabs.setActive(_("Hands and gestures"))
        gtk.main_iteration(False)
        self.charmPersonControlHandWidget.set_sensitive(True)
        self.paralysisControlHandWidget.set_sensitive(False)
        self.charmPersonControlGestureWidget.set_sensitive(False)


    def handle_MSG_ASK_PARALYSIS_CTRL_HAND(self, msg):
        targetNumber = msg['target']
        self.paralysisControlHandTargetNumber = targetNumber
        targetName = creaturesNumbersDictionary[targetNumber].getName()
        self.paralysisControlHandNameLabel.set_text(_("Target:") + targetName)
        self.myInputTabs.setActive(_("Hands and gestures"))
        self.charmPersonControlHandWidget.set_sensitive(False)
        self.paralysisControlHandWidget.set_sensitive(True)
        self.charmPersonControlGestureWidget.set_sensitive(False)


    def handle_MSG_ASK_CHARM_PERSON_CTRL_GESTURE(self, msg):
        targetNumber = msg['target']
        self.charmPersonControlGestureTargetNumber = targetNumber
        targetName = creaturesNumbersDictionary[targetNumber].getName()
        self.charmPersonControlGestureNameLabel.set_text(_("Target:") + targetName)
        self.myInputTabs.setActive(_("Hands and gestures"))
        self.charmPersonControlHandWidget.set_sensitive(False)
        self.paralysisControlHandWidget.set_sensitive(False)
        self.charmPersonControlGestureWidget.set_sensitive(True)
        if self.charmPersonControlHandHand == 1:
            self.charmPersonControlGestureImage.setRight(False)
        else:
            self.charmPersonControlGestureImage.setRight(True)
        self.charmPersonControlGestureImage.setGestureNumber(0)


    def sendMessage(self, msg):
        d("\n>>> msg: %s" % str(msg))
        self.connection.sendMessage(msg)


def setupGui():
    global mainWindow
    mainWindow = MainWindow()
    mainWindow.show_all()


if __name__=='__main__':
    setupGui()
    gtk.main()


# Local variables: ***
# mode:python ***
# py-indent-offset:4 ***
# fill-column:80 ***
# End: ***
### arch-tag: 7b48b2b6-56fe-4222-aba4-464deb2a81fa

