#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>

#include "protocol.h"

#include "gestures.h"
#include "spells.h"
#include "wizard.h"
#include "moves.h"
#include "message.h"

/* Turn on some sanity checks */
#define NOTDEBUG

/* Turn on printing out of various bits of state */
#define VERBOSE

typedef struct CreatureState {
  int         in_use;
  char        name[100];
  GestureList gl;
  int         hp;
  int         state;
  int         type;       // 0 if wizard
  int         owner_id;   // 0 if no owner (ie. a wizard)
  int         paralysed;
  int         prot_from_evil, disease, poison, invisible, blind;
} CreatureState;

/* map from list of wizards built up by rx_ask_for_gestures() to
 * the corresponding creature_ids
 */
int wizard_ids[8];
int num_wizards;

/* fd for the server we're connected to */
int server;

/* hostname to connect to, username to connect as */
char *hostname = NULL;
char *username = NULL;

CreatureState *creatures = NULL;
int num_creatures = 0;
int own_id;
int turn_timeout;
int move_num;
int best_move_move_num;
Move best_move;

static char *creature_types[7] = {
  "wizard",
  "goblin",
  "ogre",
  "troll",
  "giant",
  "fire elemental",
  "ice elemental"
};

void rx_ask_welcome(void *data) {
  assert(get_int(data) == SPELLCAST_PROTOCOL_VERSION);
  next_int(data);

  printf("Server said: %s\n", (char *)data);

  msg_send(server, MSG_RCV_CLIENT_DETAILS, username, strlen(username) + 1);
}

void set_monster_info(WizardState *ws, int wizard_creature_num) {
  int i;

  printf("set monster info for %s (%d)\n",
         creatures[wizard_creature_num].name, wizard_creature_num);
  for (i = 0; i < num_creatures; i++) {
    if (!creatures[i].in_use) {
      continue;
    }
    if (creatures[i].type && creatures[i].owner_id == wizard_creature_num) {
      int creature_category =
            creatures[i].type == SC_MONSTER_FIRE_ELEMENTAL
            ? 1 :
            creatures[i].type == SC_MONSTER_ICE_ELEMENTAL
            ? 2 :
            0;
      ws->monster_num[creature_category]++;

      /* elements have 3 max HP, other monsters happen to have
       * "type" max HP
       */
      ws->monster_max_hp[creature_category] +=
          creature_category == 0 ? creatures[i].type : 3;
      ws->monster_hp[creature_category] += creatures[i].hp;

      printf("found creature %s (%d).  sum(hp) is %d/%d\n",
             creatures[i].name, i,
             ws->monster_hp[creature_category],
             ws->monster_max_hp[creature_category]);
    }
  }
}

void set_wizard_info(WizardState *ws, int id) {
  ws->gl = creatures[id].gl;
  ws->state = creatures[id].state;
  ws->damage_inflicted = WIZARD_MAX_NUM_HIT_POINTS - creatures[id].hp;
  ws->num_hit_points = WIZARD_MAX_NUM_HIT_POINTS;
  ws->paralysed = creatures[id].paralysed;
  ws->prot_from_evil = creatures[id].prot_from_evil;
  ws->disease = creatures[id].disease;
  ws->poison = creatures[id].poison;
  ws->invisible = creatures[id].invisible;
  ws->blind = creatures[id].blind;
}

void run_find_move(void) {
  WizardState *wizards;
  int i, j;

  printf("run_find_move() called -- ");

  /* if we've already picked a move to make, don't do it again */
  if (best_move_move_num == move_num) {
    printf("move already calculated\n");
    return;
  }

  printf("need to calculate move\n");

  best_move_move_num = move_num;

#if 0
  if (move_num < 1) {
    best_move.gl = make_gp(GST_FINGER, GST_NOTHING);
    best_move.sp = 0;
    best_move.paralysis_hand = 0;
    best_move.score = 0;
    return;
  }
#endif

  num_wizards = 0;

  for (i = 0; i < num_creatures; i++) {
    if (creatures[i].in_use &&
        creatures[i].type == 0 &&
        !(creatures[i].state & CS_DEAD)) {
      num_wizards++;
    }
  }

  wizards = calloc(num_wizards, sizeof(WizardState));

  set_wizard_info(&wizards[0], own_id);
  set_monster_info(&wizards[0], own_id);
  wizard_ids[0] = own_id;

  j = 1;
  for (i = 0; i < num_creatures; i++) {
    if ((i == own_id) ||
        !creatures[i].in_use ||
        (creatures[i].type != 0) ||
        (creatures[i].state & CS_DEAD)) {
      continue;
    }
    set_wizard_info(&wizards[j], i);
    set_monster_info(&wizards[j], i);
    wizard_ids[j] = i;
    j++;
  }

  assert(j == num_wizards);

  find_move_id(wizards, num_wizards, turn_timeout, &best_move);

  for (i = 0; i < num_creatures; i++) {
    creatures[i].paralysed = 0;
  }

  free(wizards);
}

/* pack the info from creatures[] into wizards[]
 * our info goes into wizards[0], others go into wizards[1..]
 * monster info goes into wizards[n].monster_*
 */
void rx_ask_for_gestures(void *data) {
  int gesture_data[2];

  printf("rx_ask_for_gestures()\n");

  run_find_move();

  gesture_data[0] = htonl(LH(best_move.gl) & 0x0F);
  gesture_data[1] = htonl(RH(best_move.gl) & 0x0F);

  msg_send(server, MSG_RCV_GESTURES_USED, gesture_data, sizeof(gesture_data));
}

void rx_ask_for_spells_cast(void *data) {
  int num_spells;
  int spells[2] = {0, 0};
  int i;

  printf("rx_ask_for_spells_cast()\n");

  run_find_move();

  /* left hand spells */
  num_spells = get_int(data);
  next_int(data);

  if (num_spells) {
    for (i = 0; i < num_spells; i++) {
      if (get_int(data) == ((best_move.sp >> 16) & 0xFF)) {
        spells[0] = get_int(data);
      }
      next_int(data);
    }
    if (!spells[0]) {
      printf("oh-oh -- we wanted to cast %s with LH\n",
             spell_name[(best_move.sp >> 16) & 0xFF]);
    }
  }

  /* right hand spells */
  num_spells = get_int(data);
  next_int(data);

  if (num_spells) {
    for (i = 0; i < num_spells; i++) {
      if (get_int(data) == (best_move.sp & 0xFF)) {
        spells[1] = get_int(data);
      }
      next_int(data);
    }
    if (!spells[1]) {
      printf("oh-oh -- we wanted to cast %s with LH\n",
             spell_name[best_move.sp & 0xFF]);
    }
  }

  spells[0] = htonl(spells[0]);
  spells[1] = htonl(spells[1]);

  msg_send(server, MSG_RCV_SPELLS_CAST, spells, sizeof(spells));
}

void rx_send_gestures_seen(void *data) {
  int id, l, r;

  id = get_int(data) - 1;
  next_int(data);

  if (id == -1) {
    printf("end of creature gestures\n");
    return;
  }

  l = get_int(data);
  next_int(data);
  r = get_int(data);

  creatures[id].gl = gl_push(creatures[id].gl, make_gp(l, r));
  printf("gesture list for %s is now ", creatures[id].name);
  print_gesture_list(creatures[id].gl);
  printf("\n");
}

/* From spells.h:
 * 000mwwww
 * w: what wizard to apply the spell to (0: us, 1-15: someone else)
 * m: 0: apply to the wizard, 1: apply to one of their monsters
 */
int choose_spell_target(unsigned char target, int spell) {
  assert(target < num_wizards);

  int wizard_id = wizard_ids[target & 0x0F];
  int i;
  int found_monster_id, found_monster_max_hp, found_monster_hp;

  assert(wizard_id >= 0);
  assert(wizard_id < num_creatures);

  /* if target is a wizard */
  if (!(target & 0x10)) {
    return wizard_id;
  }

  found_monster_id = -1;

  /* target is a monster -- figure out which one we want to target
   * - attack bigger (more dangerous, bigger max_hp) monsters first
   * - if more than one biggest monster, attack the one with fewer HP
   *   left
   */
  for (i = 0; i < num_creatures; i++) {

    /* ignore:
     * - creature entries not in use
     * - wizards
     * - monsters not owned by the targeted wizards
     */
    if (!creatures[i].in_use ||
        creatures[i].type == 0 ||
        creatures[i].owner_id != wizard_id) {
      continue;
    }

    /* - only apply resist heat to fire elemental
     * - only apply resist cold to ice elemental
     * - only apply charm monster to normal elemental
     */
    if (((spell == SPL_CHARM_MONSTER) &&
         (creatures[i].type > SC_MONSTER_GIANT)) ||
        ((spell == SPL_RESIST_HEAT) &&
         (creatures[i].type != SC_MONSTER_FIRE_ELEMENTAL)) ||
        ((spell == SPL_RESIST_COLD) &&
         (creatures[i].type != SC_MONSTER_ICE_ELEMENTAL))) {
      continue;
    }

    /* figure the amount of damage this creature does per turn */
    int monster_max_hp =
        ((creatures[i].type == SC_MONSTER_FIRE_ELEMENTAL) ||
         (creatures[i].type == SC_MONSTER_ICE_ELEMENTAL))
        ? 3
        : creatures[i].type;

    /* first monster found? */
    if ((found_monster_id == -1) ||
        /* a more dangerous monster? */
        (monster_max_hp > found_monster_max_hp) ||
        /* fewer HP to destroy? */
        (creatures[i].hp < found_monster_hp)
    ) {
      found_monster_id = i;
      found_monster_max_hp = monster_max_hp;
      found_monster_hp = creatures[i].hp;
    }
  }

  return found_monster_id;
}

void rx_ask_spell_directions(void *data) {
  int l, r;
  int chosen_targets[2];

  printf("rx_ask_spell_directions()\n");

  run_find_move();

  l = get_int(data);
  next_int(data);

  assert(l >= 0 && l < SPL_FINAL_MARKER);

  if (l) {
    printf("Asked for directions on spell %s (LH)\n", spell_name[l]);
  }

  r = get_int(data);
  next_int(data);

  assert(r >= 0 && r < SPL_FINAL_MARKER);

  if (r) {
    printf("Asked for directions on spell %s (RH)\n", spell_name[r]);
  }

  /* we ignore the list of possible targets from the list as we
   * already know who we're going to apply it to, and if we pick
   * someone that's not on the list we can't do anything about it
   * anyway
   */

  if (l) {
    if (l == ((best_move.sp >> 16) & 0xFF)) {
      chosen_targets[0] = choose_spell_target(best_move.sp >> 24, l);
    } else {
      printf("oh-oh -- we thought the LH spell was %s\n",
             spell_name[(best_move.sp >> 16) & 0xFF]);
      chosen_targets[0] = -1;
    }
    printf("Applying LH spell to %s\n",
           (chosen_targets[0] == -1)
             ? "unknown :-("
             : creatures[chosen_targets[0]].name);
    chosen_targets[0]++;
  } else {
    if (best_move.sp >> 16) {
      printf("oh-oh -- we were expecting to be able to cast %s with LH\n",
             spell_name[(best_move.sp >> 16) & 0xFF]);
    }
    chosen_targets[0] = -1;
  }

  if (r) {
    if (r == (best_move.sp & 0xFF)) {
      chosen_targets[1] = choose_spell_target((best_move.sp >> 8) & 0xFF, r);
    } else {
      printf("oh-oh -- we thought the RH spell was %s\n",
             spell_name[best_move.sp & 0xFF]);
      chosen_targets[1] = -1;
    }
    printf("Applying RH spell to %s\n",
           (chosen_targets[1] == -1)
             ? "unknown :-("
             : creatures[chosen_targets[1]].name);
    chosen_targets[1]++;
  } else {
    if (best_move.sp & 0xFFFF) {
      printf("oh-oh -- we were expecting to be able to cast %s with RH\n",
             spell_name[best_move.sp & 0xFF]);
    }
    chosen_targets[1] = -1;
  }

  chosen_targets[0] = htonl(chosen_targets[0]);
  chosen_targets[1] = htonl(chosen_targets[1]);

  msg_send(server, MSG_RCV_SPELL_DIRECTIONS, chosen_targets, sizeof(int) * 2);
}

void rx_send_spell_cast(void *data) {
  int source_player;
  int target_creature;
  int spell_cast;
  int spell_worked;
  char  *result_message;

  source_player = get_int(data);
  next_int(data);

  if (!source_player) {
    printf("end of spells cast\n");
    return;
  }

  source_player--;
  assert(source_player >= 0);
  assert(source_player < num_creatures);

  target_creature = get_int(data) - 1;
  next_int(data);
  assert(target_creature >= 0);
  assert(target_creature < num_creatures);

  spell_cast = get_int(data);
  next_int(data);
  assert(spell_cast >= 0);
  assert(spell_cast < SPL_FINAL_MARKER);

  spell_worked = get_int(data);
  next_int(data);

  /* ignore string length */
  next_int(data);

  result_message = data;
  assert(result_message);

  printf("%s (%d) %s cast %s at %s (%d): %s\n",
         creatures[source_player].name, source_player,
         (spell_worked ? "successfully" : "unsuccessfully"),
         spell_name[spell_cast],
         creatures[target_creature].name, target_creature,
         result_message);
}

/* convert from SC_CS_* to CS_* */
int convert_state(int sc_state) {
  int state = 0;
  if (sc_state & SC_CS_DEAD)           state |= CS_DEAD;
  if (sc_state & SC_CS_SHIELD)         state |= CS_SHIELD;
  if (sc_state & SC_CS_AMNESIA)        state |= CS_AMNESIA;
  if (sc_state & SC_CS_CONFUSION)      state |= CS_CONFUSION;
  if (sc_state & SC_CS_CHARM_PERSON)   state |= CS_CHARM_PERSON;
  if (sc_state & SC_CS_PARALYSIS)      state |= CS_PARALYSIS;
  if (sc_state & SC_CS_FEAR)           state |= CS_FEAR;
  if (sc_state & SC_CS_ANTI_SPELL)     state |= CS_ANTI_SPELL;
  if (sc_state & SC_CS_PROT_FROM_EVIL) state |= CS_PROT_FROM_EVIL;
  if (sc_state & SC_CS_RESIST_HEAT)    state |= CS_RESIST_HEAT;
  if (sc_state & SC_CS_RESIST_COLD)    state |= CS_RESIST_COLD;
  if (sc_state & SC_CS_DISEASED)       state |= CS_DISEASED;
  if (sc_state & SC_CS_POISONED)       state |= CS_POISONED;
  if (sc_state & SC_CS_BLINDED)        state |= CS_BLINDED;
  if (sc_state & SC_CS_INVISIBILITY)   state |= CS_INVISIBILITY;
  printf("convert_state(%d) = %d\n", sc_state, state);
  return state;
}

void rx_send_creature_state(void *data) {
  int id;

  id = get_int(data) - 1;
  next_int(data);

  if (id == -1) {
    printf("end of creature states\n");
    return;
  }

  creatures[id].hp = get_int(data);
  next_int(data);
  creatures[id].state = convert_state(get_int(data));
  next_int(data);
  creatures[id].prot_from_evil = get_int(data);
  next_int(data);
  creatures[id].disease = get_int(data);
  next_int(data);
  creatures[id].poison = get_int(data);
  next_int(data);
  creatures[id].blind = get_int(data);
  next_int(data);
  creatures[id].invisible = get_int(data);
  next_int(data);

  printf("state of %s is %d/%d\n",
         creatures[id].name, creatures[id].hp, creatures[id].state);
}

void rx_send_end_game(void *data) {
  int num_winners;
  int winner_id;
  int i;

  printf("End of game\n");
  num_winners = get_int(data);
  next_int(data);

  printf("There were %d winners\n", num_winners);

  for (i = 0; i < num_winners; i++) {
    winner_id = get_int(data) - 1;
    assert(winner_id >= 0);
    assert(winner_id < num_creatures);

    printf("Winner %d of game is %s (%d)\n",
           i + 1,
          creatures[winner_id].name, winner_id);
  }
}

void rx_newplayer_info(void *data) {
  int id;

  id = get_int(data) - 1;
  next_int(data);

  if (id >= num_creatures) {
    int i;

    creatures = realloc(creatures, (id + 1) * sizeof(CreatureState));
    for (i = num_creatures; i <= id; i++) {
      creatures[i].in_use = 0;
    }
    num_creatures = id + 1;
  }

  assert(!creatures[id].in_use);

  creatures[id].in_use = 1;
  creatures[id].gl = 0;

  /* ignore string length */
  next_int(data);

  strncpy(creatures[id].name, data, 100);
  creatures[id].name[99] = '\0';
  creatures[id].type = 0;
  creatures[id].owner_id = 0;
  creatures[id].paralysed = 0;

  printf("player %s (%d) joined\n", creatures[id].name, id);
}

/* check if creature_id appears as a target in monster_targets[]
 * monster_targets[even] is monster_id
 * monster_targets[odd] is target_id
 */
int should_exclude(int *monster_targets, int num_targets, int creature_id) {
  int i;

  for (i = 0; i < num_targets; i++) {
    if (monster_targets[i * 2 + 1] == creature_id) {
      return 1;
    }
  }
  return 0;
}

/* monster_targets: targets already chosen
 * num_targets: number of targets already chosen
 * monster_id: the monster we want to target
 */
int choose_monster_target(int *monster_targets, int num_targets,
                          int monster_id) {
  int try, i;
  int found_creature_id, found_creature_max_hp, found_creature_hp;

  found_creature_id = -1;

  /* try two times:
   * - first time exclude creatures that we have already targeted
   * - second time don't exclude them
   */
  for (try = 0; (try < 2) && (found_creature_id == -1); try++) {

    /* go through each creature */
    for (i = 0; i < num_creatures; i++) {

      if (!creatures[i].in_use) {
        continue;
      }

      /* never put our monsters onto ourself */
      if ((i == own_id) ||
          /* never put our monsters onto our own monsters */
          ((creatures[i].type != 0) && (creatures[i].owner_id = i))) {
        continue;
      }

      /* test for already-targeted creatures */
      if (try == 0 && should_exclude(monster_targets, num_targets, i)) {
        continue;
      }

      /* note: wizards will get assigned max_hp = (6 - hp) so they get
       * attacked last by our monsters unless their hp is < 6
       */
      int creature_max_hp =
          (creatures[i].type == 0)
          ? (6 - creatures[i].hp)
          : ((creatures[i].type == SC_MONSTER_FIRE_ELEMENTAL) ||
             (creatures[i].type == SC_MONSTER_ICE_ELEMENTAL))
          ? 3
          : creatures[i].type;

      /* first creature found? */
      if ((found_creature_id == -1) ||
          /* a more dangerous creature? */
          (creature_max_hp > found_creature_max_hp) ||
          /* fewer HP to destroy? */
          (creatures[i].hp < found_creature_hp)
      ) {
        found_creature_id = i;
        found_creature_max_hp = creature_max_hp;
        found_creature_hp = creatures[i].hp;
      }
    }
  }

  return found_creature_id;
}

void rx_ask_monster_directions(void *data) {
  int num_monsters;
  int i;
  void *directions;
  int *monsters;
  void *dp;

  printf("asked for monster directions\n");
  num_monsters = get_int(data);
  next_int(data);

  dp = directions = malloc(sizeof(int) + num_monsters * sizeof(int) * 2);
  put_int(dp, num_monsters);
  next_int(dp);

  monsters = malloc(num_monsters * sizeof(int));

  /* get the list of monster IDs from data[] and put into monsters[],
   * sorting as we go so the more-powerful monsters are at the front
   */
  for (i = 0; i < num_monsters; i++) {
    monsters[i] = get_int(data) - 1;
    next_int(data);

    assert(monsters[i] >= 0);
    assert(monsters[i] < num_creatures);
    assert(creatures[monsters[i]].type >= SC_MONSTER_GOBLIN);
    assert(creatures[monsters[i]].type <= SC_MONSTER_GIANT);

    /* if we're past the first monster and the type of the most-recent
     * monster is less than the type of the current monster, we've got
     * some shuffling to do
     */
    if ((i > 0) &&
        (creatures[monsters[i - 1]].type < creatures[monsters[i]].type)) {
      int move_to = i - 1;

      /* look backwards to see where this monster should fit in */
      while (
          (move_to > 0) &&
          (creatures[monsters[move_to]].type < creatures[monsters[i]].type)
      ) {
        move_to--;
      }

      /* move monsters[i] back to monsters[move_to] and move the ones
       * inbetween down one place
       */
      int temp = monsters[i];
      memmove(&monsters[move_to + 1],
              &monsters[move_to],
              (i - move_to)* sizeof(int));
      monsters[move_to] = temp;
    }
  }

  for (i = 0; i < num_monsters; i++) {
    int target_id;

    put_int(dp, monsters[i] + 1);
    next_int(dp);

    assert(monsters[i] >= 0);
    assert(monsters[i] < num_creatures);
    assert(creatures[monsters[i]].type >= SC_MONSTER_GOBLIN);
    assert(creatures[monsters[i]].type <= SC_MONSTER_GIANT);

    target_id = choose_monster_target(((int *)directions) + 1, i, monsters[i]);

    printf("directing monster %s (%d) at %s (%d)\n",
           creatures[monsters[i]].name, monsters[i],
           creatures[target_id].name, target_id);

    put_int(dp, target_id + 1);
    next_int(dp);
  }

  msg_send(server, MSG_RCV_MONSTER_DIRECTIONS,
           directions, sizeof(int) + num_monsters * sizeof(int) * 2);
  free(directions);
  free(monsters);
}

void rx_send_new_monster_info(void *data) {
  int id;

  id = get_int(data);
  next_int(data);

  if (id > num_creatures) {
    int i;

    creatures = realloc(creatures, id * sizeof(CreatureState));
    for (i = num_creatures; i < id; i++) {
      creatures[i].in_use = 0;
    }
    num_creatures = id--;
  }

  assert(!creatures[id].in_use);

  creatures[id].in_use = 1;
  creatures[id].gl = 0;

  creatures[id].type = get_int(data);
  next_int(data);
  creatures[id].owner_id = get_int(data) - 1;
  next_int(data);

  /* ignore string length */
  next_int(data);
  strncpy(creatures[id].name, data, 100);
  creatures[id].name[99] = '\0';

  printf("monster %s (%d) type %s owned by %s (%d) joined\n",
         creatures[id].name, id,
         creature_types[creatures[id].type],
         creatures[creatures[id].owner_id].name, creatures[id].owner_id);
}

void rx_send_monster_attack_info(void *data) {
  int source, target, damage;
  char *result_message;

  source = get_int(data);
  next_int(data);
  if (!source) {
    printf("end of monsters attack\n");
    return;
  }

  source--;
  assert(source >= 0);
  assert(source < num_creatures);

  target = get_int(data) - 1;
  next_int(data);
  assert(target >= 0);
  assert(target < num_creatures);

  damage = get_int(data);
  next_int(data);
  assert(damage >= 0);

  /* ignore string length */
  next_int(data);

  result_message = data;

  printf("%s (%d) attacks %s (%d) causing %d point%s of damage: %s\n",
         creatures[source].name, source,
         creatures[target].name, target,
         damage, ((damage == 1) ? "" : "s"),
         result_message);
}

void rx_send_event_info(void *data) {
  int event_type, message_source_id, source_id, target_id, misc;
  char *event_type_s;
  char *message;

  event_type = get_int(data);
  next_int(data);

  switch (event_type) {
    case SC_EVT_GENERIC_MESSAGE:
      event_type_s = "Generic";
      break;
    case SC_EVT_ELEMENTAL_MERGE:
      event_type_s = "Elemental Merge";
      break;
    case SC_EVT_ELEMENTAL_CANCEL:
      event_type_s = "Elemental Cancel";
      break;
    case SC_EVT_PLAYER_AMNESIA:
      event_type_s = "Amnesia";
      break;
    case SC_EVT_CONFUSION:
      event_type_s = "Confusion";
      break;
    case SC_EVT_PLAYER_FEAR:
      event_type_s = "Fear";
      break;
    case SC_EVT_MONSTER_PARALYSIS:
      event_type_s = "Fear";
      break;
    case SC_EVT_NO_OWNER_MONSTER_DIES:
      event_type_s = "No Owner Monster Dies";
      break;
    case SC_EVT_PLAYER_DIED:
      event_type_s = "Player Died";
      break;
    case SC_EVT_COUNTER_SPELL_NEGATE:
      event_type_s = "Counter Spell";
      break;
    case SC_EVT_DISPEL_MAGIC_NEGATE:
      event_type_s = "Dispel Magic";
      break;
    case SC_EVT_DISPEL_MAGIC_DESTROY_MONSTER:
      event_type_s = "Dispel Magic Destroy Monster";
      break;
    case SC_EVT_MAGIC_MIRROR_FAIL:
      event_type_s = "Magic Mirror Fail";
      break;
    case SC_EVT_FOD_CANCELS_RAISE_DEAD:
      event_type_s = "Raise Dead Cancels Finger of Death";
      break;
    case SC_EVT_CHARMED_HAND:
      event_type_s = "Charmed Hand";
      break;
    case SC_EVT_PARALYSED_HAND:
      event_type_s = "Paralysed Hand";
      break;
    case SC_EVT_CHARMED_GESTURE:
      event_type_s = "Charmed Gesture";
      break;
    default:
      event_type_s = "Unknown";
  }

  message_source_id = get_int(data) - 1;
  next_int(data);

  source_id = get_int(data) - 1;
  next_int(data);

  target_id = get_int(data) - 1;
  next_int(data);

  misc = get_int(data);
  next_int(data);

  /* ignore string length */
  next_int(data);

  message = data;

  printf("%s event from %s (%d): source: %s (%d) target: %s (%d): "
             "misc %d %s\n",
         event_type_s,
         (message_source_id < 0 ? "none" : creatures[message_source_id].name),
         message_source_id,
         (source_id < 0 ? "none" : creatures[source_id].name), source_id,
         (target_id < 0 ? "none" : creatures[target_id].name), target_id,
         misc,
         message);

  if (event_type == SC_EVT_PARALYSED_HAND) {
    creatures[target_id].paralysed = misc;
  }
}

void rx_send_start_game(void *data) {
  printf("rx_send_start_game()\n");

  /* MSG_SEND_ROUND_BEGIN gets sent before the first move, so move_num
   * will be 0 when rx_ask_for_gestures() gets called for the first
   * time
   */
  move_num = -1;
  best_move_move_num = -1;

  own_id = get_int(data) - 1;
  next_int(data);
  printf("our id is %d\n", own_id);

  turn_timeout = get_int(data);
  next_int(data);
  printf("turn timeout is %d\n", turn_timeout);
  turn_timeout -= 5;

  num_creatures = 0;
  free(creatures);
  creatures = NULL;
}

void rx_ask_charm_person_ctrl_hand(void *data) {
  int target_id = get_int(data) - 1;
  int result[2];

  printf("asked for charm hand for target %d\n", target_id);

  result[0] = target_id + 1;
  result[1] = (time(NULL) % 1) ? SC_LEFT_HAND : SC_RIGHT_HAND;
  result[0] = htonl(result[0]);
  result[1] = htonl(result[1]);
  msg_send(server, MSG_RCV_CHARM_PERSON_CTRL_HAND, result, sizeof(result));

  printf("sending charm hand %d\n", result[1]);
}

void rx_ask_paralysis_ctrl_hand(void *data) {
  int         target_id = get_int(data) - 1;
  int         result[2];
  WizardState *wizards;
  Move        paralysis_move;

  printf("asked for paralysis hand for target %d\n", target_id);

  result[0] = target_id + 1;

  if (target_id == own_id) {
    printf("oops, paralysing own hand -- this shouldn't happen!\n");
    result[1] = SC_LEFT_HAND;
    result[0] = htonl(result[0]);
    result[1] = htonl(result[1]);
    msg_send(server, MSG_RCV_PARALYSIS_CTRL_HAND, result, sizeof(result));
    return;
  }

  wizards = calloc(2, sizeof(WizardState));

  set_wizard_info(&wizards[0], own_id);
  set_monster_info(&wizards[0], own_id);
  set_wizard_info(&wizards[1], target_id);
  set_monster_info(&wizards[1], target_id);

  wizards[1].paralysed = -1;
  wizards[1].state |= CS_PARALYSIS;

  find_move_id(wizards, 2, turn_timeout, &paralysis_move);

  result[1] = paralysis_move.paralysis_hand;
  if ((result[1] != SC_LEFT_HAND) && (result[1] != SC_RIGHT_HAND)) {
    printf("unexpected paralysis hand is %d\n", result[1]);
    result[1] = SC_LEFT_HAND;
  }

  printf("sending paralysis hand %d\n", result[1]);

  result[0] = htonl(result[0]);
  result[1] = htonl(result[1]);
  msg_send(server, MSG_RCV_PARALYSIS_CTRL_HAND, result, sizeof(result));

  free(wizards);
}

void rx_ask_charm_person_ctrl_gesture(void *data) {
  int target_id = get_int(data) - 1;
  int result[2];

  printf("asked for charm gesture for target %d\n", target_id);

  result[0] = target_id + 1;
  result[1] = GST_PALM;
  result[0] = htonl(result[0]);
  result[1] = htonl(result[1]);
  msg_send(server, MSG_RCV_CHARM_PERSON_CTRL_GESTURE, result, sizeof(result));

  printf("sending charm gesture %d\n", result[1]);
}

void rx_msg_send_round_begin(void *data) {
  int round_num = get_int(data);

  printf("rx_msg_send_round_begin(%d)\n", round_num);
  move_num++;
  assert(round_num == move_num);
}

void do_stuff(void) {
  while (1) {
    Message *msg = msg_recv(server);

    switch (msg->cmd) {
      case MSG_ASK_WELCOME:
        rx_ask_welcome(msg->data);
        break;
      case MSG_ASK_FOR_GESTURES:
        rx_ask_for_gestures(msg->data);
        break;
      case MSG_ASK_FOR_SPELLS_CAST:
        rx_ask_for_spells_cast(msg->data);
        break;
      case MSG_SEND_GESTURES_SEEN:
        rx_send_gestures_seen(msg->data);
        break;
      case MSG_ASK_SPELL_DIRECTIONS:
        rx_ask_spell_directions(msg->data);
        break;
      case MSG_SEND_SPELL_CAST:
        rx_send_spell_cast(msg->data);
        break;
      case MSG_SEND_CREATURE_STATE:
        rx_send_creature_state(msg->data);
        break;
      case MSG_SEND_END_GAME:
        rx_send_end_game(msg->data);
        break;
      case MSG_SEND_NEWPLAYER_INFO:
        rx_newplayer_info(msg->data);
        break;
      case MSG_ASK_MONSTER_DIRECTIONS:
        rx_ask_monster_directions(msg->data);
        break;
      case MSG_SEND_NEW_MONSTER_INFO:
        rx_send_new_monster_info(msg->data);
        break;
      case MSG_SEND_MONSTER_ATTACK_INFO:
        rx_send_monster_attack_info(msg->data);
        break;
      case MSG_SEND_EVENT_INFO:
        rx_send_event_info(msg->data);
        break;
      case MSG_SEND_START_GAME:
        rx_send_start_game(msg->data);
        break;
      case MSG_ASK_CHARM_PERSON_CTRL_HAND:
        rx_ask_charm_person_ctrl_hand(msg->data);
        break;
      case MSG_ASK_PARALYSIS_CTRL_HAND:
        rx_ask_paralysis_ctrl_hand(msg->data);
        break;
      case MSG_ASK_CHARM_PERSON_CTRL_GESTURE:
        rx_ask_charm_person_ctrl_gesture(msg->data);
        break;
      case MSG_USERNAME_IN_USE_ALREADY:
        printf("Username already in use\n");
        exit(0);
      case MSG_SEND_ROUND_BEGIN:
        rx_msg_send_round_begin(msg->data);
        break;
      default:
        printf("Unhandled message %d from server\n", msg->cmd);
        exit(1);
    }

    free(msg);
  }
}

void parse_args(int argc, char *argv[]) {
  int i;

  for (i = 1; i < argc; i++) {
    if (!strcmp(argv[i], "-h")) {
      if (++i == argc) {
        fprintf(stderr, "-h requires a hostname to follow\n");
        exit(1);
      }

      hostname = argv[i];
    } else if (!strcmp(argv[i], "-n")) {
      if (++i == argc) {
        fprintf(stderr, "-n requires a username to follow\n");
        exit(1);
      }

      username = argv[i];
    } else {
      fprintf(stderr, "unexpected argument '%s'\n", argv[i]);
      exit(1);
    }
  }

  if (!hostname) {
    hostname = "localhost";
  }

  if (!username) {
    username = "becky";
  }
}

int main(int argc, char *argv[]) {
  init_gestures();
  init_spells();
  init_moves();
  init_wizard();
  parse_args(argc, argv);
  server = connect_to_server(hostname);
  do_stuff();
  exit(0);
}
