#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "lua.h"
#include "lualib.h"
#include "tclient.h"
#include "legacy.h"

static int bot_timestamp = 0;
static char *bot_filename = "xorl.lua";
struct TC_GameData *bot_gamedata;

static const char *read_lua(lua_State *L, void *userdata, size_t *size)
{
    int fd = *(int *)userdata;
    static char buf[512];

    *size = read(fd, buf, sizeof(buf));
    if (*size == 0)
        return NULL;

    return buf;
}

void dprint(const char *fmt, ...) {
#ifdef DEBUG
    va_list ap;

    va_start(ap, fmt);
    vprintf(fmt, ap);
    va_end(ap);
#endif
}

static int api_move_block(lua_State *L)
{
    int move = lua_tonumber(L, 1);

    MoveBlock(move);

    return 0;
}

static int api_use_specials(lua_State *L)
{
    FIELD *field = lua_touserdata(L, 1);

    UseSpecials(*field);

    return 0;
}

#define CFG_OFFSET 4
typedef char CFG_FIELD[TC_FIELD_WIDTH + CFG_OFFSET][TC_FIELD_HEIGHT];

#if 0
static void dump_cfg_space(CFG_FIELD cfg_field)
{
    int y, x;
    for (y = 0; y < TC_FIELD_HEIGHT; y++) {
        for (x = 0; x < TC_FIELD_WIDTH + CFG_OFFSET; x++) {
            dprint("%c", cfg_field[x][y] + 48);
        }
        dprint("\n");
    }
}
#endif

/* FIXME: better name & merge with btetris.c */
static int blockcount[7] = { 2, 1, 4, 4, 2, 2, 4 };

/* FIXME: Move to lua */
static int api_get_configuration_spaces(lua_State *L)
{
    FIELD base_field;
    int block;
    int ypos;
    int cfg_table;
    const int oh = TC_FIELD_HEIGHT;
    const int ow = TC_FIELD_WIDTH + CFG_OFFSET;
    int o, y, x;

    memcpy(&base_field, lua_touserdata(L, 1), sizeof(base_field));
    block = lua_tonumber(L, 2);
    ypos = lua_tonumber(L, 3);

    /* Create table for return */
    lua_newtable(L);
    cfg_table = lua_gettop(L);

    for (o = 0; o < blockcount[block]; o++) {
        CFG_FIELD cfg_field;

        /* Calculate valid placements */
        for (y = 0; y < oh; y++) {
            for (x = 0; x < ow; x++) {
                /* 0: Free space, 1: blocked
                 * 2: will be used to represent possible final move */
                cfg_field[x][y] =
                    blockobstructed(base_field, block, o,
                                    x - CFG_OFFSET, y) ? 1 : 0;
            }
        }

        /* Calculate possible final resting places
         * NOTE: Only search starting from the current ypos */
        for (y = ypos; y < oh; y++) {
            for (x = 0; x < ow; x++) {
                /* Valid positions are directly above a bad position
                 * and directly above the bottom row */
                if ((y + 1 == oh || cfg_field[x][y + 1] == 1) &&
                        !cfg_field[x][y]) {
                    /* FIXME: Return this as a table to Lua instead */
                    cfg_field[x][y] = 2;
                }
            }
        }

        lua_pushnumber(L, o);
        memcpy(lua_newuserdata(L, ow * oh), cfg_field, sizeof(cfg_field));
        lua_settable(L, cfg_table);
    }

    /* FIXME: Also return a table of valid moves with the cfg_spaces */

    return 1;
}

struct move_node;
struct move_node {
    int o, x, y;
    int move;
    struct move_node *next;
};

static struct move_node *find_path(CFG_FIELD *cfg_field,
                                   int orient, int xpos, int ypos,
                                   int o, int x, int y)
{
    /* FIXME: */
    return NULL;
}

static void release_move_nodes(struct move_node *move)
{
    struct move_node *ptr;

    while (move) {
        ptr = move->next;
        free(move);
        move = ptr;
    }
}

static int field_score(FIELD field, int block, int o, int x, int y)
{
    FIELD tmp_field;
    int score, lines;

    memcpy(tmp_field, field, TC_FIELD_SIZE);
    set_current_block(block, o, x, y);
    tetris_solidify(tmp_field);
    lines = tetris_removelines(NULL, tmp_field);
    score = FieldValue(tmp_field, NULL, NULL, NULL);
    score += lines * lines * 100;
    free(tmp_field);

    return score;
}

static int api_find_best_move(lua_State *L)
{
    FIELD base_field;
    CFG_FIELD cfg_field[4];
    int cfg_table;
    int o, x, y;
    struct move_node *best_path = NULL;
    int best_score = -1;
    int block, xpos, ypos, orient;
    struct move_node *path;

    memcpy(&base_field, lua_touserdata(L, 1), sizeof(base_field));
    cfg_table = 2;
    block = lua_tonumber(L, 3);
    xpos = lua_tonumber(L, 4);
    ypos = lua_tonumber(L, 5);
    orient = lua_tonumber(L, 6);

    for (o = 0; o < blockcount[block]; o++) {
        lua_pushnumber(L, o);
        lua_gettable(L, cfg_table);
        memcpy(&cfg_field[o], lua_touserdata(L, -1), sizeof(CFG_FIELD));
    }

    for (o = 0; o < blockcount[block]; o++) {
        for (y = 0; y < TC_FIELD_HEIGHT; y++) {
            for (x = 0; x < TC_FIELD_WIDTH + CFG_OFFSET; x++) {
                if (cfg_field[o][x][y] == 2) {
                    int new_score;

                    /* Check field */
                    path = find_path(cfg_field, orient, xpos, ypos, o, x, y);
                    /* Remove an impossible move */
                    if (!path)
                        cfg_field[o][x][y] = 0;
                    new_score = field_score(base_field, block, o, x, y);
                    if (best_score < new_score) {
                        release_move_nodes(best_path);
                        best_path = path;
                        best_score = new_score;
                    }
                }
            }
        }
    }

    /* FIXME: create_move_table(L, move); */

    return 1;
}

static int api_translate_to_local_rep(lua_State *L)
{
    int blocktype, localblocktype, orientation;

    blocktype = lua_tonumber(L, 1);

    TranslateToLocalRep(blocktype, &localblocktype, &orientation);

    lua_pushnumber(L, localblocktype);
    lua_pushnumber(L, orientation);

    return 2;
}

static int api_get_current_block(lua_State *L)
{
    int status, blocktype, xpos, ypos;

    status = GetCurrentBlock(&blocktype, &xpos, &ypos);

    lua_pushnumber(L, status);
    lua_pushnumber(L, blocktype);
    lua_pushnumber(L, xpos);
    lua_pushnumber(L, ypos);

    return 4;
}

static int api_get_best_move(lua_State *L)
{
    int localblocktype, xpos, ypos, orientation;
    FIELD *field;
    int use_offset, use_orientation;

    localblocktype = lua_tonumber(L, 1);
    xpos = lua_tonumber(L, 2);
    ypos = lua_tonumber(L, 3);
    orientation = lua_tonumber(L, 4);
    field = lua_touserdata(L, 5);

    GetBestMove(localblocktype, xpos, ypos, orientation, *field, &use_offset,
                &use_orientation, 0, -20000);

    lua_pushnumber(L, use_offset);
    lua_pushnumber(L, use_orientation);

    return 2;
}

static int api_is_dead_field(lua_State *L)
{
    char *field;

    field = lua_touserdata(L, 1);

    lua_pushboolean(L, (field == NULL));

    return 1;
}

static int api_dprint(lua_State *L)
{
    dprint(lua_tostring(L, 1));

    return 0;
}

static int api_usleep(lua_State *L)
{
    usleep(lua_tonumber(L, 1));

    return 0;
}

static int api_check_bot_change(lua_State *L)
{
    struct stat info;

    if (stat(bot_filename, &info) < 0) {
        lua_pushstring(L, "Failed to get bot change status");
        lua_error(L);
    }

    if (info.st_mtime == bot_timestamp)
        lua_pushnumber(L, 1);
    else
        lua_pushnil(L);

    return 1;
}

/* No arguments.
 * Returns: boolean, New connection established? */
static int api_bot_connect(lua_State *L)
{
    static int bot_connected = 0;
    const char *bot_name = lua_tostring(L, -1);

    if (bot_connected) {
        lua_pushboolean(L, 0);
    } else if (connectToServer("localhost", 9467, bot_name, "", "") < 0) {
        lua_pushstring(L, "Failed to connect to tetribot server");
        lua_error(L);
    } else {
        lua_pushboolean(L, 1);
    }

    bot_connected = 1;

    dprint("Connect to tetribot server.\n");

    return 1;
}

static int api_get_field(lua_State *L)
{
    char *field;
    char *userdata;

    field = GetField((int)lua_tonumber(L, 1));

    /* Transfer field to Lua GC'able object on the stack */
    userdata = lua_newuserdata(L, TC_FIELD_SIZE);
    memcpy(userdata, field, TC_FIELD_SIZE);
    free(field);

    return 1;
}

static int api_get_own_player_number(lua_State *L)
{
    lua_pushnumber(L, GetOwnPlayerNumber(bot_gamedata));

    return 1;
}

/* No arguments, Nothing returned
 * Exceptions: On failure */
static int api_wait_start(lua_State *L)
{
    dprint("Waiting for game.\n");

    if (Start(&bot_gamedata) != 0) {
        lua_pushstring(L, "Game start failed\n");
        lua_error(L);
    }

    dprint("Game started.\n");

    return 0;
}

static void bot_load_functions(lua_State *L)
{
    luaopen_base(L);
    luaopen_table(L);
    luaopen_string(L);
    luaopen_math(L);

    lua_register(L, "dprint",  api_dprint);
    lua_register(L, "bot_connect",  api_bot_connect);
    lua_register(L, "check_bot_change",  api_check_bot_change);
    lua_register(L, "usleep",  api_usleep);
    lua_register(L, "wait_start", api_wait_start);
    lua_register(L, "get_field", api_get_field);
    lua_register(L, "get_own_player_number", api_get_own_player_number);
    lua_register(L, "translate_to_local_rep", api_translate_to_local_rep);
    lua_register(L, "get_best_move", api_get_best_move);
    lua_register(L, "is_dead_field", api_is_dead_field);
    lua_register(L, "get_current_block", api_get_current_block);
    lua_register(L, "move_block", api_move_block);
    lua_register(L, "use_specials", api_use_specials);
    lua_register(L, "get_configuration_spaces", api_get_configuration_spaces);
    lua_register(L, "find_best_move", api_find_best_move);
}

struct lua_constant {
    const char *name;
    int value;
};

static void bot_load_constants(lua_State *L)
{
    struct lua_constant *ptr;

    struct lua_constant tc_table[] = {
        /* Move def */
        { "TC_MOVE_FIRST_MOVE", TC_MOVE_FIRST_MOVE },
        { "TC_MOVE_LEFT", TC_MOVE_LEFT },
        { "TC_MOVE_RIGHT", TC_MOVE_RIGHT },
        { "TC_ROTATE_CLOCKWISE", TC_ROTATE_CLOCKWISE },
        { "TC_ROTATE_ANTICLOCKWISE", TC_ROTATE_ANTICLOCKWISE },
        { "TC_MOVE_DOWN", TC_MOVE_DOWN },
        { "TC_MOVE_DROP", TC_MOVE_DROP },
        { "TC_MOVE_LAST_MOVE", TC_MOVE_LAST_MOVE },

        /* Field dimensions */
        { "TC_FIELD_WIDTH", TC_FIELD_WIDTH },
        { "TC_FIELD_HEIGHT", TC_FIELD_HEIGHT },
        { "TC_FIELD_SIZE", TC_FIELD_SIZE },

        /* Special definitions */
        { "TC_SPECIAL_FIRST_SPECIAL", TC_SPECIAL_FIRST_SPECIAL },
        { "TC_SPECIAL_ADD_LINE", TC_SPECIAL_ADD_LINE },
        { "TC_SPECIAL_CLEAR_SPECIALS", TC_SPECIAL_CLEAR_SPECIALS },
        { "TC_SPECIAL_CLEAR_LINE", TC_SPECIAL_CLEAR_LINE },
        { "TC_SPECIAL_BLOCK_GRAVITY", TC_SPECIAL_BLOCK_GRAVITY },
        { "TC_SPECIAL_NUKE", TC_SPECIAL_NUKE },
        { "TC_SPECIAL_BLOCK_BOMB", TC_SPECIAL_BLOCK_BOMB },
        { "TC_SPECIAL_BLOCK_QUAKE", TC_SPECIAL_BLOCK_QUAKE },
        { "TC_SPECIAL_CLEAR_RANDOM", TC_SPECIAL_CLEAR_RANDOM },
        { "TC_SPECIAL_SWITCH_FIELDS", TC_SPECIAL_SWITCH_FIELDS },
        { "TC_SPECIAL_LAST_SPECIAL", TC_SPECIAL_LAST_SPECIAL },

        /* These are not really "specials" */
        { "TC_SPECIAL_ADDALL1", TC_SPECIAL_ADDALL1 },
        { "TC_SPECIAL_ADDALL2", TC_SPECIAL_ADDALL2 },
        { "TC_SPECIAL_ADDALL4", TC_SPECIAL_ADDALL4 },

        { "TC_BLOCK_NONE", TC_BLOCK_NONE },
        { "TC_BLOCK_O", TC_BLOCK_O },
        { "TC_BLOCK_I_1", TC_BLOCK_I_1 },
        { "TC_BLOCK_I_2", TC_BLOCK_I_2 },
        { "TC_BLOCK_S_1", TC_BLOCK_S_1 },
        { "TC_BLOCK_S_2", TC_BLOCK_S_2 },
        { "TC_BLOCK_Z_1", TC_BLOCK_Z_1 },
        { "TC_BLOCK_Z_2", TC_BLOCK_Z_2 },
        { "TC_BLOCK_L_1", TC_BLOCK_L_1 },
        { "TC_BLOCK_L_2", TC_BLOCK_L_2 },
        { "TC_BLOCK_L_3", TC_BLOCK_L_3 },
        { "TC_BLOCK_L_4", TC_BLOCK_L_4 },
        { "TC_BLOCK_J_1", TC_BLOCK_J_1 },
        { "TC_BLOCK_J_2", TC_BLOCK_J_2 },
        { "TC_BLOCK_J_3", TC_BLOCK_J_3 },
        { "TC_BLOCK_J_4", TC_BLOCK_J_4 },
        { "TC_BLOCK_T_1", TC_BLOCK_T_1 },
        { "TC_BLOCK_T_2", TC_BLOCK_T_2 },
        { "TC_BLOCK_T_3", TC_BLOCK_T_3 },
        { "TC_BLOCK_T_4", TC_BLOCK_T_4 },

        { NULL, 0 }
    };

    lua_newtable(L);
    lua_pushstring(L, "tc");
    lua_pushvalue(L, -2);
    lua_settable(L, LUA_GLOBALSINDEX);

    for (ptr = tc_table; ptr->name != NULL; ptr++) {
        lua_pushstring(L, ptr->name);
        lua_pushnumber(L, ptr->value);
        lua_settable(L, -3);
    }
}

static int bot_run(void)
{
    lua_State *L;
    struct stat info;
    int fd;

    L = lua_open();

    bot_load_functions(L);
    bot_load_constants(L);

    fd = open(bot_filename, O_RDONLY);
    if (fd < 0) {
        perror("open failed");
        return 0;
    }

    if (fstat(fd, &info) < 0) {
        perror("stat failed");
        return 0;
    }

    bot_timestamp = info.st_mtime;

    if (lua_load(L, read_lua, &fd, "bot")) {
        fprintf(stderr, "Load failed: %s\n", lua_tostring(L, -1));
        return 0;
    }

    if (lua_pcall(L, 0, 0, 0)) {
        fprintf(stderr, "Bot failed: %s\n", lua_tostring(L, -1));
        return -1;
    }

    lua_close(L);
    close(fd);

    return 1;
}

int main(int argc, char **argv)
{
    int i = 1;

    do {
        dprint("Loading bot... (%d)\n", i++);
    } while (bot_run());

    return -1;
}

/* vim:et sw=4 ts=4
 */
