/*
 * Tigerbot AI for Tetrinet
 *  Copyright (C) 2003 Matthew Chapman (matthewc@cse.unsw.edu.au)
 *  Copyright (C) 2003 Chris Yeoh (cyeoh@samba.org)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <unistd.h>

#include "tclient.h"
#include "btetris.h"

#ifdef WITH_DEBUG
#define DEBUG(args) printf args
#else
#define DEBUG(args) /* nothing */
#endif

struct TC_GameData *gdata;
int trash_field = 0; /* trying to destroy our field in preparation for a switch? */

#ifdef WITH_DEBUG
/*
 * Debugging only: print out the contents of a field
 */

static void
PrintField(FIELD field)
{
	int i,j;
	char display;

	printf("\n");
	for (i=0; i<TC_FIELD_HEIGHT; i++)
	{
		for (j=0; j<TC_FIELD_WIDTH; j++)
		{
			display = field[i][j];
			display += 48;
			printf("%c", display);
		}
		printf("\n");
	}
}
#endif


/*
 * Analyse the given field, returning an overall "goodness" measure as well as
 * filling out the maximum height, buried holes, total specials and whether there
 * are any block bombs.
 */

#if 0
static int
FieldValue(FIELD field, int *ret_height, int *ret_buried_holes, int *ret_specials_value,
	   int *ret_has_o_block)
{
	int max_height = TC_FIELD_HEIGHT;
	int num_buried_holes = 0;
	int height_weighted_cells = 0;
	int total_well_heights = 0;
	int specials_value = 0;
	int has_o_block = 0;
	int got_block;
	int goodness;
	int x, y;

	/* now calculate remaining goodness parameters: num_buried_holes, height_weighted_cells, total_well_height, specials_value */
	for (x=0; x<TC_FIELD_WIDTH; x++)
	{
		for (y=0, got_block=0; y<TC_FIELD_HEIGHT; y++)
		{
			if (field[y][x]) 
			{
				got_block = 1;
				height_weighted_cells += (TC_FIELD_HEIGHT-y);
				if (y < max_height)
					max_height = y;

				if (field[y][x] >= TC_SPECIAL_FIRST_SPECIAL)
				{
					switch (field[y][x])
					{
						case TC_SPECIAL_BLOCK_BOMB:
							has_o_block = 1;
							break;
						case TC_SPECIAL_BLOCK_GRAVITY:
						case TC_SPECIAL_SWITCH_FIELDS:
							specials_value += 10;
							break;
						default:
							specials_value++;
					}
				}
			}
			else 
			{
				if (got_block)
					num_buried_holes++;

				/* well detection - to cater for pieces randomly strewn on the board, we consider
				 * a well as any hole between two pieces */
				if (((x == 0) || field[y][x-1])
				  && ((x == TC_FIELD_WIDTH-1) || field[y][x+1]))
					total_well_heights++;
			}
		}
	}

	max_height = TC_FIELD_HEIGHT-max_height-1;

	/* Now calculate the goodness.
	 * The relative weights here are from Colin Fahey's tetris algorithm, which in turn supposedly
	 * comes from xtbot, and were derived using a genetic algorithm.
	 * Whether this works equally well for tetrinet remains to be seen.
	 * http://www.colinfahey.com/2003jan_tetris/2003jan_tetris.htm
	 */
	goodness = -13 * num_buried_holes;
	goodness += -2 * height_weighted_cells;
	goodness += -4 * total_well_heights;

	if (ret_height)
		*ret_height = max_height;
	if (ret_buried_holes)
		*ret_buried_holes = num_buried_holes;
	if (ret_specials_value)
		*ret_specials_value = specials_value;
	if (ret_has_o_block)
		*ret_has_o_block = has_o_block;

	return goodness;

}
#endif

static int
FieldValue(FIELD field, int *Height, int *NumHoles, int *SpecialsCount, int *HasOBlock)
{
	int max_height = TC_FIELD_HEIGHT; /* which is actually a minimum */
	int lowest_height = 0;
	int num_buried_holes = 0;
	int well_count_4 = 0;
	int well_count_3 = 0;
	int well_count_5 = 0;
	int x, y;
	int got_block;
	int well_accum, well_started_left, well_started_right;
	int goodness = 0;
	int lh_col;
	int max_well_depth = 0;
	int max_height_for_column[TC_FIELD_WIDTH];
	int jaggedness = 0;

	if (SpecialsCount)
		*SpecialsCount = 0;

	if (HasOBlock)
		*HasOBlock = 0;

	for (x=0; x<TC_FIELD_WIDTH; x++)
	{
		max_height_for_column[x] = TC_FIELD_HEIGHT;

		well_started_left = 0;
		well_started_right =0;
		if (x==0)	well_started_left = 1;
		else if (x==TC_FIELD_WIDTH-1) well_started_right = 1;

		for (y=0, got_block=0, well_accum=0, lh_col=0,
					 well_started_right = 0; y<TC_FIELD_HEIGHT; y++)
		{
			/* Look for buried holes */
			if (field[y][x]) 
			{
				got_block = 1;
				if (y<max_height) max_height = y;
				if (y<max_height_for_column[x]) max_height_for_column[x] = y;
				if (!lh_col && y>lowest_height) 
				{
					lh_col = 1;
					lowest_height = y;
				}
				well_accum = 0;

				if (field[y][x]>=TC_SPECIAL_FIRST_SPECIAL && SpecialsCount)
				{
					*SpecialsCount++;
				}
				if (field[y][x]==TC_SPECIAL_BLOCK_BOMB && HasOBlock)
				{
					*HasOBlock = 1;
				}
			}
			else 
			{
				if (got_block) num_buried_holes++;
			
				/* well detection */
				if (x>0 && field[y][x-1] && !well_started_left) well_started_left = 1;
				if (x<TC_FIELD_WIDTH-1 && field[y][x+1] && !well_started_right)
					well_started_right = 1;
				if (well_started_left && well_started_right) well_accum++;
				if (well_accum==3) well_count_3++;
				if (well_accum==4)
				{
					well_count_3--;
					well_count_4++;
				}
				if (well_accum==5)
				{
					well_count_5++;
					well_count_4--;
				}
				if (well_accum>max_well_depth) max_well_depth = well_accum;
			}
		}
	}

	max_height = TC_FIELD_HEIGHT-max_height-1;
	lowest_height = TC_FIELD_HEIGHT-lowest_height-1;
	goodness = 0;
	goodness -= num_buried_holes*250;

	/* Only more than one well is bad */
/* 	if (well_count_3 + well_count_4 > 1) */
/* 	{ */
	goodness -= (well_count_3+well_count_4*30);
/* 	} */

	/* See how jagged our field looks like */
	for (x=1; x<TC_FIELD_WIDTH; x++)
	{
		jaggedness += abs(max_height_for_column[x]-max_height_for_column[x-1]);
	}
	
	goodness -= jaggedness*10;

	goodness -= max_height*20;
	if (max_height>10) goodness -= (max_height-10)*(max_height-10)*10;

	/* Don't like too much unevenness */
	if (max_height-lowest_height>5)
		goodness -= (max_height - lowest_height - 5)*50;

	/* Really deep wells are bad */
	if (max_well_depth>5)
		goodness -= (max_well_depth-5)*(max_well_depth-5)*20;

	printf("\n\nTest field:");
/* 	PrintField(field); */
	printf("Ht=%i, bh=%i, w3=%i, w4=%i, gd %i\n", max_height,
				 num_buried_holes, well_count_3, well_count_4, goodness);

	if (NumHoles) *NumHoles = num_buried_holes;
	if (Height) *Height = max_height;

	return goodness;

}


/*
 * Check whether dropping the block in the given way is better than the current
 * best move.  A cache is checked to see whether we've already tried dropping
 * from this position before.  The "best move" information is stored in globals
 * for convenience.
 */

static int tried_dropping_from[TC_FIELD_HEIGHT][TC_FIELD_WIDTH];
static int best_value, best_orientation, best_xpos, best_ypos, best_xfinal;

static void
TryDrop(FIELD field, int block, int orientation, int xpos, int ypos, int xfinal)
{
	FIELD tmp_field;
	char specials[100];
	int x = xfinal;
	int y = ypos;
	int lines_removed, value, i;

	if (tried_dropping_from[y][x+2]) /* +2 since x can start at -2 */
		return;

	/* drop the block */
	while (1)
	{
		tried_dropping_from[y][x+2] = 1; /* +2 since x can start at -2 */
		if (blockobstructed(field, block, orientation, x, y+1))
			break;
		y++;
	}

	/* Make a copy of the field */
	memcpy(tmp_field, field, TC_FIELD_SIZE);

	/* Set block to drop */
	set_current_block(block, orientation, x, y);
	tetris_solidify(tmp_field);

	/* Check if line removal will happen */
	lines_removed = tetris_removelines(specials, tmp_field);

	/* score line removal and specials acquired */
	value = (lines_removed)*(lines_removed)*50;
	for (i = 0; specials[i] != 0; i++)
	{
		switch (specials[i])
		{
		/* Block bomb included here as we want to pick it up ASAP!! */
		case TC_SPECIAL_BLOCK_BOMB:
		case TC_SPECIAL_BLOCK_GRAVITY:
		case TC_SPECIAL_SWITCH_FIELDS:
			value += (lines_removed)*50;
			break;
		default:
			value += (lines_removed)*5;
		}
	}

	value += FieldValue(tmp_field, NULL, NULL, NULL, NULL);

	if (value>best_value)
	{
		best_value = value;
		best_orientation = orientation;
		best_xpos = xpos;
		best_ypos = ypos;
		best_xfinal = xfinal;
	}
}


/* 
 * Determine the best move (using the above TryDrop function to evaluate sensible
 * possibilities.  The algorithm attempts to cover cases which require some maneuvering
 * of pieces around obstacles, which most commonly happens when a block bomb goes off.
 * We move the piece horizontally, then move the piece vertically, then move
 * the piece horizontally again, then drop.
 */

static int
GetBestMove(FIELD field, int block, int orientation, int initial_ypos)
{
	int rotation;
	int minx, maxx, xpos, ypos, xfinal;

	best_value = INT_MIN;

	for (rotation=0; rotation < num_rotations_for_blocktype(block); rotation++)
	{
		memset(tried_dropping_from, 0, sizeof(tried_dropping_from));
		get_valid_x_positions(block, orientation, &minx, &maxx);
		for (xpos = minx; xpos <= maxx; xpos++)
		{
			ypos = initial_ypos;
			if (blockobstructed(field, block, orientation, xpos, ypos))
				continue;

			/* try dropping from here - simple case as in harmless */
			TryDrop(field, block, orientation, xpos, ypos, xpos);

			/* more interesting methods - drop a bit, move horizontally, drop fully */
			for (ypos++; !blockobstructed(field, block, orientation, xpos, ypos); ypos++)
			{
				/* try moving right and then dropping */
				for (xfinal = xpos+1; !blockobstructed(field, block, orientation, xfinal, ypos); xfinal++)
					TryDrop(field, block, orientation, xpos, ypos, xfinal);

				/* try moving left and then dropping */
				for (xfinal = xpos-1; !blockobstructed(field, block, orientation, xfinal, ypos); xfinal--)
					TryDrop(field, block, orientation, xpos, ypos, xfinal);
			}
		}

		orientation = get_orientation_after_rotation(block, orientation, 1);

	}
	return best_value;
}


/*
 * Check the specials we have in our collection and use them if appropriate.
 * The algorithm is quite eager.  If the special might be beneficial then we use
 * it immediately, else if there are more specials we drop that special and move
 * on, if we only have one special we keep it until needed.  The only type that
 * bears special mention is the Switch special, which causes it to "reverse" the
 * search such that the bot chooses the worst moves, only when the field is
 * sufficiently high does it switch with another player.
 */

static void
UseSpecials(void)
{
	char *field;
	char specials[100];
	int num_players;
	int best_field_index = -1;
	int best_field_value = INT_MIN;
	int best_specials_index = -1;
	int best_specials_value = INT_MIN;
	int highest_field_index = -1;
	int highest_field_value = INT_MIN;
	int use_special, target, bomb_target = -1;
	int field_height, num_buried_holes, have_o_block;
	int value, height, specials_value, has_o_block;
	int i, j;

	GetSpecials(specials);
	if (!specials[0])
		return;

	field = GetField(GetOwnPlayerNumber(gdata));
	if (field == NULL)
		return;

	FieldValue(field, &field_height, &num_buried_holes, NULL, &have_o_block);
	num_players = GetNumPlayers(gdata);

	for (i=0; specials[i]!=0; i++)
	{
		/* Get Fields for other players */
		for (j=0; j<num_players; j++)
		{
			if (GetPlayerNumber(GetPlayer(gdata,j))==GetOwnPlayerNumber(gdata))
				continue;

			field = GetField(GetPlayerNumber(GetPlayer(gdata,j)));
			if (field == NULL)
				continue;

			value = FieldValue(field, &height, NULL, &specials_value, &has_o_block);
			if (value>best_field_value) 
			{
				best_field_index = j;
				best_field_value = value;
				if (has_o_block)
					bomb_target = j;
			}
			if (specials_value>best_specials_value)
			{
				best_specials_index = j;
				best_specials_value = specials_value;
			}
			if (height>highest_field_value)
			{
				highest_field_index = j;
				highest_field_value = height;
			}
		}

		/* safety */
		if ((best_field_index == -1) || (best_specials_index == -1) || (highest_field_index == -1))
			break;

		use_special = 1;
		target = -1;
		switch (specials[i])
		{
		case TC_SPECIAL_CLEAR_LINE:
		case TC_SPECIAL_NUKE:
			/* Play on self if buried holes or too high */
			if ((num_buried_holes>0) || (field_height>15))
				target = GetOwnPlayerNumber(gdata);
			break;

		case TC_SPECIAL_BLOCK_GRAVITY:
			/* Only has effect if we have buried holes */
			if (num_buried_holes>0)
				target = GetOwnPlayerNumber(gdata);
			break;

		case TC_SPECIAL_SWITCH_FIELDS:
			if (field_height>15)
			{
				/* Switch with best field */
				target = GetPlayerNumber(GetPlayer(gdata, best_field_index));
				trash_field = 0;
				DEBUG(("Trash field behaviour stopped\n"));
			}
			else
			{
				/* Try to make our field worse before using it */
				use_special = 0;
				trash_field = 1;
				DEBUG(("Now trying to trash field\n"));
			}
			break;

		case TC_SPECIAL_CLEAR_SPECIALS:
			/* If we have a block bomb get rid of our specials */
			/* Else play against field with highest specials value */

			if (have_o_block)
				target = GetOwnPlayerNumber(gdata);
			else
				target = GetPlayerNumber(GetPlayer(gdata, best_specials_index));
			break;

		case TC_SPECIAL_ADD_LINE:
			/* Play against highest field to try and get the player out */
			target = GetPlayerNumber(GetPlayer(gdata, highest_field_index));
			break;

		case TC_SPECIAL_BLOCK_BOMB:
			/* Play against best field with an O block (bomb_target) */
			if (bomb_target != -1)
				target = GetPlayerNumber(GetPlayer(gdata, bomb_target));
			break;

		case TC_SPECIAL_CLEAR_RANDOM:
		case TC_SPECIAL_BLOCK_QUAKE:
			/* Always play against best field */
			target = GetPlayerNumber(GetPlayer(gdata, best_field_index));
			break;
		}

		if (!use_special)
		{
			DEBUG(("Leaving special %d\n", specials[i]));
			break;
		}
		else if (target == -1)
		{
			if (specials[i+1] == 0)
			{
				DEBUG(("Leaving special %d\n", specials[i]));
				break;
			}
			DEBUG(("Discarding special %d\n", specials[i]));
			DiscardSpecial();
		}
		else
		{
			DEBUG(("Using special %d on player %d\n", specials[i], target));
			UseSpecial(target);
		}

		/* Have this test here as a failsafe to make sure we exit
			 if the game is over */
		{
			int dummy1, dummy2, dummy3;
			if (GetCurrentBlock(&dummy1, &dummy2, &dummy3)) break;
		}
	}
}


/*
 * Split blocktype (which contains both the piece type and orientation) into
 * those separate components.
 */

static void
TranslateToLocalRep(int blocktype, int *lblock, int *orientation)
{
	switch (blocktype)
	{
	case TC_BLOCK_I_1:
		*lblock = 0;
		*orientation = 0;
		return;
		
	case TC_BLOCK_I_2:
		*lblock = 0;
		*orientation = 1;
		return;

	case TC_BLOCK_O:
		*lblock = 1;
		*orientation = 0;
		return;

	case TC_BLOCK_J_1:
		*lblock = 2;
		*orientation = 3;
		return;

	case TC_BLOCK_J_2:
		*lblock = 2;
		*orientation = 2;
		return;

	case TC_BLOCK_J_3:
		*lblock = 2;
		*orientation = 1;
		return;

	case TC_BLOCK_J_4:
		*lblock = 2;
		*orientation = 0;
		return;

	case TC_BLOCK_L_1:
		*lblock = 3;
		*orientation = 1;
		return;

	case TC_BLOCK_L_2:
		*lblock = 3;
		*orientation = 0;
		return;

	case TC_BLOCK_L_3:
		*lblock = 3;
		*orientation = 3;
		return;

	case TC_BLOCK_L_4:
		*lblock = 3;
		*orientation = 2;
		return;

	case TC_BLOCK_Z_1:
		*lblock = 4;
		*orientation = 1;
		return;

	case TC_BLOCK_Z_2:
		*lblock = 4;
		*orientation = 0;
		return;

	case TC_BLOCK_S_1:
		*lblock = 5;
		*orientation = 1;
		return;

	case TC_BLOCK_S_2:
		*lblock = 5;
		*orientation = 0;
		return;

	case TC_BLOCK_T_1:
		*lblock = 6;
		*orientation = 3;
		return;

	case TC_BLOCK_T_2:
		*lblock = 6;
		*orientation = 2;
		return;

	case TC_BLOCK_T_3:
		*lblock = 6;
		*orientation = 1;
		return;

	case TC_BLOCK_T_4:
		*lblock = 6;
		*orientation = 0;
		return;

	default:
		fprintf(stderr, "bad blocktype %d\n", blocktype);
		abort();

	}
}



/*
 * main
 */

int
main(int argc, char *argv[])
{
	char *username = "tigerbot";
	char *hostname = "localhost";
	char *field;
	int blocktype, xpos, ypos;
	int localblocktype, orientation;
	int status;

	if ((argc==3) && (!strcmp(argv[1], "-h")))
	{
		hostname = argv[2];
	}
	else if (argc>=2)
	{
		fprintf(stderr, "usage: %s [ -h server ]\n", argv[0]);
		exit(1);
	}

	if (connectToServer(hostname, 9467, username, "", "")<0)
	{
		fprintf(stderr, "error: %s\n", tc_error_string);
		exit(1);
	}

	while (1)
	{
		DEBUG(("Waiting for game start\n"));
		if (Start(&gdata) != 0)
		{
			fprintf(stderr, "Game start failed\n");
			exit(1);
		}

		while (1)
		{
			status = GetCurrentBlock(&blocktype, &xpos, &ypos);
		
			if (status==1)
			{
				DEBUG(("*** We lost! ***\n"));
				break;
			}
			else if (status==2)
			{
				DEBUG(("*** We won! ***\n"));
				break;
			}

			if (blocktype==TC_BLOCK_NONE) 
			{
				usleep(200000);
				continue;
			}

			if (!trash_field)
			{
				field = GetField(GetOwnPlayerNumber(gdata));
				if (field == NULL)
					continue;

				TranslateToLocalRep(blocktype, &localblocktype, &orientation);
				GetBestMove(field, localblocktype, orientation, ypos);
				
				while (orientation != best_orientation)
				{
					MoveBlock(orientation<best_orientation ? TC_ROTATE_CLOCKWISE
										: TC_ROTATE_ANTICLOCKWISE);
					orientation<best_orientation ? orientation++ : orientation--;
				}
				while (xpos != best_xpos)
				{
					MoveBlock(xpos<best_xpos ? TC_MOVE_RIGHT : TC_MOVE_LEFT);
					xpos<best_xpos ? xpos++ : xpos--;
				}
				while (ypos != best_ypos)
				{
					MoveBlock(TC_MOVE_DOWN);
					ypos++;
				}
				while (xpos != best_xfinal)
				{
					MoveBlock(xpos<best_xfinal ? TC_MOVE_RIGHT : TC_MOVE_LEFT);
					xpos<best_xfinal ? xpos++ : xpos--;
				}
			}
			MoveBlock(TC_MOVE_DROP);

			UseSpecials();
		}
	}	

	exit(0);
}
