/*
   Copyright (C) 2003 Christopher Yeoh <cyeoh@samba.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA. 
*/
#ifndef __TC_CLIENT_H
#define __TC_CLIENT_H

/* Move def */
#define TC_MOVE_LEFT 1
#define TC_MOVE_RIGHT 2
#define TC_ROTATE_CLOCKWISE 3
#define TC_ROTATE_ANTICLOCKWISE 4
#define TC_MOVE_DOWN 5
#define TC_MOVE_DROP 6

/* Field dimensions */
#define TC_FIELD_WIDTH 12
#define TC_FIELD_HEIGHT 22
#define TC_FIELD_SIZE (TC_FIELD_WIDTH*TC_FIELD_HEIGHT)

/* Special definitions */
#define TC_SPECIAL_ADD_LINE 6
#define TC_SPECIAL_CLEAR_SPECIALS 11
#define TC_SPECIAL_CLEAR_LINE 7
#define TC_SPECIAL_BLOCK_GRAVITY 12
#define TC_SPECIAL_NUKE 8
#define TC_SPECIAL_BLOCK_BOMB 14
#define TC_SPECIAL_BLOCK_QUAKE 13
#define TC_SPECIAL_CLEAR_RANDOM 9
#define TC_SPECIAL_SWITCH_FIELDS 10

/* Block type definitions

TC_BLOCK_O
.xx.
.xx.
....
....
 	 
 	 
TC_BLOCK_I_1 2
xxxx
....
....
....
 	 
 	 
TC_BLOCK_I_2 3
..x.
..x.
..x.
..x.
 	 
 	 
TC_BLOCK_S_1 4
..xx
.xx.
....
....
	 
	 
TC_BLOCK_S_2 5
.x..
.xx.
..x.
....
	 
	 
TC_BLOCK_Z_1 6
.xx.
..xx
....
....
		
		
TC_BLOCK_Z_2 7
..x.
.xx.
.x..
....
	
	
TC_BLOCK_L_1 8
.xxx
.x..
....
....
	
	
TC_BLOCK_L_2 9
.x..
.x..
.xx.
....
	 
	 
TC_BLOCK_L_3 10
...x
.xxx
....
....
	
	
TC_BLOCK_L_4 11
.xx.
..x.
..x.
....
	 
	 
TC_BLOCK_J_1 12
.xxx
...x
....
....
		
		
TC_BLOCK_J_2 13
.xx.
.x..
.x..
....
	
	
TC_BLOCK_J_3 14
.x..
.xxx
....
....
		
		
TC_BLOCK_J_4 15
..x.
..x.
.xx.
....
	
	
TC_BLOCK_T_1 16
.xxx
..x.
....
....
	 
	 
TC_BLOCK_T_2 17
.x..
.xx.
.x..
....
	
	
TC_BLOCK_T_3 18
..x.
.xxx
....
....
	 	
	 	
TC_BLOCK_T_4 19
..x.
.xx.
..x.
....

*/

#define TC_BLOCK_NONE 0
#define TC_BLOCK_O 1
#define TC_BLOCK_I_1 2
#define TC_BLOCK_I_2 3
#define TC_BLOCK_S_1 4
#define TC_BLOCK_S_2 5
#define TC_BLOCK_Z_1 6
#define TC_BLOCK_Z_2 7
#define TC_BLOCK_L_1 8
#define TC_BLOCK_L_2 9
#define TC_BLOCK_L_3 10
#define TC_BLOCK_L_4 11
#define TC_BLOCK_J_1 12
#define TC_BLOCK_J_2 13
#define TC_BLOCK_J_3 14
#define TC_BLOCK_J_4 15
#define TC_BLOCK_T_1 16
#define TC_BLOCK_T_2 17
#define TC_BLOCK_T_3 18
#define TC_BLOCK_T_4 19
/* Rotating a block will (with the exception of 'O' block
	 effectively transition it to another a block type. This
	 type of representation removes the need for the concept of
   orientation */

struct TC_GameData;
struct TC_PlayerData;

/* Connect */
int connectToServer(const char *Hostname, int Port, const char *Username,
										char *TeamName, char *Channel);
/* Pass empty string as TeamName if not member of a team */
/* Pass empty string as Channel if you don't care */
/* 0 if sucessfully connected, non zero otherwise */

/* Start */
int Start(struct TC_GameData **Data);
/* 0 if game has started, non zero otherwise (eg error)
	 Call will block until game starts
	 Start should be called again as soon as you die if you
	 wish to be notified when the next game starts
	 Data will stay constant for duration of game */

/* Error string */
extern char *tc_error_string;

/* Returns 0 if valid move */
int MoveBlock(int MoveCommand);
/* Note move may also no longer be valid if the field has changed
	 since you calculated what you should do. */

/* Use special */
int UseSpecial(int PlayerNum);
/* Returns 0 if ok
	 1 otherwise (eg target player is no longer playing) */

/* Discard the special at the head of the list */
void DiscardSpecial(void);

/* Get field for player */
char *GetField(int PlayerNum);
/* Returns NULL if player is dead
	 Its up to you to free this memory

	 Note that the field returned is a 2 dimensional array
	 field[TC_FIELD_WIDTH][TC_FIELD_HEIGHT]

	 field[0][0] is the top left corner,
	 field[TC_FIELD_WIDTH-1][TC_FIELD_HEIGHT-1] is the bottom right corner
 */

/* Get list of specials */
void GetSpecials(char *Specials);
/* Null terminated array of specials */

/* Get current block dropping */
void GetCurrentBlock(int *BlockType, int *XPosition, int *YPosition);
/* Note that BlockType will be TC_BLOCK_NONE when no block
	 is dropping. This will be the case for approximately a second
	 after the previous block has finished falling. In this case
   the value of XPosition and YPosition is undefined 

	 XPosition and YPosition refers to the centre of the block
	 defined by the black dot in:

	 http://www.colinfahey.com/2003jan_tetris/tetris_standard_specifications.htm
*/


/* Get next block to drop */
void GetNextBlock(int *BlockType);

/* Messages */
int GetMessage(int *FromPlayer, char **Message);
/* Normally out of band, but in this case you will need to poll for
	 this information. One line returned each time.

	 Returns 0 on sucess, 1 if there is no message waiting.
	 
	 Take a copy before calling GetMessage again if you want to keep it.

	 Note, you don't have to poll for messages if you don't want to.
*/

/* Send a message */
void SendMessage(int ToPlayer, char *Message);
/* ToPlayer should be 0 if you want to send it to everyone */

/* Access Game data */

/* Get Player Information */
int GetNumPlayers(struct TC_GameData *Data);
struct TC_PlayerData *GetPlayer(struct TC_GameData *Data, int Player);
/* Don't free this! */

/* Maximum number of specials you can store */
int GetMaxSpecials(struct TC_GameData *Data);

/* Get Own PlayerNumber */
int GetOwnPlayerNumber(struct TC_GameData *Data);

/* Free memory allocated for game data by Start call */
void FreeGameData(struct TC_GameData *Data);


/* Get Player data */
int GetPlayerNumber(struct TC_PlayerData *Player);
char *GetPlayerName(struct TC_PlayerData *Player);
char *GetPlayerTeam(struct TC_PlayerData *Player);


#endif /* __TC_CLIENT_H */
