/*
 *  Copyright (C) 2003 Chris Yeoh (cyeoh@samba.org)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <termios.h>
#include <errno.h>
#include <limits.h>
#include <signal.h>
#include <string.h>
#include <time.h>

#include "tclient.h"
#include "btetris.h"

#define PRINTF( ... ) 	do { } while ( 0 )
#define FPRINTF( ... )  do { } while ( 0 )
//#define PRINTF printf
//#define FPRINTF fprintf

#define DEFAULT_PORT 9467

const int LEVEL_MUL = 50;
const int ROCKY_MUL = 2;
const int HOLE_MUL = 6;
const int SIDE_MUL = 0;

struct TC_GameData *gdata;
char *Specials = NULL;
int status = 0;

int self_destruct = 0;

enum MoveType { MoveLeft, MoveRight, MoveUp, MoveDown, RotCW, RotCC };

typedef struct
{
	int n;
	enum MoveType moves[640];
	int good;
	int x, y; // final (or current) pos
} Path;

Path NewPath( int x, int y )
{
	Path p;
	p.n = 0;
	p.good = 0;
	p.x = x;
	p.y = y;
	return p;
}

typedef struct
{
	int orient;
	int val;
	Path p;
} Move;

Move NewMove()
{
	Move m = { 0, 0 };
	m.p = NewPath( 0, 0 );
	return m;
}

typedef struct
{
	int block, orient;
	int w, h;
	int xoff, yoff;
	P_TETRISBLOCK b;
} BlockInfo;

BlockInfo GetBlockInfo( int block, int orient );

void TranslateToLocalRep(int blocktype, int *lblock, int *orientation);

volatile int no_time = 0;

void alarmed( int a )
{
	no_time = 1;
	FPRINTF( stderr, "alarmed!\n" );
}

void PrintField(FIELD field)
{
	int i,j;
	char display;

	PRINTF("\n");
	for (i=0; i<TC_FIELD_HEIGHT; i++)
	{
		for (j=0; j<TC_FIELD_WIDTH; j++)
		{
			display = field[i][j];
			display += 48;
			PRINTF("%c", display);
		}
		PRINTF("\n");
	}
}

int FieldValue(FIELD field, int *NumHoles, int *Height, int *SpecialsCount)
{
	int max_height = TC_FIELD_HEIGHT; /* which is actually a minimum */
	int lowest_height = 0;
	int num_buried_holes = 0;
	int well_count_4 = 0;
	int well_count_3 = 0;
	int well_count_5 = 0;
	int x, y;
	int got_block;
	int well_accum, well_started_left, well_started_right;
	int goodness = 0;
	int lh_col;
	int max_well_depth = 0;
	int i = 666;
	int max_height_for_column[TC_FIELD_WIDTH];
	int jaggedness = 0;

	if (SpecialsCount)
	{
		for (i=TC_SPECIAL_FIRST_SPECIAL; i<=TC_SPECIAL_LAST_SPECIAL; i++)
		{
			SpecialsCount[TC_SPECIAL_LAST_SPECIAL-i] = 0;
		}
	}

	for (x=0; x<TC_FIELD_WIDTH; x++)
	{
		max_height_for_column[x] = TC_FIELD_HEIGHT;

		if (x==0)	well_started_left = 1;
		else if (x==TC_FIELD_WIDTH-1) well_started_right = 1;
		else
		{
			well_started_left = 0;
			well_started_right =0;

		}
		for (y=0, got_block=0, well_accum=0, lh_col=0,
					 well_started_right = 0; y<TC_FIELD_HEIGHT; y++)
		{
			/* Look for buried holes */
			if (field[y][x]) 
			{
				got_block = 1;
				if (y<max_height) max_height = y;
				if (y<max_height_for_column[x]) max_height_for_column[x] = y;
				if (!lh_col && y>lowest_height) 
				{
					lh_col = 1;
					lowest_height = y;
				}
				well_accum = 0;

				if (field[y][x]>=TC_SPECIAL_FIRST_SPECIAL && SpecialsCount)
				{
					SpecialsCount[field[y][x]-TC_SPECIAL_FIRST_SPECIAL]++;
				}
			}
			else 
			{
				if (got_block) num_buried_holes++;
			
				/* well detection */
				if (x>0 && field[y][x-1] && !well_started_left) well_started_left = 1;
				if (x<TC_FIELD_WIDTH-1 && field[y][x+1] && !well_started_right)
					well_started_right = 1;
				if (well_started_left && well_started_right) well_accum++;
				if (well_accum==3) well_count_3++;
				if (well_accum==4)
				{
					well_count_3--;
					well_count_4++;
				}
				if (well_accum==5)
				{
					well_count_5++;
					well_count_4--;
				}
				if (well_accum>max_well_depth) max_well_depth = well_accum;
			}
		}
	}

	max_height = TC_FIELD_HEIGHT-max_height-1;
	lowest_height = TC_FIELD_HEIGHT-lowest_height-1;
	goodness = 0;
	goodness -= num_buried_holes*250;

	/* Only more than one well is bad */
/* 	if (well_count_3 + well_count_4 > 1) */
/* 	{ */
	goodness -= (well_count_3+well_count_4*30);
/* 	} */

	/* See how jagged our field looks like */
	for (x=1; x<TC_FIELD_WIDTH; x++)
	{
		jaggedness += abs(max_height_for_column[x]-max_height_for_column[x-1]);
	}
	
	goodness -= jaggedness*10;

	goodness -= max_height*20;
	if (max_height>10) goodness -= (max_height-10)*(max_height-10)*10;

	/* Don't like too much unevenness */
	if (max_height-lowest_height>5)
		goodness -= (max_height - lowest_height - 5)*50;

	/* Really deep wells are bad */
	if (max_well_depth>5)
		goodness -= (max_well_depth-5)*(max_well_depth-5)*20;

//	PRINTF("\n\nTest field:");
/* 	PrintField(field); */
//	PRINTF("Ht=%i, bh=%i, w3=%i, w4=%i, gd %i\n", max_height,
//				 num_buried_holes, well_count_3, well_count_4, goodness);

	if (NumHoles) *NumHoles = num_buried_holes;
	if (Height) *Height = max_height;

    //FPRINTF(stderr, "Field has goodness %d\n", goodness);
	if (goodness) return goodness;
    return -20000;

}

int GettableHole( FIELD field, int x, int y )
{
	if ( y < 2 )
		return 1;
	/* want empty blocks nearby as follows:
	    **
	    **
	    **X
	   OR
	   **
	   **
	   ***X
	   OR mirror image
	   so common blocks are (hashes):
	   *#%
	   *#%
	   *##X
	 */
	if ( x > 1 )
	{
				
		int common = 0 == field[y][x-1]
				&& 0 == field[y][x-2]
				&& 0 == field[y-1][x-2]
				&& 0 == field[y-2][x-2];
		if ( common && 0 == field[y-1][x-1]
				&& 0 == field[y-2][x-1] )
			return 1;
		if ( x > 2 && common
				&& 0 == field[y][x-3]
				&& 0 == field[y-1][x-3]
				&& 0 == field[y-2][x-3] )
			return 1;
	}
	if ( x < FIELD_WIDTH-2 )
	{
		int common = 0 == field[y][x+1]
				&& 0 == field[y][x+2]
				&& 0 == field[y-1][x+2]
				&& 0 == field[y-2][x+2];
		if ( common && 0 == field[y-1][x+1]
				&& 0 == field[y-2][x+1] )
			return 1;
		if ( x < FIELD_WIDTH-3 && common
				&& 0 == field[y][x+3]
				&& 0 == field[y-1][x+3]
				&& 0 == field[y-2][x+3] )
			return 1;
	}
	/* special case */
	if ( y < FIELD_HEIGHT-5 
			&& 0 == field[y+1][x]
			&& 0 == field[y+2][x]
			&& 0 == field[y+3][x] )
	{
		int k;
		if ( x > 0 )
		{
			for ( k=0; k < 8; k++ )
				if ( 0 != field[k-3][x-1] )
					break;
			if ( 8 == k )
				return 1;
		}
		if ( x < FIELD_WIDTH-1 )
		{
			for ( k=0; k < 8; k++ )
				if ( 0 != field[k-3][x+1] )
					break;
			if ( 8 == k )
				return 1;
		}
	}
	return 0;
}

int Rockiness( FIELD field )
{
	int heights[FIELD_WIDTH];
	int i, j, num=0;
	for ( i=0; i < FIELD_WIDTH; i++ )
		heights[i] = 0;

	for ( i=0; i < FIELD_WIDTH; i++ )
	{
		int next_max = 0;
		for ( j=0; j < FIELD_HEIGHT; j++ )
		{
			if ( 0 != field[j][i] )
			{
				heights[i] = FIELD_HEIGHT -j;
				break;
			}
		}
		for ( ; j < FIELD_HEIGHT; j++ )
		{
			if ( 0 == field[j][i] )
			{
				if ( !GettableHole(field,i,j) )
					break;
				next_max = 1;
			}
			else if ( next_max )
			{
				heights[i] = FIELD_HEIGHT -j;
				next_max = 0;
			}
		}
	}

	{
		int diff, last_diff = heights[1] -heights[0];
		for ( i=2; i < FIELD_WIDTH; i++ )
		{
			diff = heights[i] -heights[i-1];
			if ( diff > 0 != last_diff > 0 )
			{
				int ad = abs(diff), lad = abs(last_diff);
				if ( ad > 1 && lad > 1 )
				{
					//ad = ad > 4 ? 4 : ad;
					//lad = lad > 4 ? 4 : lad;
					num += (lad+ad)*ROCKY_MUL;
				}
			}
		}
	}

	return num;
}

int BlockFits( BlockInfo bi, int x, int y, FIELD field )
{
	int i, j;
	if ( bi.xoff +x < 0 )
		return 0;
	for ( j=0; j < bi.yoff+bi.h && j+y < FIELD_HEIGHT; j++ )
		for ( i=bi.xoff; i < bi.xoff+bi.w && i+x < FIELD_WIDTH; i++ )
		{
			if ( 0 == bi.b[j][i] )
				continue;
			if ( 0 != field[j+y][i+x] )
				return 0;
		}
	return 1;
}

// these two function check if we're on the edge of the board
int CanMoveLeft( BlockInfo bi, int x )
{
	return x -bi.xoff > 0;
}

int CanMoveRight( BlockInfo bi, int x )
{
	return (x +bi.xoff +bi.w) < FIELD_WIDTH;
}

// return 0 if cant do move, else 1
int PathAddMove( Path* p, BlockInfo bi, enum MoveType m )
{
	switch ( m )
	{
		case MoveLeft:
			if ( !CanMoveLeft(bi,p->x) )
				return 0;
			p->x--;
			p->moves[p->n] = MoveRight;
			break;

		case MoveRight:
			if ( !CanMoveRight(bi,p->x) )
				return 0;
			p->x++;
			p->moves[p->n] = MoveLeft;
			break;

		case MoveUp:
			p->y--;
			p->moves[p->n] = MoveDown;
			break;

		case RotCC:
			p->moves[p->n] = RotCW;
			break;

		case RotCW:
			p->moves[p->n] = RotCC;
			break;

		default:
			FPRINTF( stderr, "Invalid Move.\n" );
			return 0;
	}
	p->n++;
	return 1;
}

int path_out_count = 0;

FIELD seen;

void ClearSeen()
{
	int i, j;
	for ( j=0; j < FIELD_HEIGHT; j++ )
		for ( i=0; i < FIELD_HEIGHT; i++ )
			seen[j][i] = 0;
}

Path GetPathOut( Path p, BlockInfo* bi, FIELD field )
{
	int r, num_rot, orient = bi->orient;
	PRINTF( "path out: %d\n", ++path_out_count );
	// try going up
	if ( 0 == p.y )
	{
		p.good = 1;
		return p;
	}
	if ( 0 != seen[p.y][p.x] )
		return p;
	seen[p.y][p.x] = 1;

	// for each rotation
	// TODO: only do necessary number of rotations
	//num_rot = num_rotations_for_blocktype(bi->block);
	//for ( r=0; r < num_rot; r++ )
	{
		int i;
		Path qr = p;
		/*
		if ( r != 0 )
		{
			if ( !tetris_blockrotate(1,field) )
				break;
			*bi = GetBlockInfo( bi->block, (orient+r)%num_rot );
			PathAddMove(&qr,*bi,RotCW);
		}
		*/

		enum MoveType move_types[] = { MoveUp, MoveLeft, MoveRight };
		int num_moves = sizeof(move_types)/sizeof(enum MoveType);
		for ( i=0; i < num_moves; i++ )
		{
			Path q = qr;
			if ( !PathAddMove(&q,*bi,move_types[i]) )
				continue;
			if ( !BlockFits(*bi,q.x,q.y,field) )
				continue;

			q = GetPathOut( q, bi, field );
			if ( q.good )
				return q;
		}
	}
	return p;
}

BlockInfo GetBlockInfo( int block, int orient )
{
	// TODO: optimise (think case switch)
	BlockInfo bi;
	bi.block = block;
	bi.orient = orient;
	bi.b = tetris_getblock( block, orient );
	int i, j, minx=4, miny=4, maxx=0, maxy=0;
	for ( j=0; j < 4; j++ )
		for ( i=0; i < 4; i++ )
		{
			if ( bi.b[j][i] == 0 )
					continue;
			minx = i < minx ? i : minx;
			miny = j < miny ? j : miny;
			maxx = i > maxx ? i : maxx;
			maxy = j > maxy ? j : maxy;
		}
	bi.w = maxx -minx +1;
	bi.h = maxy -miny +1;
	bi.xoff = minx;
	bi.yoff = miny;
	return bi;
}

void print_field_only( FIELD field )
{
	int i, j;
	for ( j=0; j < FIELD_HEIGHT; j++ )
	{
		for ( i=0; i < FIELD_WIDTH; i++ )
			PRINTF("%s", field[j][i] == 0 ? " " : "#" );
		PRINTF("|\n");
	}
	PRINTF("----------------\n");
}

// DEBUG
void print_field( P_TETRISBLOCK b, int x, int y, FIELD field )
{
	int i, j;
	return;
	for ( j=0; j < FIELD_HEIGHT; j++ )
	{
		for ( i=0; i < FIELD_WIDTH; i++ )
		{
			int bx = i-x, by = j-y;
			if ( y == j && x == i )
				PRINTF("*");
			else if ( by >= 0 && by < 4
							&& bx >= 0 && bx < 4
							&& 0 != b[by][bx] )
				PRINTF("-");
			else
				PRINTF("%s", field[j][i] == 0 ? " " : "#" );
		}
		PRINTF("|\n");
	}
	PRINTF("...............\n");
}

// DEBUG
void print_path( Path p )
{
	int i;
	for ( i=p.n-1; i >= 0; i-- )
	{
		switch ( p.moves[i] )
		{
			case MoveRight:
				PRINTF( "Right\n" );
				break;
			case MoveLeft:
				PRINTF( "Left\n" );
				break;
			case MoveDown:
				PRINTF( "Down\n" );
				break;
            default:
                break;
		}
	}
}

/*
   Returns 1 if pos is better.
 */
Move GetBetterMove( BlockInfo bi, Move m, FIELD field )
{
	int x=m.p.x, y=m.p.y;
	int j;

	/* NB: all blocks have a block in the top row, so don't need to
	   care about offset
	 */
	for ( j=FIELD_HEIGHT-bi.h; j > y; j-- )
	{
			if ( !BlockFits(bi,x,j,field) )
					continue;
			// figure out a path out
			//PRINTF( "better pos at %d, %d ... %d\n", x, j, bi.h );
			//print_field( bi.b, x, j, field );
			path_out_count = 0;
			ClearSeen();
			m.p = GetPathOut( NewPath(x,j), &bi, field );
			if ( m.p.good )
			{
				//PRINTF( "can get out at %d, %d ... %d\n", x, j, bi.h );
				//print_path( m.p );
				break;
			}
	}

	/*
	if ( j == y )
			return 0;
			*/
	// TODO: osetup orientation
	/*
	if ( m.p.good )
	{
		PRINTF("==== good ====\n");
		for ( j=0; j < 4; j++ )
		{
			for ( i=0; i < 4; i++ )
				PRINTF("%s", bi.b[j][i] == 0 ? " " : "#" );
			PRINTF("\n");
		}
	}
	*/
	m.orient = bi.orient;

	return m;
}

Move GetMove( BlockInfo bi, int x, int y, FIELD field )
{
	Move m = NewMove(), better;
	m.orient = bi.orient;
	m.p.good = 0;
	//int i,j;
	/* Set block to drop */
	/* PRINTF("Set current block: %i %i %i %i\n",  */
	/* block, orientation, xpos, ypos); */
	set_current_block( bi.block, bi.orient, x, y );

	/* drop the block */
	tetris_blockdrop(field);
	get_current_block_position( &m.p.x, &m.p.y );

	if ( !no_time )
		better = GetBetterMove( bi, m, field );
	if ( better.p.good )
	{
		int i;
		Path p = better.p;
		m = better;
		//print_field_only( field );
		set_current_block( bi.block, bi.orient, m.p.x, m.p.y );
		// move to the right pos (left/right)
		// do the move (backwards)
		for ( i=p.n-1; i >= 0; i-- )
		{
			switch ( p.moves[i] )
			{
				case MoveDown:
					tetris_blockdown(field);
					break;
				case MoveLeft:
					tetris_blockmove(-1,field);
					break;
				case MoveRight:
					tetris_blockmove(1,field);
					break;
                		default:
					;
			}
		}
		get_current_block_position( &m.p.x, &m.p.y );
		//print_field_only( field );
		//PRINTF( "after move: %d %d\n", x, y );
	}

	tetris_solidify(field);

	/*
	for ( i=0; i < FIELD_HEIGHT; i++ )
	{
			for ( j=0; j < FIELD_WIDTH; j++ )
			{
					PRINTF( "%3d ", tmp_field[i][j] );
			}
			PRINTF( "\n" );
	}
	*/
	return m;
}

int CountHoles( FIELD field )
{
	int i, j, num=0;
	for ( i=0; i < FIELD_WIDTH; i++ )
	{
		for ( j=0; j < FIELD_HEIGHT; j++ )
		{
			if ( 0 != field[j][i] )
				break;
		}
		for ( ; j < FIELD_HEIGHT; j++ )
		{
			// lower holes are worse
			if ( 0 == field[j][i] )
				num += (5+j)*HOLE_MUL;
		}
	}
	return num;
}

Move GetBestMove(int block, int xpos, int ypos, int orientation,
		FIELD field, int recurse)
{
	FIELD tmp_field;
	int rotation;
	int minx, maxx;
	int lines_removed;
	int next_block;
	int next_orientation;
	int holes;
	Move best = NewMove();
	best.val = self_destruct ? INT_MAX : INT_MIN;

/* 	PRINTF("Input field\n"); */
/* 	PrintField(field); */
/* 	PRINTF("Given block %i, orientation %i\n", block, orientation); */

	if (recurse)
	{
		int rnb;
		GetNextBlock(&rnb);
		TranslateToLocalRep(rnb, &next_block, &next_orientation);
		PRINTF("Next block is %i %i %i\n", rnb, next_block, next_orientation);
	}

	for (rotation=0; rotation<num_rotations_for_blocktype(block); rotation++)
	{
		//PRINTF("+++++++++++++++++++Rotation\n");
		get_valid_x_positions(block, orientation, &minx, &maxx);
/*  		PRINTF("valid x positions %i %i\n", minx, maxx);  */

		for (xpos = minx; xpos<=maxx; xpos++)
		{
			Move m = NewMove();
			BlockInfo bi = GetBlockInfo( block, orientation );
			/* Make a copy of the field */
			memcpy(tmp_field, field, TC_FIELD_SIZE);

			m = GetMove( bi, xpos, ypos, tmp_field );
			/* Check if line removal will happen */
			lines_removed = tetris_removelines(NULL, tmp_field);

			if (recurse)
			{
				Move m2 = GetBestMove(next_block, 4, 0, next_orientation,
						tmp_field, 0 );
				m.val = m2.val +(lines_removed*lines_removed*1100);
				holes = CountHoles( tmp_field );
				m.val -= holes;
				m.val += LEVEL_MUL*m.p.y;
				PRINTF( "level mul: %d\n", LEVEL_MUL*m.p.y );
				m.val -= Rockiness( tmp_field );
				if ( m.p.x < FIELD_WIDTH/2 )
					m.val += SIDE_MUL*(FIELD_WIDTH/2-m.p.x-1);
				else
					m.val += SIDE_MUL*(m.p.x-FIELD_WIDTH/2);
			}
			else
			{
				m.val = FieldValue(tmp_field, NULL, NULL, NULL);
 				m.val += (lines_removed)*(lines_removed)*900;
				holes = CountHoles( tmp_field );
				m.val -= holes;
				m.val += LEVEL_MUL*m.p.y;
				m.val -= Rockiness( tmp_field );

				if ( m.p.x < FIELD_WIDTH/2 )
					m.val += SIDE_MUL*(FIELD_WIDTH/2-m.p.x-1);
				else
					m.val += SIDE_MUL*(m.p.x-FIELD_WIDTH/2);
			}

			if ( (m.val > best.val && !self_destruct)
							|| (m.val < best.val && self_destruct) )
			{
				best = m;
				/*
				if ( recurse )
				{
					print_field_only( tmp_field );
					PRINTF( "holes: %d\n", holes );
					PRINTF( "score: %d\n", m.val );
				}
				*/
			}
		}
		
		orientation = get_orientation_after_rotation(block, orientation, 1);
	}
	return best;
}


void TranslateToLocalRep(int blocktype, int *lblock, int *orientation)
{
	switch (blocktype)
	{
	case TC_BLOCK_I_1:
		*lblock = 0;
		*orientation = 0;
		return;
		
	case TC_BLOCK_I_2:
		*lblock = 0;
		*orientation = 1;
		return;

	case TC_BLOCK_O:
		*lblock = 1;
		*orientation = 0;
		return;

	case TC_BLOCK_J_1:
		*lblock = 2;
		*orientation = 3;
		return;

	case TC_BLOCK_J_2:
		*lblock = 2;
		*orientation = 2;
		return;

	case TC_BLOCK_J_3:
		*lblock = 2;
		*orientation = 1;
		return;

	case TC_BLOCK_J_4:
		*lblock = 2;
		*orientation = 0;
		return;

	case TC_BLOCK_L_1:
		*lblock = 3;
		*orientation = 1;
		return;

	case TC_BLOCK_L_2:
		*lblock = 3;
		*orientation = 0;
		return;

	case TC_BLOCK_L_3:
		*lblock = 3;
		*orientation = 3;
		return;

	case TC_BLOCK_L_4:
		*lblock = 3;
		*orientation = 2;
		return;

	case TC_BLOCK_Z_1:
		*lblock = 4;
		*orientation = 1;
		return;

	case TC_BLOCK_Z_2:
		*lblock = 4;
		*orientation = 0;
		return;

	case TC_BLOCK_S_1:
		*lblock = 5;
		*orientation = 1;
		return;

	case TC_BLOCK_S_2:
		*lblock = 5;
		*orientation = 0;
		return;

	case TC_BLOCK_T_1:
		*lblock = 6;
		*orientation = 3;
		return;

	case TC_BLOCK_T_2:
		*lblock = 6;
		*orientation = 2;
		return;

	case TC_BLOCK_T_3:
		*lblock = 6;
		*orientation = 1;
		return;

	case TC_BLOCK_T_4:
		*lblock = 6;
		*orientation = 0;
		return;

	default:
		PRINTF("oops in translation\n");
		abort();

	}

}

int HasGoodSpecial(int *Specials)
{
	return Specials[TC_SPECIAL_BLOCK_GRAVITY-TC_SPECIAL_FIRST_SPECIAL]
		|| Specials[TC_SPECIAL_NUKE-TC_SPECIAL_FIRST_SPECIAL]
		|| Specials[TC_SPECIAL_SWITCH_FIELDS-TC_SPECIAL_FIRST_SPECIAL];
}


void UseSpecials(FIELD field)
{
	int field_value;
	int num_holes, field_height;
	char specials[100];
	int i, try_another_special = 0, changed_field = 0;
	int num_players;
	char *fields[6];
	int field_values[6];
	int field_specials[6][9];
	int best_field_index = -1;
	int best_field_value = -20000;
	int use_special_failed;

	GetSpecials(specials);

	if (self_destruct && specials[0] != TC_SPECIAL_SWITCH_FIELDS) {
		FPRINTF(stderr, "self_destruct: Argh! Nearly killed meself!\n");
		self_destruct = 0;
	}
	if (specials[0])
	{
		num_players = GetNumPlayers(gdata);

		/* Get Fields for all players */
		for (i=0; i<num_players; i++)
		{
            field_values[i] = -20000;
			if (GetPlayerNumber(GetPlayer(gdata,i))==GetOwnPlayerNumber(gdata))
			{
				fields[i] = NULL;
				field_values[i] = best_field_value;
			}
			else
			{
				fields[i] = GetField(GetPlayerNumber(GetPlayer(gdata,i)));
				if (fields[i])
				{
					field_values[i] = FieldValue(fields[i], NULL, NULL, field_specials[i]);
				}
				else
				{
					field_values[i] = -20000;
				}
				if (field_values[i] && field_values[i]>best_field_value) 
				{
					best_field_index = i;
					best_field_value = field_values[i];
				}
			}
		}

        field_value = FieldValue(field, &num_holes, &field_height, NULL);
		if (self_destruct) {
			if (field_height >= 15) {
				/* guaranteed special is a switch */
				FPRINTF(stderr, "self_destruct: Complete.\n");
				use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata,best_field_index)));
				self_destruct = 0;
				changed_field = 1;
			} else
				return;
		}

		/* is there a switch coming up? */
		for (i=0; specials[i]!=0; i++) {
			if (specials[i] == TC_SPECIAL_SWITCH_FIELDS) {
				if (field_height<=15) {
					if (best_field_value > -2000 && abs(best_field_value-field_value) > 200) { /* ensure this is the same as below */
						FPRINTF(stderr, "self_destruct: activated - target %d (has %d, vs my %d)\n", best_field_index, best_field_value, field_value);
                        SendMessage(0, "Muahahahah.");
						self_destruct = 1;
						return;
					} else {
						FPRINTF(stderr, "self_destruct: chose not to - best field is %d with %d and i'm on %d\n", best_field_index, best_field_value, field_value);
					}
                }
                break;
			}
		}

		/* If all other special cases fail */
		for (i=0; specials[i]!=0; i++)
		{
			field_value = FieldValue(field, &num_holes, &field_height, NULL);
			try_another_special = 0;
			changed_field = 0;
			switch (specials[i])
			{
			case TC_SPECIAL_CLEAR_LINE:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata,best_field_index)));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_CLEAR_LINE: self_destruct: on best field\n");
				} else if (num_holes) 
				{
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 1;
					changed_field = 1;
                    FPRINTF(stderr, "TC_SPECIAL_CLEAR_LINE: on my field\n");
				}
				break;

			case TC_SPECIAL_BLOCK_GRAVITY:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata, best_field_index)));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_BLOCK_GRAVITY: self_destruct: on best field\n");
				} else if (field_height>15)
				{
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 0;
					changed_field = 1;
                    FPRINTF(stderr, "TC_SPECIAL_BLOCK_GRAVITY: on my field\n");
				}
				break;

			case TC_SPECIAL_NUKE:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata, best_field_index)));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_NUKE: self_destruct: on best field\n");
				} else if (field_height>15)
				{
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 0;
					changed_field = 1;
                    FPRINTF(stderr, "TC_SPECIAL_NUKE: on my field\n");
				}
				break;

			case TC_SPECIAL_SWITCH_FIELDS:
				if (self_destruct) {
					/* guaranteed special is a switch */
					FPRINTF(stderr, "self_destruct: complete\n");
					use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata,best_field_index)));
					self_destruct = 0;
					changed_field = 1;
				} else if (field_height>15)
				{
					if (best_field_value-field_value>100)
					{
						use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata,best_field_index)));
						sleep(1);
						/* We sleep to ensure we don't do anything too 
							 fast after switching */
					}
					else
					{
						use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
						try_another_special = 1;
					}
					changed_field = 1;
				} else {
					if (best_field_value > -1000 && best_field_value-field_value > 200) {
						FPRINTF(stderr, "SWITCHING TO SELF DESTRUCT MODE!!!\n");
						self_destruct = 1;
						try_another_special = 0;
						return;
					} else {
						FPRINTF(stderr, "Not Switching TO SELF DESTRUCT MODE!!!\n");
						use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
						try_another_special = 1;
					}
				}
				break;

			case TC_SPECIAL_CLEAR_SPECIALS:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_CLEAR_SPECIALS: self_destruct: on self\n");
				} else {
					int j, used = 0;
					for (j=0; j<num_players; j++)
					{
						if (fields[j] && HasGoodSpecial(field_specials[j]))
						{
							use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata, j)));
                            used = 1;
							break;
						}
					}
                    if (!used) {
                        DiscardSpecial();
                        try_another_special = 1;
                    }
                    FPRINTF(stderr, "TC_SPECIAL_CLEAR_SPECIALS: on good field\n");
				}
			break;

			case TC_SPECIAL_ADD_LINE:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_ADD_LINE: self_destruct: on self\n");
				} else {
					use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata, best_field_index)));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_ADD_LINE: on best field\n");
				}
				break;

			case TC_SPECIAL_BLOCK_BOMB:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_BLOCK_BOMB: self_destruct: on self\n");
				} else {
                    int bombable[6];
                    int *nextbombable = bombable;
                    int j;
                    for (j=num_players-1; j>=0; j--)
                    {
                        if (fields[j] && 
                                field_specials[j][TC_SPECIAL_BLOCK_BOMB-TC_SPECIAL_FIRST_SPECIAL])
                            *(nextbombable++) = j;
                    }
                    if (nextbombable > bombable) {
                        use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata, bombable[random()%(nextbombable-bombable)] )));
                    } else {
                        DiscardSpecial();
                        try_another_special = 1;
                    }
                    FPRINTF(stderr, "TC_SPECIAL_BLOCK_BOMB: on vulnerable field\n");
                }
				break;

			case TC_SPECIAL_CLEAR_RANDOM:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_CLEAR_RANDOM: self_destruct: on self\n");
				} else {
					use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata, best_field_index)));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_CLEAR_RANDOM: on best field\n");
				}
				break;

			case TC_SPECIAL_BLOCK_QUAKE:
				if (self_destruct) {
					use_special_failed = UseSpecial(GetOwnPlayerNumber(gdata));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_BLOCK_QUAKE: self_destruct: on self\n");
				} else {
					use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(gdata, best_field_index)));
					try_another_special = 1;
                    FPRINTF(stderr, "TC_SPECIAL_BLOCK_QUAKE: on best field\n");
				}
				break;

			}
			/* Have this test here as a failsafe to make sure we exit
				 if the game is over */
			{
				int dummy1, dummy2, dummy3;
				if (GetCurrentBlock(&dummy1, &dummy2, &dummy3)) break;
			}
			if (!try_another_special || use_special_failed) break;
		}
	}
}


int main(int argc, char *argv[])
{
	char *username = "nabnab";
	char *field;
	char hostname[4096];
	int blocktype, xpos, ypos;
	int localblocktype, orientation;
	int i, status;
	signed char option;
	int port = DEFAULT_PORT;
	

	srandom(time(NULL));
	sprintf( hostname, "%s", "localhost" );
	while ( (option=getopt(argc, argv, "p:h:")) > 0)
	{
		switch (option)
		{
		case 'p':
			port = strtol(optarg,(char**)NULL, 10);
			if ( 0 != errno || 0 >= port )
				FPRINTF( stderr, "Invalid port specified.\n" );
			else
			{
				FPRINTF( stderr, "Using port %d\n", port );
				break;
			}

		case 'h':
			sprintf( hostname, "%s", optarg );
			break;
		default:
			PRINTF("Usage: harmless [-h hostname] [-p port]\n");
			exit(0);
		}
	}

	argc -= (optind-1);
	argv += (optind-1);
	if (argc>1)
	{
		username = argv[1];
	}
	FPRINTF( stderr, "Username: %s\n", username );

	if (connectToServer(hostname, port, username, "", "")<0)
	{
		PRINTF("error: %s\n", tc_error_string);
		exit(1);
	}
	else
	{
		PRINTF("success\n");
	}

	signal( SIGALRM, alarmed );

	while (1)
	{
		PRINTF("Waiting for game start\n");
		if (Start(&gdata) != 0)
		{
			PRINTF("Game start failed\n");
			exit(1);
		}
		else
		{
			PRINTF("Game starting\n");
		}

		while (1)
		{
			status = GetCurrentBlock(&blocktype, &xpos, &ypos);
		
			if (status==1)
			{
				PRINTF("*** We lost! ***\n");
				PRINTF("========================================================\n");
				break;
			}
			else if (status==2)
			{
				PRINTF("========================================================\n");
				PRINTF("*** We won! ***\n");
				break;
			}

			if (blocktype==TC_BLOCK_NONE) 
			{
				usleep(200000);
				continue;
			}
			else
			{
				PRINTF("--------------------------------------------------\n");
				PRINTF("Got block %i\n", blocktype);

			}

			field = GetField(GetOwnPlayerNumber(gdata));
			if (field)
			{
				Move m = NewMove();

				PRINTF("************************Have field\n");
				
/* 				PrintField(field); */
				TranslateToLocalRep(blocktype, &localblocktype, &orientation);
				
				alarm( 1 );
				m = GetBestMove(localblocktype, xpos, ypos, orientation, field, 1);
				alarm( 0 );
				no_time = 0;
				
				PRINTF("**** **** ****\n" );
				PRINTF("**** Think best position is %i and orientation is %i\n",
							 m.p.x, m.orient);
				PRINTF("**** Current position %i and orientation is %i\n",
							 xpos, orientation);
				
				for (i=orientation; i != m.orient; orientation < m.orient? i++ : i--)
				{
					MoveBlock( orientation<m.orient ? TC_ROTATE_CLOCKWISE
										: TC_ROTATE_ANTICLOCKWISE );
				}
				for (i=xpos; i!=m.p.x; xpos<m.p.x ? i++ : i--)
				{
					MoveBlock(xpos<m.p.x ? TC_MOVE_RIGHT : TC_MOVE_LEFT);
				}

				if ( m.p.good )
				{
					int y=0, bt, xb, yb;
					int break_out=0;
					for ( i=m.p.n-1; i >= 0 && !break_out; i-- )
					{
						switch ( m.p.moves[i] )
						{
							case MoveDown:
								if ( 0 != GetCurrentBlock(&bt,&xb,&yb)
									|| TC_BLOCK_NONE == bt )
									break_out=1;
								else if ( y >= yb )
								{
									MoveBlock( TC_MOVE_DOWN );
									y++;
								}
								else
								{
									PRINTF( "NOT MOVING DOWN!\n" );
									PRINTF( "want: %d are: %d!\n", y, yb );
								}
								break;
							case MoveRight:
								MoveBlock( TC_MOVE_RIGHT );
								break;
							case MoveLeft:
								MoveBlock( TC_MOVE_LEFT );
								break;

							case RotCW:
								PRINTF( "warning trying to rotate cw\n" );
								break;
								
							case RotCC:
								PRINTF( "ROTATING\n" );
								MoveBlock( TC_ROTATE_ANTICLOCKWISE );
								break;

							default:
								; // bleh
						}
					}
				}
				MoveBlock(TC_MOVE_DROP);
				
				/* Specials on self? */
				UseSpecials(field);
			}
		}
	}	
	exit(0);
}
