/* Downfall - A tetrinet AI
 * Copyright (C) 2003-2004 Jonathan Gray
 * Copyright (C) 2003-2004 Nathan Parslow
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tetris.h"
#include "tclient.h"
#include "util.h"

/**
 * print out a field to stdout for debugging
 */
void printField(char *field)
{
	int x, y;

	if (field == NULL)
		return;

	for(y = 0; y < TC_FIELD_HEIGHT; y++)
	{
		debug(4, "%2d:", y);
		for(x = 0; x < TC_FIELD_WIDTH; x++)
		{
			/* what are these magic numbers? */
			if(getFieldPos(field, x, y) == 125)
				debug(4, "*");
			else if(getFieldPos(field, x, y) == 126)
				debug(4, "+");
			else if(getFieldPos(field, x, y) == 127)
				debug(4, "-");
			else if(getFieldPos(field, x, y))
				debug(4, "H");
			else
				debug(4, " ");
		}
		debug(4, "\n");
	}
	debug(4, "   012345678901\n");
}


void tetrinetShiftline(int line, int shift, char* field)
{
	int i;
	if (shift > 0) /* to the right */
	{
		for (i = TC_FIELD_WIDTH - 1; i >= shift; i--)
			getFieldPos(field, i, line) = getFieldPos(field, i - shift, line);
		for (; i >= 0; i--) 
			getFieldPos(field, i, line) = 0;
	}
	if (shift < 0) /* to the left */
	{
		for (i = 0; i < TC_FIELD_WIDTH + shift; i++)
			getFieldPos(field, i, line) = getFieldPos(field, i - shift, line);
		for (; i < TC_FIELD_WIDTH; i++)
			getFieldPos(field, i, line) = 0;
	}
	/* if shift == 0 do nothing */
}

/**
 * get baseType and rotations for a given block
 */
void blockProperties(char blockType, char *baseType, int *rotations)
{
	*baseType = blockType;
	*rotations = 0;
	switch (blockType) {
		case TC_BLOCK_NONE:
			break;
		case TC_BLOCK_O:
			*rotations = 0;
			*baseType = TC_BLOCK_O;
			break;
		case TC_BLOCK_I_1:
		case TC_BLOCK_I_2:
			*rotations = 1;
			*baseType = TC_BLOCK_I_1;
			break;
		case TC_BLOCK_S_1:
		case TC_BLOCK_S_2:
			*rotations = 1;
			*baseType = TC_BLOCK_S_1;
			break;
		case TC_BLOCK_Z_1:
		case TC_BLOCK_Z_2:
			*rotations = 1;
			*baseType = TC_BLOCK_Z_1;
			break;
		case TC_BLOCK_L_1:
		case TC_BLOCK_L_2:
		case TC_BLOCK_L_3:
		case TC_BLOCK_L_4:
			*rotations = 3;
			*baseType = TC_BLOCK_L_1;
			break;
		case TC_BLOCK_J_1:
		case TC_BLOCK_J_2:
		case TC_BLOCK_J_3:
		case TC_BLOCK_J_4:
			*rotations = 3;
			*baseType = TC_BLOCK_J_1;
			break;
		case TC_BLOCK_T_1:
		case TC_BLOCK_T_2:
		case TC_BLOCK_T_3:
		case TC_BLOCK_T_4:
			*rotations = 3;
			*baseType = TC_BLOCK_T_1;
			break;
		default:
			debug(1, "unknown block: %d\n", blockType);
	}
}
/**
 * drop, takes position and rotation.
 *
 * int pos - column for top left corner of block as in:
 * http://www.colinfahey.com/2003jan_tetris/tetris_standard_specifications.htm
 * (NOT dot location)
 *
 * int rotation required number of anticlockwise rotations (incr. piece)
 */
void drop(int pos, int rotation) {
	int r, x, y, t;

	for(r = 0; r < rotation; r++)
		MoveBlock(TC_ROTATE_ANTICLOCKWISE);
	for(r = 0; r > rotation; r--)
		MoveBlock(TC_ROTATE_CLOCKWISE);

	GetCurrentBlock(&t, &x, &y);


	/* Fix for dodginess - Shouldnt be needed any more
	switch (t) {
		case TC_BLOCK_S_2:
		case TC_BLOCK_Z_2:
		case TC_BLOCK_L_2:
		case TC_BLOCK_J_2:
		case TC_BLOCK_T_2:
			pos++;
	} */
	
	for(; x < pos; x++)
		MoveBlock(TC_MOVE_RIGHT);
	for(; x > pos; x--)
		MoveBlock(TC_MOVE_LEFT);

	MoveBlock(TC_MOVE_DROP);
}


void detectProblem(char *field, char* keptField)
{
	int x, y;
	
	if(keptField != NULL)
	{
		for(x = 0; x < TC_FIELD_WIDTH; x++)
			for(y = 0; y < TC_FIELD_HEIGHT; y++)
				if(getFieldPos(field, x, y) != getFieldPos(keptField, x, y))
				{
					debug(3, "Problem found, x:%2d y:%2d kF:%d f:%d\n", x, y, 
							getFieldPos(keptField, x, y), getFieldPos(field, x, y));
					if(getFieldPos(field, x, y) && getFieldPos(keptField, x, y))
						getFieldPos(keptField, x, y) = 125;
					else if(getFieldPos(field, x, y))
						getFieldPos(keptField, x, y) = 126;
					else
						getFieldPos(keptField, x, y) = 127;
				}
	}
	printField(keptField);
}

/**
 * takes game data struct.
 *
 * Goes through all possible placement combinations, uses addToField and
 * rateField then uses drop in highest rated position.
 */
void placePiece(struct TC_GameData *gData)
{
	int blockType, x, y, rotations, i, score, pos, rot;
	int nextI, nextX, nextRot, nextBlockType;
	char baseType, nextBaseType;
	double tmpRank, rank;
	char *field, *nextField, *tmpField;
	
	/* For debugging */
	static char *keptField = NULL;

	GetCurrentBlock(&blockType, &x, &y);

	blockProperties(blockType, &baseType, &rotations);
	
	if(blockType == TC_BLOCK_NONE)
		return;

	debug(1, "Ranking possible fields\n");

	score = pos = rot = 0;
	rank = (double)10000000.0;

	field = GetField(gData->PlayerNumber);
	if (field == NULL)
		return;

	/* Debugging */
	detectProblem(field, keptField);

	GetNextBlock(&nextBlockType);
	blockProperties(nextBlockType, &nextBaseType, &nextRot);
	debug(2, "Current Block: Type:%d baseType:%d Rotations:%d\n", blockType, baseType, rotations);
	debug(2, "Next Block: Type:%d baseType:%d Rotations:%d\n", nextBlockType, nextBaseType, nextRot);

	/**
	 * This is being really odd and only adding the second block, not the
	 * first so judgements are made on the second blocks position
	 *
	 * In further though this is also very odd because it would not give
	 * preference to a first peice position due to this error
	 *
	 * Clearly something is very wrong
	 */
	for(i = 0; i <= rotations; i++)
	{
		/* Go past the field each way until we work out if the block
		 * location is always in the block itself.  addToField can
		 * just drop bad positions
		 */
		for(x = TC_FIELD_WIDTH - 1; x >= -2; x--)
		{
			nextField = addToField(field, baseType + i, x);
			for(nextI = 0; nextI <= nextRot; nextI++)
			{
				for(nextX = TC_FIELD_WIDTH - 1; nextX >= -2; nextX--)
				{

					tmpField = addToField(nextField, nextBaseType + nextI, nextX);

					tmpRank = rateField(tmpField);
					debug(3, "Field Rank:%f i:%d x:%d nextI:%d nextX:%d\n", tmpRank, i, x, nextI, nextX);
					printField(tmpField);
					free(tmpField);
					if(tmpRank < rank)
					{
						rank = tmpRank;
						pos = x;
						rot = i - blockType + baseType;
					}
				}
			}
			free(nextField);
		}
	}

	/* For debugging */
	free(keptField);
	keptField = addToField(field, blockType + rot, pos);

	free(field);

	debug(1, "Dropping piece (pos: %d rot: %d)\n", pos, rot);
	drop(pos, rot);
}

/* return the height of the highest block in a column, 0 is empty */
int colHeight(char* field, int col) {
	int row;

	for (row = 0; row < TC_FIELD_HEIGHT; row++) {
		if (getFieldPos(field, col, row) != 0)
			return TC_FIELD_HEIGHT - row;
	}
	return 0;
}

/* rate the desirability of a given field
 * large numbers are bad 
 */
double rateField(char *field)
{
	int y, y2, x, score, lines;

	if (field == NULL)
		return 9999;

	score = 0;
	lines = 0;

	/* remove complete lines */

	for(y = TC_FIELD_HEIGHT - 1; y >= 0; y--)
	{
		for(x = 0; x < TC_FIELD_WIDTH && getFieldPos(field, x, y); x++)
		{}
		if(x == TC_FIELD_WIDTH)
		{
			lines++;
			for(x = 0; x < TC_FIELD_WIDTH; x++)
				score -= specialScore(getFieldPos(field, x, y)) * RATE_GET_SPECIAL;
			for(y2 = y - 1; y2 >= 0; y2--)
				for(x = 0; x < TC_FIELD_WIDTH; x++)
					getFieldPos(field, x, y2 + 1) = getFieldPos(field, x, y2);
			for(x = 0; x < TC_FIELD_WIDTH; x++)
				getFieldPos(field, x, 0) = 0;
			y++;
		}
	}
	if(lines)
		debug(4, "Full lines %d\n", lines);
	switch(lines)
	{
		case 4:
			score -= RATE_GET_4_LINES;
			break;
		case 3:
			score -= RATE_GET_3_LINES;
			break;
		case 2:
			score -= RATE_GET_2_LINES;
			break;
		case 1:
			score -= RATE_GET_1_LINE;
			break;
		default:
			score -= (RATE_GET_4_LINES / 4) * lines;
			break;
	}

	for(y = 0; y < TC_FIELD_HEIGHT; y++)
	{
		score += rateRow(field, y);
	}

	return score;
}

double rateRow(char *field, int row)
{
	int y, x, score, blockCount, unfillable, caves, specialScale, lastPiece, higherBlocks;

	blockCount = unfillable = caves = higherBlocks = 0;

	for(x = 0; x < TC_FIELD_WIDTH; x++)
	{
		if(!getFieldPos(field, x, row))
		{
			lastPiece = getFieldPos(field, x, row);
			for(y = row - 1; y >= 0; y--)
			{
				if(getFieldPos(field, x, y))
				{
					higherBlocks++;
					if(!lastPiece)
					{
						unfillable++;
						if(y == row - 1)
							caves++;
					}
				}
				lastPiece = getFieldPos(field, x, y);
			}
		}
		else
			blockCount++;
	}

	if(!blockCount)
		return (caves * RATE_CAVES);
	specialScale = (blockCount * RATE_BLOCKCOUNT) - (unfillable * RATE_UNFILLABLE);
	score = ((TC_FIELD_HEIGHT - row) * RATE_HEIGHT) + (caves * RATE_CAVES);
	if(blockCount > (TC_FIELD_WIDTH << 2))
		score -= (blockCount * RATE_BLOCKCOUNT);
	if(specialScale > 0)
		for(x = 0; x < TC_FIELD_WIDTH; x++)
			score -= specialScore(getFieldPos(field, x, row)) * specialScale;
	return score;
}

/* OLD */
#if 0
/* rate the desirability of a given field
 * large numbers are bad 
 * currently using algorithm from 
 * http://cslibrary.stanford.edu/112/Tetris-Architecture.html
 *
 * Hacked because this sucks - Bruce
 */
double rateField(char *field) {
	int sumHeight = 0, holes = 0, x;
	int height, y;
	double avgHeight;

	int variance = 0;

	if(field == NULL)
		return (double)10000000000.0; /* $BIGNUM */

	for (x = 0; x < TC_FIELD_WIDTH; x++) {
		height = colHeight(field, x);
		sumHeight += height;

		y = height - 2; // first possible hole

		while (y >= 0) {
			if (getFieldPos(field, x, y) == 0)
				holes++;
			y--;
		}
	}

	avgHeight = ((double)sumHeight / TC_FIELD_WIDTH);

	/* added to make this actually work */

	for(x = 0; x < TC_FIELD_WIDTH; x++)
		variance += abs(colHeight(field, x) - avgHeight);
	/* Add up the counts to make an overall score
 	 * The weights, 8, 40, etc., are just made up numbers that 
	 * appear to work
	 */
 	return (8*TC_FIELD_HEIGHT + 40*avgHeight + 1.25*holes + variance);
}
#endif

/**
 * call rateField based on a player number, used to make things easier
 */
double ratePlayerField(int playerNumber) {
	char *field;
	double result;
	field = GetField(playerNumber);
	if (field == NULL)
		return -1;
	result = rateField(field);
	free(field);
	return result;
}


/**
 * add a piece to a field and return a new field with said piece
 *
 * *field - field in
 * piece - piece to add
 * pos - posistion that x index 0 in blockmap should go
 */
char *addToField(char *field, char piece, int pos) {
	int x, y, height;
	char *newField;

	if(field == NULL)
		return NULL;
	/* check boundaries */
	if(pos < -3)
		return NULL;
	if(pos > TC_FIELD_WIDTH - 1)
		return NULL;
	for(x = 0; pos + x < 0; x++)
	{
		for(y = 3; y >= 0; y--)
		{
			if(blockmap[(int)piece][x][y])
				return NULL;
		}
	}
	for(x = 3; pos + x >= TC_FIELD_WIDTH; x--)
	{
		for(y = 3; y >= 0; y--)
		{
			if(blockmap[(int)piece][x][y])
				return NULL;
		}
	}

	/* find height peice will go at, from top of array, cos i can
	 * height counts up, unlike the field
	 */
	height = 0;
	for(x = 3; x >= 0; x--)
	{
		for(y = 3; y >= 0; y--)
		{
			if(blockmap[(int)piece][x][y])
			{
				if(y + colHeight(field, pos+x) > height)
					height = y + colHeight(field, pos+x);
				y = -1;
			}
		}
	}

	/* check for top boundary */
	if(height >= TC_FIELD_HEIGHT)
		return NULL;

	/* copy field */
	newField = (char *)malloc(TC_FIELD_SIZE);
	memcpy(newField, field, TC_FIELD_SIZE);

	/* add block */
	for(x = 3; x >= 0; x--)
	{
		for(y = 3; y >= 0; y--)
		{
			if(blockmap[(int)piece][x][y])
			{
				getFieldPos(newField, pos + x, TC_FIELD_HEIGHT - height - 1 + y) = blockmap[(int)piece][x][y];
			}
		}
	}
	return newField;
}

int specialScore(char special)
{
	switch(special)
	{
		case TC_SPECIAL_ADD_LINE:
			return RATE_SPECIAL_ADD_LINE;
			break;
		case TC_SPECIAL_CLEAR_SPECIALS:
			return RATE_SPECIAL_CLEAR_SPECIALS;
			break;
		case TC_SPECIAL_CLEAR_LINE:
			return RATE_SPECIAL_CLEAR_LINE;
			break;
		case TC_SPECIAL_BLOCK_GRAVITY:
			return RATE_SPECIAL_BLOCK_GRAVITY;
			break;
		case TC_SPECIAL_NUKE:
			return RATE_SPECIAL_NUKE;
			break;
		case TC_SPECIAL_BLOCK_BOMB:
			return RATE_SPECIAL_BLOCK_BOMB;
			break;
		case TC_SPECIAL_BLOCK_QUAKE:
			return  RATE_SPECIAL_BLOCK_QUAKE;
			break;
		case TC_SPECIAL_CLEAR_RANDOM:
			return RATE_SPECIAL_CLEAR_RANDOM;
			break;
		case TC_SPECIAL_SWITCH_FIELDS:
			return RATE_SPECIAL_SWITCH_FIELDS;
			break;
	}
	return 0;
}
