#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tclient.h"
#include "field.h"
#include "tetris.h"
#include "specials.h"
#include "move.h"

#define SB_Q_SIZE 1000

#define NUM_BLOCK_TYPES 15

/* set to 0 for no debugging, 1 for the input and the results, or 2
 * for the working out as well
 */
#define DEBUG_SPECIALS 0

static const int sb_in_tray[NUM_BLOCK_TYPES] = {
  0, 0, 0, 0, 0, 0,
  100,  /* Add Line */
  100,  /* Clear Line */
  500,  /* Nuke Field */
  100,  /* Clear Random Blocks */
  500,  /* Switch Fields */
  200,  /* Clear Special Blocks */
  500,  /* Block Gravity */
  100,  /* Blockquake */
  50    /* Block Bomb */
};

/* returns a random number in the range 0 to n-1 --
 * Note both n==0 and n==1 always return 0 */
static int randomnum (int n) {
  return (float)n * rand() / (RAND_MAX + 1.0);
}

static signed char *dup_field(signed char *field) {
  signed char *new_field = malloc(TC_FIELD_SIZE);

  return memcpy(new_field, field, TC_FIELD_SIZE);
}

static signed char *dup_field_and_lower (signed char *field) {
  int x, y, lower_by;
  signed char *new_field = malloc(TC_FIELD_SIZE);

  /* look for any blocks in the top 6 rows */
  for (y = 0; y < 6; y++)
    for (x = 0; x < TC_FIELD_WIDTH; x++)
      if (fblock(field, x, y))
        goto done;

  done:

  lower_by = 6 - y;

  if (lower_by) {
    memcpy(new_field + TC_FIELD_WIDTH * lower_by, field,
           TC_FIELD_SIZE - TC_FIELD_WIDTH * lower_by);
    memset(new_field, 0, TC_FIELD_WIDTH * lower_by);
  } else {
    memcpy(new_field, field, TC_FIELD_SIZE);
  }

  return new_field;
}

/* return false if the line couldn't be added */
static int addline(signed char *field) {
  int x;

  /* check top row */
  for (x = 0; x < TC_FIELD_WIDTH; x++) {
    if (fblock(field, x, 0)) {
      /* player is dead */
      return 0;
    }
  }

  /* move everything up one */
  memmove(field, field + TC_FIELD_WIDTH,
          TC_FIELD_WIDTH * (TC_FIELD_HEIGHT - 1));

  /* fill up the bottom line */
  memset(&(fblock(field, 0, TC_FIELD_HEIGHT - 1)), 1, TC_FIELD_WIDTH);

  /* add a single space */
  fblock(field, randomnum(TC_FIELD_WIDTH), TC_FIELD_HEIGHT - 1) = 0;

  return 1;
}

static void clearline(signed char *field) {
  /* move everything down one */
  memmove(field + TC_FIELD_WIDTH, field,
          TC_FIELD_WIDTH * (TC_FIELD_HEIGHT - 1));

  /* clear the top line */
  memset(field, 0, TC_FIELD_WIDTH);
}

static void clearrandom(signed char *field) {
  int i;

  for (i = 0; i < 10; i++) {
    fblock(field, randomnum(TC_FIELD_WIDTH), randomnum(TC_FIELD_HEIGHT)) = 0;
  }
}

static void lowerfield(signed char *field) {
  int x, y, lower_by;

  /* look for any blocks in the top 6 rows */
  for (y = 0; y < 6; y++)
    for (x = 0; x < TC_FIELD_WIDTH; x++)
      if (fblock(field, x, y))
        goto done;

  done:

  lower_by = 6 - y;

  /* if any blocks were found, move the field down so the top 6 rows
   * are clear
   */
  if (lower_by) {
    memmove(field + TC_FIELD_WIDTH * lower_by, field,
            TC_FIELD_WIDTH * lower_by);
    memset(field, 0, TC_FIELD_WIDTH * lower_by);
  }
}

static void clearspecials(signed char *field) {
  int x, y;

  for (y = 0; y < TC_FIELD_HEIGHT; y++)
    for (x = 0; x < TC_FIELD_WIDTH; x++)
      if (fblock(field, x, y) > 5)
        fblock(field, x, y) = 1;
}

static void blockgravity(signed char *field) {
  int x, y, i;

  for (y = 0; y < TC_FIELD_HEIGHT; y ++)
    for (x = 0; x < TC_FIELD_WIDTH; x ++)
      if (fblock(field, x, y) == 0) {
        for (i = y; i > 0; i--)
          fblock(field, x, i) = fblock(field, x, i - 1);
        fblock(field, x, 0) = 0;
      }
}

static void shiftline(signed char *field, int line, int shift)
{
  int i;
  if (shift > 0) { /* to the right */
    for (i = TC_FIELD_WIDTH - 1; i >= shift; i--)
      fblock(field, i, line) = fblock(field, i - shift, line);
    for (; i >= 0; i--)
      fblock(field, i, line) = 0;
  } else { /* to the left */
    for (i = 0; i < TC_FIELD_WIDTH + shift; i++)
      fblock(field, i, line) = fblock(field, i - shift, line);
    for (; i < TC_FIELD_WIDTH; i ++)
      fblock(field, i, line) = 0;
  }
}

static void blockquake(signed char *field) {
  int i, y;

  for (y = 0; y < TC_FIELD_HEIGHT; y ++) {
    int s = 0;
    i = randomnum (22);
    if (i < 1) s ++;
    if (i < 4) s ++;
    if (i < 11) s ++;
    if (s > 0) {
      if (randomnum(2))
        s = -s;
      shiftline(field, y, s);
    }
  }
}

static void blockbomb(signed char *field) {
  int ax[] = {-1, 0, 1, 1, 1, 0, -1, -1};
  int ay[] = {-1, -1, -1, 0, 1, 1, 1, 0};
  int c = 0;
  int x, y, i;
  signed char block;
  signed char buf[512];

  /* find all bomb blocks */
  for (y = 0; y < TC_FIELD_HEIGHT; y ++)
    for (x = 0; x < TC_FIELD_WIDTH; x ++)
      if (fblock(field, x, y)  == 14) {
        /* remove the bomb */
        fblock(field, x, y) = 0;
        /* grab the squares around it */
        for (i = 0; i < 8; i ++) {
          if (y + ay[i] >= TC_FIELD_HEIGHT || y + ay[i] < 0 ||
              x + ax[i] >= TC_FIELD_WIDTH || x + ax[i] < 0)
            continue;
          block = fblock(field, x + ax[i], y + ay[i]);
          if (block == 14)
            block = 0;
          else
            fblock(field, x + ax[i], y+ay[i]) = 0;
          buf[c] = block;
          c++;
        }
      }

  /* scatter blocks, but higher than they would normally be (otherwise
   * it's more likely that we'll bomb ourselves)
   */
  for (i = 0; i < c; i ++)
    fblock(field,
           randomnum(TC_FIELD_WIDTH),
           randomnum(TC_FIELD_HEIGHT - 12) + 3)
        = buf[i];
}

/* given a field and a special, apply the special to that field (for
 * switch fields, just lower the field)
 * return false if the special couldn't applied (ie. add line failed
 * because already at the top)
 */
static int apply_special(signed char *field, signed char sb) {
  switch (sb) {
    case TC_SPECIAL_ADD_LINE:
      return addline(field);
      break;
    case TC_SPECIAL_CLEAR_LINE:
      clearline(field);
      break;
    case TC_SPECIAL_NUKE:
      memset(field, 0, TC_FIELD_SIZE);
      break;
    case TC_SPECIAL_CLEAR_RANDOM:
      clearrandom(field);
      break;
    case TC_SPECIAL_SWITCH_FIELDS:
      lowerfield(field);
      break;
    case TC_SPECIAL_CLEAR_SPECIALS:
      clearspecials(field);
      break;
    case TC_SPECIAL_BLOCK_GRAVITY:
      blockgravity(field);
      break;
    case TC_SPECIAL_BLOCK_QUAKE:
      blockquake(field);
      break;
    case TC_SPECIAL_BLOCK_BOMB:
      blockbomb(field);
      break;
  }
  return 1;
}

/* Calculate the number of points each special in our tray is worth
 * based on sb_in_tray[].  If more than one of a particular block is
 * in the tray, each extra one is worth only half as many points (ones
 * earlier in the tray get fewer points).
 */
static void figure_sb_points(signed char *sb, int *sb_points) {
  signed char this_sb;
  int  i;
  int  sb_in_tray_points[NUM_BLOCK_TYPES];
  int  num_sb;

  memcpy(sb_in_tray_points, sb_in_tray, sizeof(int) * NUM_BLOCK_TYPES);

  num_sb = strlen(sb);
  for (i = num_sb - 1; i >= 0; i--) {
    this_sb = sb[i];
    sb_points[i] = sb_in_tray_points[(int)this_sb] * (101 - num_sb) / 100;
    sb_in_tray_points[(int)this_sb] /= 2;
  }
}

/* Figure out what additional points can be added to our score if we
 * use some specials before dropping a piece and picking up some more.
 * scores[i] will be set with the number of additional points that can
 * be picked up if the special block sb[i] is used.
 */
int figure_pickup_sb_scores(int *scores, signed char *field, signed char *sb,
                            int block, int orient, int bx, int by) {
  MOVE_RESULT *move;
  int i, j;

  move = find_best_move(field, "", block, orient, bx, by);

  /* if we can't move, we won't be getting any points from picking up
   * specials
   */
  if (!move) {
    memset(scores, 0, MAX_SPECIALS * sizeof(int));
    return 0;
  }

  /* figure out how many specials we need to discard to end up with
   * MAX_SPECIALS after dropping the block
   */
  j = strlen(sb) + strlen(move->specials) - MAX_SPECIALS;

  /* if our current blocks and any we can pick up all fit in our tray
   * then there's no advantage in discarding any specials
   */
  if (j <= 0) {
    memset(scores, 0, MAX_SPECIALS * sizeof(int));
    return 1;
  }

  for (i = 0; i < MAX_SPECIALS && j; i++, j--) {
    if (!move->specials[i]) {
      printf("hm, this shouldn't happen :-)\n");
      break;
    }
    scores[i] = sb_in_tray[(int)move->specials[i]];
  }
  while (i < MAX_SPECIALS) {
    scores[i++] = 0;
  }

  return 1;
}

SB_RESULT *eval_specials(signed char **in_fields, int *friend_or_foe,
                         signed char *sb, int own_num,
                         int block, int orient, int bx, int by) {
  /* the specials we've determined that we'll be using
   * (0-5 for players, -1 for discard)
   */
  int apply_num = 0;
  signed char apply[MAX_SPECIALS];

  /* index into sb of the special to evaluate next */
  int cur_sb;

  /* the number of points each sb in our tray is worth */
  int sb_points[MAX_SPECIALS];

  /* what the fields look like after applying the specials as per
   * apply[]
   */
  signed char *fields[6];

  /* the score of each field in fields
   * (our own field is the score after placing the current block, and
   * includes scoring the specials tray)
   */
  int scores[6];

  /* the sum of the elements in scores[] */
  /* to start with */
  int orig_total_score;
  /* the running total */
  int total_score;
  /* after applying each of apply[] */
  int apply_score[MAX_SPECIALS];

  /* the special block that we're looking at */
  int this_sb;

  /* the result we're returning */
  SB_RESULT *sb_result;

  /* the scores of the special blocks that we can pick up with the
   * next block
   */
  int pickup_sb_scores[MAX_SPECIALS];

  /* pointer into pickup_sb_scores[] */
  int *pickup_sb_scores_ptr;

  int i;

  /* nothing to do if we have no specials */
  if (!*sb) {
    return NULL;
  }

  /* calculate the number of points each special in our tray is worth */
  figure_sb_points(sb, sb_points);

  /* copy from in_fields to fields and calculate scores[] and total */
  total_score = 0;
  for (i = 0; i < 6; i++) {
    if (!friend_or_foe[i] || !in_fields[i]) {

      /* can't use specials if we're dead */
      if (i == own_num) {
        return NULL;
      }

      fields[i] = NULL;
      continue;
    }

#if DEBUG_SPECIALS
    printf("player %d is a %s\n", i, friend_or_foe[i] > 0 ? "friend" : "foe");
    show_field(in_fields[i], -1, 0, 0, 0);
#endif
    fields[i] = in_fields[i];
    if (i == own_num) {
      scores[i] = find_best_move_score(fields[i], sb + 1,
                                       block, orient, bx, by);
    } else {
      scores[i] = eval_field(fields[i], friend_or_foe[i]);
    }
    total_score += scores[i];
  }
  orig_total_score = total_score;

  /* figure out what specials we'd be picking up if we had no specials
   * and dropped a piece into our field
   */
  if (!figure_pickup_sb_scores(pickup_sb_scores, fields[own_num], sb,
                               block, orient, bx, by)) {
    /* if we can't place the next piece, the current specials aren't
     * worth anything
     */
    printf("better do something quick :-)\n");
    memset(sb_points, 0, MAX_SPECIALS * sizeof(int));
  }
  pickup_sb_scores_ptr = pickup_sb_scores;

  for (cur_sb = 0; sb[cur_sb]; cur_sb++) {
    /* the player that we're going to use the special on */
    int player_num;

    /* the best move we've found so far (-1 for discard), and its
     * effect on total_score
     */
    int best_move = -1;
    int best_score_delta = 0;

    /* our field and their field (and scores) after applying best_move */
    signed char *best_fields[2] = {NULL, NULL};
    int best_scores[2];

    this_sb = sb[cur_sb];
#if DEBUG_SPECIALS > 1
    printf("current special is %c (%d)\n", special_names[this_sb], this_sb);
#endif

    for (player_num = 0; player_num < 6; player_num++) {

      /* can't use a special on a player that doesn't exist */
      if (!friend_or_foe[player_num] || !fields[player_num]) {
        continue;
      }

#if DEBUG_SPECIALS > 1
     printf("trying special on %d\n", player_num);
#endif

      /* switch field on someone else gets special handling because it
       * affects two fields */
      if (this_sb == TC_SPECIAL_SWITCH_FIELDS && player_num != own_num) {

        /* their new field, our new field, and their scores */
        signed char *new_field[2];
        int new_scores[2];
        int score_delta;

        new_field[0] = dup_field_and_lower(fields[own_num]);
        new_field[1] = dup_field_and_lower(fields[player_num]);
        new_scores[0] = eval_field(new_field[0], friend_or_foe[player_num]);
        new_scores[1] = find_best_move_score(new_field[1], sb + cur_sb + 1,
                                             block, orient, bx, by);

#if DEBUG_SPECIALS > 1
        printf("other field is now:\n");
        show_field(new_field[0], -1, 0, 0, 0);
        printf("own field is now:\n");
        show_field(new_field[1], -1, 0, 0, 0);
#endif
        score_delta = new_scores[0] + new_scores[1]
                      - scores[own_num] - scores[player_num];
#if DEBUG_SPECIALS > 1
        printf("score_delta is %d\n", score_delta);
#endif
        if (score_delta > best_score_delta) {
#if DEBUG_SPECIALS > 1
          printf("seems like a good move\n");
#endif
          best_score_delta = score_delta;
          best_move = player_num;

          if (best_fields[0]) {
            free(best_fields[0]);
          }
          if (best_fields[1]) {
            free(best_fields[1]);
          }

          best_fields[0] = new_field[0];
          best_fields[1] = new_field[1];
          best_scores[0] = new_scores[0];
          best_scores[1] = new_scores[1];
        } else {
#if DEBUG_SPECIALS > 1
          printf("seems like a bad move\n");
#endif
          free(new_field[0]);
          free(new_field[1]);
        }

      /* not switch field or we're switching on ourselves; only one
       * field is involved
       */
      } else {
        signed char *new_field;
        int new_score;
        int score_delta;

        /* create a new field, apply the special, and see if the score
         * delta is better than best_score_delta
         */
        new_field = dup_field(fields[player_num]);

        if (apply_special(new_field, this_sb)) {
          if (player_num == own_num) {
            new_score = find_best_move_score(new_field, sb + cur_sb + 1,
                                             block, orient, bx, by);
          } else {
            new_score = eval_field(new_field, friend_or_foe[player_num]);
          }
          score_delta = new_score - scores[player_num];

#if DEBUG_SPECIALS > 1
          printf("other field is now:\n");
          show_field(new_field, -1, 0, 0, 0);
          printf("score_delta is %d\n", score_delta);
#endif
        } else {
#if DEBUG_SPECIALS > 1
          printf("applying special caused player to die\n");
#endif
          free(new_field);
          new_field = NULL;
          new_score = 0;
          score_delta = best_score_delta - 10000 * friend_or_foe[player_num];
        }

        /* if the score delta is better, free the old best_fields[]
         * and put ours in, otherwise free new_field
         */
        if (score_delta > best_score_delta) {
#if DEBUG_SPECIALS > 1
          printf("seems like a good move\n");
#endif

          best_score_delta = score_delta;
          best_move = player_num;

          if (best_fields[0]) {
            free(best_fields[0]);
          }
          if (best_fields[1]) {
            free(best_fields[1]);
          }

          best_fields[0] = new_field;
          best_fields[1] = NULL;
          best_scores[0] = new_score;
        } else {
#if DEBUG_SPECIALS > 1
          printf("seems like a bad move\n");
#endif
          free(new_field);
        }
      }
    }

    /* if we applied the special to ourselves or we switched on
     * someone else, figure out pickup_sb_scores[] again
     */
    if (best_move == own_num || best_fields[1]) {
      figure_pickup_sb_scores(pickup_sb_scores,
                              best_move == own_num
                                ? best_fields[0]
                                : best_fields[1],
                              sb + cur_sb + 1,
                              block, orient, bx, by);
      pickup_sb_scores_ptr = pickup_sb_scores;
    }

    total_score -= sb_points[cur_sb];
    total_score += *pickup_sb_scores_ptr++;
    total_score += best_score_delta;

    apply[apply_num] = best_move;
    apply_score[apply_num++] = total_score;

    /* if we're not discarding, update fields */
    if (best_move >= 0) {

      /* don't free in_fields[] */
      if (fields[best_move] != in_fields[best_move]) {
        free(fields[best_move]);
      }
      fields[best_move] = best_fields[0];
      scores[best_move] = best_scores[0];

      /* true if we did a switch on someone else */
      if (best_fields[1]) {
        if (fields[own_num] != in_fields[own_num]) {
          free(fields[own_num]);
        }
        fields[own_num] = best_fields[1];
        scores[own_num] = best_scores[1];
      }
    }
  }

#if DEBUG_SPECIALS
  for (i = 0; i < apply_num; i++) {
    printf("%c     ", special_names[(int)sb[i]]);
  }
  printf("\n");

  for (i = 0; i < apply_num; i++) {
    printf("%c     ", apply[i] < 0 ? 'D' : '0' + apply[i]);
  }
  printf("\n");

  for (i = 0; i < apply_num; i++) {
    printf("%5d ", apply_score[i]);
  }
  printf("\n");
#endif

  /* remove discards and moves that result in an overall lower score
   * from the end of the queue
   */
  for (i = apply_num - 1; i >= 0; i--) {
    if (apply[i] != -1 && apply_score[i] > orig_total_score) {
      break;
    }
    apply_num--;
  }

#if DEBUG_SPECIALS
  printf("applying %d specials\n", apply_num);
#endif

  for (i = 0; i < 6; i++) {
    if (fields[i] && fields[i] != in_fields[i]) {
      free(fields[i]);
    }
  }

  if (!apply_num) {
    return NULL;
  }

  sb_result = malloc(sizeof(SB_RESULT));
  if (!sb_result) {
    return NULL;
  }

  sb_result->total_score = total_score;
  sb_result->apply_num = apply_num;
  memcpy(sb_result->apply, apply, apply_num);

  return sb_result;
}

int eval_specials_in_tray(signed char *specials) {
  int score = 0;
  int sb;

  while ((sb = *specials++)) {
    score += sb_in_tray[sb];
  }
  return score;
}
