/* Beth
 * Geoffrey D. Bennett
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#include "tclient.h"
#include "tcommon.h"
#include "tetris.h"
#include "specials.h"
#include "move.h"
#include "field.h"

static struct TC_GameData *gdata;

static int block, orient, bx, by;
static int pendingblock, pendingorient;
static signed char *fields[6] = {NULL, NULL, NULL, NULL, NULL, NULL};
static signed char *field;

static signed char sb[MAX_SPECIALS + 1];

struct timeval tv1, tv2;

/* +1 for friend, -1 for foe, 0 for no player (either not playing or dead) */
static int friend_or_foe[6];

/* FIXME: tune this when we use a three-piece strategy */
static int best_move_cutoff = 100;

static void malloc_fields(void) {
  int i;

  for (i = 0; i < 6; i++) {
    fields[i] = malloc(TC_FIELD_SIZE);
  }
}

static void get_other_player_info(struct TC_GameData *data) {
  int i;

  for (i = 0; i < 6; i++) {
    friend_or_foe[i] = 0;
  }

  for (i = 0; i < 6; i++) {
    struct TC_PlayerData *player;

    player = GetPlayer(data, i);
    if (player) {
/* enable this to consider "beth*" as friendly */
/*
      if (!strncmp(GetPlayerName(player), "beth", 4)) {
        friend_or_foe[player->PlayerNumber] = 1;
      } else {
*/
        friend_or_foe[player->PlayerNumber] = -1;
/*
      }
*/
    }
  }

  friend_or_foe[GetOwnPlayerNumber(data)] = 1;

  /* FIXME: use GetPlayerTeam to set as friendly if they're on our team */
}

static void show_next() {
  int i, j;

  if (pendingblock < 0) {
    printf("No pending\n\n\n\n\n");
    return;
  }

  for (i = 0; i < 4; i++) {
    printf("              ");
    for (j = 0; j < 4; j++) {
      printf(blocks[pendingblock][pendingorient][i][j] ? "XX" : "  ");
    }
    printf("\n");
  }
}

static void show_status(void) {
  /* clear screen */
  printf("\033[H\033[2J");

  if (!field) {
    printf("player is dead\r\n");
    return;
  }

  show_next();
  show_field(field, block, orient, bx, by);
}

static void blocktype_to_blockorient(int blocktype, int *block, int *orient) {
  switch (blocktype) {
    case TC_BLOCK_O:   *block = 1; *orient = 0; break;
    case TC_BLOCK_I_1: *block = 0; *orient = 0; break;
    case TC_BLOCK_I_2: *block = 0; *orient = 1; break;
    case TC_BLOCK_S_1: *block = 5; *orient = 1; break;
    case TC_BLOCK_S_2: *block = 5; *orient = 0; break;
    case TC_BLOCK_Z_1: *block = 4; *orient = 1; break;
    case TC_BLOCK_Z_2: *block = 4; *orient = 0; break;
    case TC_BLOCK_L_1: *block = 3; *orient = 1; break;
    case TC_BLOCK_L_2: *block = 3; *orient = 0; break;
    case TC_BLOCK_L_3: *block = 3; *orient = 3; break;
    case TC_BLOCK_L_4: *block = 3; *orient = 2; break;
    case TC_BLOCK_J_1: *block = 2; *orient = 3; break;
    case TC_BLOCK_J_2: *block = 2; *orient = 2; break;
    case TC_BLOCK_J_3: *block = 2; *orient = 1; break;
    case TC_BLOCK_J_4: *block = 2; *orient = 0; break;
    case TC_BLOCK_T_1: *block = 6; *orient = 3; break;
    case TC_BLOCK_T_2: *block = 6; *orient = 2; break;
    case TC_BLOCK_T_3: *block = 6; *orient = 1; break;
    case TC_BLOCK_T_4: *block = 6; *orient = 0; break;
    default:
      *block = -1;
  }
}

static int tetribot_get_status(int *blocktype, int *bx, int *by,
                               int *pendingblocktype,
                               int *friend_or_foe,
                               signed char **fields, signed char *sb) {
  int player_num, status;

  txGetCurrentBlock();
  txGetNextBlock();
  txGetSpecials();

  for (player_num = 0; player_num < 6; player_num++) {
    if (friend_or_foe[player_num]) {
      txGetField(player_num);
    }
  }

  status = rxGetCurrentBlock(blocktype, bx, by);
  rxGetNextBlock(pendingblocktype);
  rxGetSpecials(sb);

  for (player_num = 0; player_num < 6; player_num++) {
    if (friend_or_foe[player_num]) {
      if (!rxGetField(fields[player_num])) {
        friend_or_foe[player_num] = 0;
      }
    }
  }

  return status;
}

static void get_status(void) {
  int blocktype;
  int pendingblocktype;
  int result;

  while ((result = tetribot_get_status(&blocktype, &bx, &by, &pendingblocktype,
                                       friend_or_foe, fields, sb)) != 0) {
/*
    printf("GetStatus returned %d\n", result);
*/
    while ((result = Start(&gdata)) != 0) {
      printf("Game start failed: %d\n", result);
      exit(1);
    }

/*
    printf("Game starting\n");
*/
    get_other_player_info(gdata);
  }

  blocktype_to_blockorient(pendingblocktype, &pendingblock, &pendingorient);

  /* pretend that the pending block is current if there is no current block */
  if (blocktype == TC_BLOCK_NONE) {
    blocktype_to_blockorient(pendingblocktype, &block, &orient);
    pendingblock = -1;
    bx = TC_FIELD_WIDTH / 2 - 2;
    by = 0;
  } else {
    blocktype_to_blockorient(blocktype, &block, &orient);
  }

  field = fields[GetOwnPlayerNumber(gdata)];
}

static void show_move_results(MOVE_RESULT **move_results) {
  MOVE_RESULT **search_move_results;
  MOVE_RESULT *move_result;
  int i;

  search_move_results = move_results;
  while ((move_result = *search_move_results++)) {
    printf("score %d move_count %d moves ",
           move_result->score,
           move_result->move_count);
    for (i = 0; i < move_result->move_count; i++) {
      switch (move_result->moves[i]) {
        case 1:
          printf("ML ");
          break;
        case 2:
          printf("MR ");
          break;
        case 3:
          printf("RR ");
          break;
        case 4:
          printf("RL ");
          break;
        case 5:
          printf("MD ");
          break;
      }
    }
    printf("\n");
    show_field(move_result->field, -1, 0, 0, 0);
    printf("------------------------------\n");
  }
}

/* Given an array of MOVE_RESULT structures, do a one-piece lookahead
 * with the block supplied and return the index of the best
 * MOVE_RESULT.  Only searches the best few moves from move_results
 * (assumes they're already sorted from best to worst).
 */
static MOVE_RESULT *move_results_lookahead(MOVE_RESULT **move_results,
                                           int block, int orient,
                                           int bx, int by) {
  MOVE_RESULT **search_move_results;
  MOVE_RESULT *best_move_result = NULL;
  MOVE_RESULT *move_result;
  int         best_score;
  MOVE_RESULT *sub_move_result;
  int         move_result_count = 0;

  best_move_result = NULL;
  search_move_results = move_results;
  while ((move_result = *search_move_results++) &&
         (move_result_count++ < best_move_cutoff)) {
    sub_move_result = find_best_move(move_result->field, move_result->specials,
                                     block, orient, bx, by);
    if (sub_move_result &&
        (!best_move_result || sub_move_result->score > best_score)) {
      best_move_result = move_result;
      best_score = sub_move_result->score;
    }
    if (sub_move_result) {
      free(sub_move_result);
    }
  }

  return best_move_result;
}

static int do_move(MOVE_RESULT *move) {
  int i, result = 0;

  if (!move) {
/*
    printf("ohoh!  I'm finished!\n");
*/
    return 1;
  }

  for (i = 0; i < move->move_count; i++) {
    txMoveBlock(move->moves[i]);
  }
  for (i = 0; i < move->move_count; i++) {
    result |= rxMoveBlock();
  }

  /* if any move couldn't be done, don't do the drop */
  if (result != 0) {
    return 1;
  }

  MoveBlock(TC_MOVE_DROP);

  return 0;
}

static void free_move_results(MOVE_RESULT **move_results) {
  MOVE_RESULT **search_move_results = move_results;

  if (!move_results) {
    return;
  }

  while (*search_move_results) {
    free(*search_move_results);
    search_move_results++;
  }

  free(move_results);
}

static void do_specials(SB_RESULT *sb_result) {
  int i;

  for (i = 0; i < sb_result->apply_num; i++) {
    if (sb_result->apply[i] == -1) {
      DiscardSpecial();
    } else {
      txUseSpecial(sb_result->apply[i]);
    }
  }
  for (i = 0; i < sb_result->apply_num; i++) {
    if (sb_result->apply[i] != -1) {
      rxUseSpecial();
    }
  }
}

static void play(void) {
  /* the results of applying the current block to the current field */
  MOVE_RESULT **first_move_results = NULL;

  /* the field and block that was current at the time
   * first_move_results was figured out (block1 == block as the block
   * won't just change on us; the block orientation might change if we
   * had a guess at it and guessed wrongly)
   */
  signed char field1[TC_FIELD_SIZE];
  int         orient1, x1, y1;

  MOVE_RESULT *move_result_guesses[7] =
                  {NULL, NULL, NULL, NULL, NULL, NULL, NULL};
  MOVE_RESULT *best_move_result;

  SB_RESULT   *sb_result;

  int i;

  while (1) {
    int restart = 0;

    get_status();

    /* something's not right if we don't have a block to play with */
    if (block == -1) {
      printf("play(): block == -1\n");
      continue;
    }

    /* can't play if we don't have a field */
    if (!field) {
      printf("play(): no field\n");
      continue;
    }

    /* if the move_results we have don't relate to the block or the
     * field we have, restart the process
     */
    if (!first_move_results) {
      restart = 1;
    } else if (memcmp(field1, field, TC_FIELD_SIZE)) {
/*
      printf("restart due changed field\n");
*/
      restart = 1;
    } else if (bx != x1 || by != y1) {
/*
      printf("restart due moved block\n");
*/
      restart = 1;
      /* FIXME: if just the y position has changed, we can maybe recover */
    } else if (orient != orient1) {
/*
      printf("restart due rotated block FIXME!\n");
*/
      restart = 1;
    }

    sb_result = eval_specials(fields, friend_or_foe, sb,
                              GetOwnPlayerNumber(gdata),
                              block, orient, bx, by);
    if (sb_result) {
      do_specials(sb_result);
      free(sb_result);
      continue;
    }

    if (restart) {
      free_move_results(first_move_results);
      first_move_results = NULL;

      orient1 = orient;
      x1 = bx;
      y1 = by;

      for (i = 0; i < 7; i++) {
        move_result_guesses[i] = NULL;
      }

      memcpy(field1, field, TC_FIELD_SIZE);
      first_move_results = find_moves(field1, sb, block, orient1, x1, y1);

      /* FIXME: emergency mode if !first_move_results */
      /* FIXME: if restart due changed field, apply the first_move_result */

      continue;
    }

    /* if we can't move, don't do anything */
    if (!first_move_results) {
      continue;
    }

//  gettimeofday(&tv1, NULL);
//  gettimeofday(&tv2, NULL);
//  printf("guess took %.3fs\n", ((tv2.tv_sec + tv2.tv_usec / 1000000.0) -
//                                 (tv1.tv_sec + tv1.tv_usec / 1000000.0)));
    /* if we don't have the pending block yet, start looking ahead anyway */
    if (pendingblock == -1) {
      for (i = 0; i < 7; i++) {
        if (!move_result_guesses[i]) {
          /* for all pieces except I, choose orientation 0 as it is most
           * likely to create a collision when the field is very high
           */
          move_result_guesses[i] =
              move_results_lookahead(first_move_results, i, i == 0 ? 1 : 0,
                                     TC_FIELD_WIDTH / 2 - 2, 0);
        }
      }

      /* check if we have our piece yet */
      continue;
    }

    best_move_result = move_result_guesses[pendingblock];
    move_result_guesses[pendingblock] = NULL;
    for (i = 0; i < 7; i++) {
      move_result_guesses[i] = NULL;
    }

    if (!best_move_result) {
      best_move_result = move_results_lookahead(first_move_results,
                                                pendingblock, pendingorient,
                                                TC_FIELD_WIDTH / 2 - 2, 0);
    }

    /* show what we're attempting to achieve */
//  show_field(best_move_result->field, -1, 0, 0, 0);
    do_move(best_move_result);
    free_move_results(first_move_results);
    first_move_results = NULL;
  }
}

int main(int argc, char *argv[]) {
  if (connectToServer("localhost", 9467, "beth", "", "") < 0) {
    printf("error: %s\n", tc_error_string);
    exit(1);
  }

  malloc_fields();

/*
  printf("Waiting for game start\n");
*/
  if (Start(&gdata) != 0) {
    printf("Game start failed\n");
    exit(1);
  }

  if (GetMaxSpecials(gdata) != MAX_SPECIALS) {
    fprintf(stderr,
            "see specials.h: MAX_SPECIALS should be %d\n",
            GetMaxSpecials(gdata));
    exit(1);
  }

  init_beth_move();

  get_other_player_info(gdata);

  play();

  exit(0);
}
