/*
 * Lock up CPUs with or without interrupts disabled.
 *
 * Example usage, lockup cpus 1 and 2 for 30 seconds:
 *
 * insmod ./badguy.ko timeout=30 cpus=1,2
 *
 * Leaving interrupts enabled should result in a soft lockup warning and
 * disabling interrupts should result in a hard lockup warning.
 *
 * Copyright 2015 Anton Blanchard, IBM Corporation <anton@au1.ibm.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/delay.h>
#include <linux/kthread.h>
#include <linux/time.h>

static bool disable_interrupts;
module_param(disable_interrupts, bool, S_IRUGO);
MODULE_PARM_DESC(disable_interrupts, "Disable interrupts (default = 0)");

#ifdef CONFIG_PPC64
static bool hard_disable;
module_param(hard_disable, bool, S_IRUGO);
MODULE_PARM_DESC(hard_disable, "Hard disable interrupts (default = 0)");
#endif

static unsigned int num_cpus;
static int cpus[NR_CPUS] = { 0 };
module_param_array(cpus, int, &num_cpus, S_IRUGO);
MODULE_PARM_DESC(cpus, "List of cpus to run on (default = 0)");

static long timeout = 60;
module_param(timeout, long, S_IRUGO);
MODULE_PARM_DESC(timeout, "Timeout in seconds (default = 60)");

static atomic_t running;

static struct task_struct *kthreads[NR_CPUS];

static int badguy_thread(void *data)
{
	unsigned long flags = 0;
	unsigned long cpu = (long)data;
	u64 tout = cpu_clock(cpu) + timeout * NSEC_PER_SEC;
	unsigned long jiffies_start, jiffies_prev;

	atomic_inc(&running);

	pr_info("Entering on cpu %lu\n", cpu);

	while (atomic_read(&running) < num_cpus)
		msleep(1);

	mb();

	if (disable_interrupts)
		local_irq_save(flags);

#ifdef CONFIG_PPC64
	if (hard_disable)
		hard_irq_disable();
#endif

	pr_info("Running on cpu %lu\n", cpu);

	jiffies_start = jiffies_prev = jiffies;

	while (cpu_clock(cpu) < tout) {
		if (kthread_should_stop()) {
			pr_info("Exiting early on cpu %lu\n", cpu);
			return 0;
		}

		if (jiffies - jiffies_prev > 1)
			pr_info("cpu %lu jiffies jumped %lu\n", cpu,
					jiffies - jiffies_prev);
		jiffies_prev = jiffies;
		barrier();
	}

	pr_info("Exiting on cpu %lu\n", cpu);

	if ((jiffies - jiffies_start) < timeout/2)
		pr_err("cpu %lu jiffies stuck (%lu)\n", cpu,
			jiffies - jiffies_start);

	if (disable_interrupts)
		local_irq_restore(flags);

	while (!kthread_should_stop()) {
		set_current_state(TASK_INTERRUPTIBLE);
		schedule();
	}

	return 0;
}

static int __init badguy_init(void)
{
	int i;

#ifdef CONFIG_PPC64
	if (hard_disable)
		disable_interrupts = 1;
#endif

	if (num_cpus == 0)
		num_cpus = 1;

	for (i = 0; i < num_cpus; i++) {
		unsigned long cpu = cpus[i];

		if (cpu != -1) {
			kthreads[i] = kthread_create(badguy_thread, (void *)cpu,
					   "badguy/%lu", cpu);
			if (IS_ERR(kthreads[i])) {
				pr_err("kthread_create on CPU %ld failed\n", cpu);
				atomic_inc(&running);
			} else {
				kthread_bind(kthreads[i], cpu);
				wake_up_process(kthreads[i]);
			}
		}
	}

	return 0;
}

static void __exit badguy_exit(void)
{
	int i;

	for (i = 0; i < num_cpus; i++) {
		if (!IS_ERR(kthreads[i]))
			kthread_stop(kthreads[i]);
	}
}

module_init(badguy_init)
module_exit(badguy_exit)

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Anton Blanchard");
MODULE_DESCRIPTION("Cause trouble");
